/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.server.blob.deduplication;

import com.google.common.base.MoreObjects;
import com.google.common.hash.BloomFilter;
import com.google.common.hash.Funnel;
import com.google.common.hash.Funnels;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.Clock;
import java.time.Instant;
import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.james.blob.api.BlobReferenceSource;
import org.apache.james.blob.api.BlobStoreDAO;
import org.apache.james.blob.api.BucketName;
import org.apache.james.server.blob.deduplication.GenerationAwareBlobId;
import org.apache.james.task.Task;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class BloomFilterGCAlgorithm {
    private static final Logger LOGGER = LoggerFactory.getLogger(BloomFilterGCAlgorithm.class);
    private static final Funnel<CharSequence> BLOOM_FILTER_FUNNEL = Funnels.stringFunnel((Charset)StandardCharsets.US_ASCII);
    private static final int DELETION_BATCH_SIZE = 1000;
    private final BlobReferenceSource referenceSource;
    private final BlobStoreDAO blobStoreDAO;
    private final GenerationAwareBlobId.Factory generationAwareBlobIdFactory;
    private final GenerationAwareBlobId.Configuration generationAwareBlobIdConfiguration;
    private final Instant now;
    private final String salt;

    public BloomFilterGCAlgorithm(BlobReferenceSource referenceSource, BlobStoreDAO blobStoreDAO, GenerationAwareBlobId.Factory generationAwareBlobIdFactory, GenerationAwareBlobId.Configuration generationAwareBlobIdConfiguration, Clock clock) {
        this.referenceSource = referenceSource;
        this.blobStoreDAO = blobStoreDAO;
        this.generationAwareBlobIdFactory = generationAwareBlobIdFactory;
        this.generationAwareBlobIdConfiguration = generationAwareBlobIdConfiguration;
        this.salt = UUID.randomUUID().toString();
        this.now = clock.instant();
    }

    public Mono<Task.Result> gc(int expectedBlobCount, int deletionWindowSize, double associatedProbability, BucketName bucketName, Context context) {
        return this.populatedBloomFilter(expectedBlobCount, associatedProbability, context).flatMap(bloomFilter -> this.gc((BloomFilter<CharSequence>)bloomFilter, bucketName, context, deletionWindowSize)).onErrorResume(error -> {
            LOGGER.error("Error when running blob deduplicate garbage collection", error);
            return Mono.just((Object)Task.Result.PARTIAL);
        });
    }

    private Mono<Task.Result> gc(BloomFilter<CharSequence> bloomFilter, BucketName bucketName, Context context, int deletionWindowSize) {
        return Flux.from((Publisher)this.blobStoreDAO.listBlobs(bucketName)).doOnNext(blobId -> context.incrementBlobCount()).flatMap(blobId -> Mono.fromCallable(() -> this.generationAwareBlobIdFactory.parse(blobId.asString()))).filter(blobId -> !blobId.inActiveGeneration(this.generationAwareBlobIdConfiguration, this.now)).filter(blobId -> !bloomFilter.mightContain((Object)(this.salt + blobId.asString()))).window(deletionWindowSize).flatMap(blobIdFlux -> this.handlePagedDeletion(bucketName, context, (Flux<GenerationAwareBlobId>)blobIdFlux), 16).reduce(Task::combine).switchIfEmpty(Mono.just((Object)Task.Result.COMPLETED));
    }

    private Mono<Task.Result> handlePagedDeletion(BucketName bucketName, Context context, Flux<GenerationAwareBlobId> blobIdFlux) {
        return blobIdFlux.collectList().flatMap(orphanBlobIds -> Mono.from((Publisher)this.blobStoreDAO.delete(bucketName, (Collection)orphanBlobIds)).then(Mono.fromCallable(() -> {
            context.incrementGCedBlobCount(orphanBlobIds.size());
            return Task.Result.COMPLETED;
        })).onErrorResume(error -> {
            LOGGER.error("Error when gc orphan blob", error);
            context.incrementErrorCount();
            return Mono.just((Object)Task.Result.PARTIAL);
        }));
    }

    private Mono<BloomFilter<CharSequence>> populatedBloomFilter(int expectedBlobCount, double associatedProbability, Context context) {
        return Mono.fromCallable(() -> BloomFilter.create(BLOOM_FILTER_FUNNEL, (int)expectedBlobCount, (double)associatedProbability)).flatMap(bloomFilter -> Flux.from((Publisher)this.referenceSource.listReferencedBlobs()).doOnNext(ref -> context.incrementReferenceSourceCount()).map(ref -> bloomFilter.put((Object)(this.salt + ref.asString()))).then().thenReturn(bloomFilter));
    }

    public static class Context {
        private final AtomicLong referenceSourceCount = new AtomicLong();
        private final AtomicLong blobCount = new AtomicLong();
        private final AtomicLong gcedBlobCount = new AtomicLong();
        private final AtomicLong errorCount = new AtomicLong();
        private final Long bloomFilterExpectedBlobCount;
        private final Double bloomFilterAssociatedProbability;

        public Context(long bloomFilterExpectedBlobCount, double bloomFilterAssociatedProbability) {
            this.bloomFilterExpectedBlobCount = bloomFilterExpectedBlobCount;
            this.bloomFilterAssociatedProbability = bloomFilterAssociatedProbability;
        }

        public void incrementBlobCount() {
            this.blobCount.incrementAndGet();
        }

        public void incrementReferenceSourceCount() {
            this.referenceSourceCount.incrementAndGet();
        }

        public void incrementGCedBlobCount(int count) {
            this.gcedBlobCount.addAndGet(count);
        }

        public void incrementErrorCount() {
            this.errorCount.incrementAndGet();
        }

        public Snapshot snapshot() {
            return Snapshot.builder().referenceSourceCount(this.referenceSourceCount.get()).blobCount(this.blobCount.get()).gcedBlobCount(this.gcedBlobCount.get()).errorCount(this.errorCount.get()).bloomFilterExpectedBlobCount(this.bloomFilterExpectedBlobCount).bloomFilterAssociatedProbability(this.bloomFilterAssociatedProbability).build();
        }

        public static class Snapshot {
            private final long referenceSourceCount;
            private final long blobCount;
            private final long gcedBlobCount;
            private final long errorCount;
            private final long bloomFilterExpectedBlobCount;
            private final double bloomFilterAssociatedProbability;

            public static Builder builder() {
                return new Builder();
            }

            Snapshot(long referenceSourceCount, long blobCount, long gcedBlobCount, long errorCount, long bloomFilterExpectedBlobCount, double bloomFilterAssociatedProbability) {
                this.referenceSourceCount = referenceSourceCount;
                this.blobCount = blobCount;
                this.gcedBlobCount = gcedBlobCount;
                this.errorCount = errorCount;
                this.bloomFilterExpectedBlobCount = bloomFilterExpectedBlobCount;
                this.bloomFilterAssociatedProbability = bloomFilterAssociatedProbability;
            }

            public long getReferenceSourceCount() {
                return this.referenceSourceCount;
            }

            public long getBlobCount() {
                return this.blobCount;
            }

            public long getGcedBlobCount() {
                return this.gcedBlobCount;
            }

            public long getErrorCount() {
                return this.errorCount;
            }

            public long getBloomFilterExpectedBlobCount() {
                return this.bloomFilterExpectedBlobCount;
            }

            public double getBloomFilterAssociatedProbability() {
                return this.bloomFilterAssociatedProbability;
            }

            public final boolean equals(Object o) {
                if (o instanceof Snapshot) {
                    Snapshot that = (Snapshot)o;
                    return Objects.equals(this.referenceSourceCount, that.referenceSourceCount) && Objects.equals(this.blobCount, that.blobCount) && Objects.equals(this.gcedBlobCount, that.gcedBlobCount) && Objects.equals(this.errorCount, that.errorCount) && Objects.equals(this.bloomFilterExpectedBlobCount, that.bloomFilterExpectedBlobCount) && Objects.equals(this.bloomFilterAssociatedProbability, that.bloomFilterAssociatedProbability);
                }
                return false;
            }

            public final int hashCode() {
                return Objects.hash(this.referenceSourceCount, this.blobCount, this.gcedBlobCount, this.errorCount, this.bloomFilterExpectedBlobCount, this.bloomFilterAssociatedProbability);
            }

            public String toString() {
                return MoreObjects.toStringHelper((Object)this).add("referenceSourceCount", this.referenceSourceCount).add("blobCount", this.blobCount).add("gcedBlobCount", this.gcedBlobCount).add("errorCount", this.errorCount).add("bloomFilterExpectedBlobCount", this.bloomFilterExpectedBlobCount).add("bloomFilterAssociatedProbability", this.bloomFilterAssociatedProbability).toString();
            }

            static class Builder {
                private Optional<Long> referenceSourceCount = Optional.empty();
                private Optional<Long> blobCount = Optional.empty();
                private Optional<Long> gcedBlobCount = Optional.empty();
                private Optional<Long> errorCount = Optional.empty();
                private Optional<Long> bloomFilterExpectedBlobCount = Optional.empty();
                private Optional<Double> bloomFilterAssociatedProbability = Optional.empty();

                Builder() {
                }

                public Snapshot build() {
                    return new Snapshot(this.referenceSourceCount.orElse(0L), this.blobCount.orElse(0L), this.gcedBlobCount.orElse(0L), this.errorCount.orElse(0L), this.bloomFilterExpectedBlobCount.orElse(0L), this.bloomFilterAssociatedProbability.orElse(0.0));
                }

                public Builder referenceSourceCount(long referenceSourceCount) {
                    this.referenceSourceCount = Optional.of(referenceSourceCount);
                    return this;
                }

                public Builder blobCount(long blobCount) {
                    this.blobCount = Optional.of(blobCount);
                    return this;
                }

                public Builder gcedBlobCount(long gcedBlobCount) {
                    this.gcedBlobCount = Optional.of(gcedBlobCount);
                    return this;
                }

                public Builder errorCount(long errorCount) {
                    this.errorCount = Optional.of(errorCount);
                    return this;
                }

                public Builder bloomFilterExpectedBlobCount(long bloomFilterExpectedBlobCount) {
                    this.bloomFilterExpectedBlobCount = Optional.of(bloomFilterExpectedBlobCount);
                    return this;
                }

                public Builder bloomFilterAssociatedProbability(double bloomFilterAssociatedProbability) {
                    this.bloomFilterAssociatedProbability = Optional.of(bloomFilterAssociatedProbability);
                    return this;
                }
            }
        }
    }
}

