/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.server.core;

import jakarta.mail.MessagingException;
import jakarta.mail.Session;
import jakarta.mail.internet.MimeMessage;
import jakarta.mail.util.SharedByteArrayInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.Properties;
import org.apache.commons.io.IOUtils;
import org.apache.james.lifecycle.api.LifecycleUtil;
import org.apache.james.server.core.ContentTypeCleaner;
import org.apache.james.server.core.MailImpl;
import org.apache.james.server.core.MimeMessageFromStreamTest;
import org.apache.james.server.core.MimeMessageInputStreamSource;
import org.apache.james.server.core.MimeMessageSource;
import org.apache.james.server.core.MimeMessageWrapper;
import org.apache.james.util.ClassLoaderUtils;
import org.apache.james.util.MimeMessageUtil;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class MimeMessageWrapperTest
extends MimeMessageFromStreamTest {
    TestableMimeMessageWrapper mw = null;
    TestableMimeMessageWrapper onlyHeader = null;
    final String content = "Subject: foo\r\nContent-Transfer-Encoding2: plain";
    final String sep = "\r\n\r\n";
    final String body = "bar\r\n";

    protected TestableMimeMessageWrapper getMessageFromSources(String sources) throws Exception {
        MimeMessageInputStreamSource mmis = MimeMessageInputStreamSource.create((String)"test", (InputStream)new SharedByteArrayInputStream(sources.getBytes()));
        return new TestableMimeMessageWrapper(this, (MimeMessageSource)mmis);
    }

    @BeforeEach
    public void setUp() throws Exception {
        this.mw = this.getMessageFromSources("Subject: foo\r\nContent-Transfer-Encoding2: plain\r\n\r\nbar\r\n");
        this.onlyHeader = this.getMessageFromSources("Subject: foo\r\nContent-Transfer-Encoding2: plain");
        ContentTypeCleaner.initialize();
    }

    @AfterEach
    public void tearDown() throws Exception {
        LifecycleUtil.dispose((Object)((Object)this.mw));
        LifecycleUtil.dispose((Object)((Object)this.onlyHeader));
    }

    @Test
    void testMessageWithWrongContentTypeShouldNotThrow() throws Exception {
        MimeMessageWrapper mmw = new MimeMessageWrapper((MimeMessage)this.mw);
        new MimeMessageWrapper((MimeMessage)this.mw).addHeader("Content-Type", "file;name=\"malformed.pdf\"");
        mmw.saveChanges();
    }

    @Test
    void testMessageWithStarContentTypeShouldNotThrow() throws Exception {
        MimeMessageWrapper mmw = new MimeMessageWrapper((MimeMessage)this.mw);
        new MimeMessageWrapper((MimeMessage)this.mw).addHeader("Content-Type", "image/*; name=\"20230720_175854.jpg\"");
        mmw.saveChanges();
    }

    @Test
    void setHeaderShouldNotAlterValidMimeVersionWhenComment() throws Exception {
        MimeMessageWrapper mw = new MimeMessageWrapper((MimeMessage)this.getMessageFromSources("Subject: foo\r\nMime-Version: 1.0 (Mac OS X Mail 15.0 \\(3693.60.0.1.1\\))\r\rContent-Transfer-Encoding2: plain\r\n\r\nbar\r\n"));
        mw.setHeader("abc", "def");
        mw.saveChanges();
        Assertions.assertThat((String)mw.getHeader("Mime-Version")[0]).isEqualTo("1.0 (Mac OS X Mail 15.0 \\(3693.60.0.1.1\\))");
    }

    @Test
    public void testDeferredMessageLoading() throws MessagingException, IOException {
        Assertions.assertThat((String)this.mw.getSubject()).isEqualTo("foo");
        Assertions.assertThat((boolean)this.mw.messageParsed()).isFalse();
        Assertions.assertThat((Object)this.mw.getContent()).isEqualTo((Object)"bar\r\n");
        Assertions.assertThat((boolean)this.mw.messageParsed()).isTrue();
        Assertions.assertThat((boolean)this.mw.isModified()).isFalse();
    }

    @Test
    public void testDeferredHeaderLoading() throws MessagingException, IOException {
        this.mw.setHeadersLoadable(false);
        Assertions.assertThatThrownBy(() -> this.mw.getSubject()).isInstanceOf(IllegalStateException.class);
    }

    @Test
    public void testMessageCloned() throws MessagingException, IOException, InterruptedException {
        MimeMessageWrapper mmw = new MimeMessageWrapper((MimeMessage)this.mw);
        LifecycleUtil.dispose((Object)((Object)this.mw));
        this.mw = null;
        System.gc();
        Thread.sleep(200L);
        mmw.writeTo((OutputStream)System.out);
    }

    @Test
    public void testGetSubjectFolding() throws Exception {
        StringBuilder res = new StringBuilder();
        BufferedReader r = new BufferedReader(new InputStreamReader(this.mw.getInputStream()));
        while (r.ready()) {
            String line = r.readLine();
            res.append(line).append("\r\n");
        }
        r.close();
        Assertions.assertThat((String)res.toString()).isEqualTo("bar\r\n");
    }

    @Test
    public void testAddHeaderAndSave() throws Exception {
        this.mw.addHeader("X-Test", "X-Value");
        Assertions.assertThat((String)this.mw.getHeader("X-Test")[0]).isEqualTo("X-Value");
        this.mw.saveChanges();
        ByteArrayOutputStream rawMessage = new ByteArrayOutputStream();
        this.mw.writeTo(rawMessage);
        Assertions.assertThat((String)this.mw.getHeader("X-Test")[0]).isEqualTo("X-Value");
        String res = rawMessage.toString();
        boolean found = res.indexOf("X-Test: X-Value") > 0;
        Assertions.assertThat((boolean)found).isTrue();
    }

    @Test
    public void testReplaceReturnPathOnBadMessage() throws Exception {
        MimeMessage message = this.getMessageWithBadReturnPath();
        message.setHeader("Return-Path", "<test@test.de>");
        Enumeration e = message.getMatchingHeaderLines(new String[]{"Return-Path"});
        Assertions.assertThat((String)((String)e.nextElement())).isEqualTo("Return-Path: <test@test.de>");
        Assertions.assertThat((boolean)e.hasMoreElements()).isFalse();
        Enumeration h = message.getAllHeaderLines();
        Assertions.assertThat((String)((String)h.nextElement())).isEqualTo("Return-Path: <test@test.de>");
        Assertions.assertThat((boolean)((String)h.nextElement()).startsWith("Return-Path:")).isFalse();
        LifecycleUtil.dispose((Object)message);
    }

    @Test
    public void testAddReturnPathOnBadMessage() throws Exception {
        MimeMessage message = this.getMessageWithBadReturnPath();
        message.addHeader("Return-Path", "<test@test.de>");
        Enumeration e = message.getMatchingHeaderLines(new String[]{"Return-Path"});
        Assertions.assertThat((String)((String)e.nextElement())).isEqualTo("Return-Path: <test@test.de>");
        Assertions.assertThat((String)((String)e.nextElement())).isEqualTo("Return-Path: <mybadreturn@example.com>");
        Enumeration h = message.getAllHeaderLines();
        Assertions.assertThat((String)((String)h.nextElement())).isEqualTo("Return-Path: <test@test.de>");
        LifecycleUtil.dispose((Object)message);
    }

    @Test
    public void testMessageStreamWithUpdatedHeaders() throws MessagingException, IOException {
        this.mw.addHeader("X-Test", "X-Value");
        Assertions.assertThat((String)this.mw.getHeader("X-Test")[0]).isEqualTo("X-Value");
        this.mw.saveChanges();
        BufferedReader reader = new BufferedReader(new InputStreamReader(this.mw.getMessageInputStream()));
        boolean headerUpdated = reader.lines().anyMatch(line -> line.equals("X-Test: X-Value"));
        reader.close();
        Assertions.assertThat((boolean)headerUpdated).isTrue();
    }

    @Test
    public void testMessageStreamWithUpatedContent() throws MessagingException, IOException {
        String newContent = "This is the new message content!";
        this.mw.setText(newContent);
        Assertions.assertThat((Object)this.mw.getContent()).isEqualTo((Object)newContent);
        this.mw.saveChanges();
        BufferedReader reader = new BufferedReader(new InputStreamReader(this.mw.getMessageInputStream()));
        boolean contentUpdated = reader.lines().anyMatch(line -> line.equals(newContent));
        reader.close();
        Assertions.assertThat((boolean)contentUpdated).isTrue();
    }

    @Test
    public void testSize() throws MessagingException {
        Assertions.assertThat((int)this.mw.getSize()).isEqualTo("bar\r\n".length());
    }

    @Test
    public void getSizeShouldReturnZeroWhenNoHeaderAndAddHeader() throws MessagingException {
        this.onlyHeader.addHeader("a", "b");
        Assertions.assertThat((int)this.onlyHeader.getSize()).isEqualTo(0);
    }

    @Test
    public void getSizeShouldReturnZeroWhenNoHeader() throws MessagingException {
        Assertions.assertThat((int)this.onlyHeader.getSize()).isEqualTo(0);
    }

    @Test
    public void getSizeShouldReturnZeroWhenSingleChar() throws Exception {
        TestableMimeMessageWrapper message = this.getMessageFromSources("a");
        Assertions.assertThat((int)message.getSize()).isEqualTo(0);
        LifecycleUtil.dispose((Object)((Object)message));
    }

    @Test
    public void getSizeShouldReturnZeroWhenSingleCharBody() throws Exception {
        TestableMimeMessageWrapper message = this.getMessageFromSources("a: b\r\n\r\na");
        Assertions.assertThat((int)message.getSize()).isEqualTo(1);
        LifecycleUtil.dispose((Object)((Object)message));
    }

    @Test
    public void getSizeShouldReturnZeroWhenEmptyString() throws Exception {
        TestableMimeMessageWrapper message = this.getMessageFromSources("");
        Assertions.assertThat((int)message.getSize()).isEqualTo(0);
        LifecycleUtil.dispose((Object)((Object)message));
    }

    @Test
    public void getMessageSizeShouldReturnExpectedValueWhenNoHeader() throws MessagingException {
        Assertions.assertThat((long)this.onlyHeader.getMessageSize()).isEqualTo((long)"Subject: foo\r\nContent-Transfer-Encoding2: plain".length());
    }

    @Test
    public void getMessageSizeShouldReturnExpectedValueWhenNoHeaderAndAddHeader() throws Exception {
        this.onlyHeader.addHeader("new", "value");
        Assertions.assertThat((long)this.onlyHeader.getMessageSize()).isEqualTo(IOUtils.consume((InputStream)this.onlyHeader.getMessageInputStream()));
    }

    @Test
    public void getMessageSizeShouldReturnExpectedValueWhenSingleChar() throws Exception {
        TestableMimeMessageWrapper message = this.getMessageFromSources("a");
        Assertions.assertThat((long)message.getMessageSize()).isEqualTo(1L);
        LifecycleUtil.dispose((Object)((Object)message));
    }

    @Test
    public void getMessageSizeShouldReturnExpectedValueWhenEmptyString() throws Exception {
        TestableMimeMessageWrapper message = this.getMessageFromSources("");
        Assertions.assertThat((long)message.getMessageSize()).isEqualTo(0L);
        LifecycleUtil.dispose((Object)((Object)message));
    }

    @Test
    public void testSizeModifiedHeaders() throws MessagingException {
        this.mw.addHeader("whatever", "test");
        Assertions.assertThat((int)this.mw.getSize()).isEqualTo("bar\r\n".length());
    }

    @Test
    public void testSizeModifiedBodyWithoutSave() throws MessagingException {
        String newBody = "This is the new body of the message";
        this.mw.setText(newBody);
        Assertions.assertThat((int)this.mw.getSize()).isEqualTo(-1);
    }

    @Test
    public void testSizeModifiedBodyWithSave() throws MessagingException {
        String newBody = "This is the new body of the message";
        this.mw.setText(newBody);
        this.mw.saveChanges();
        Assertions.assertThat((int)this.mw.getSize()).isEqualTo(-1);
    }

    @Test
    public void jiraJames1593() throws MessagingException, IOException {
        Properties noProperties = new Properties();
        Session session = Session.getDefaultInstance((Properties)noProperties);
        InputStream stream = ClassLoader.getSystemResourceAsStream("JAMES-1593.eml");
        MimeMessage message = new MimeMessage(session, stream);
        MimeMessageWrapper wrapper = new MimeMessageWrapper(message);
        Assertions.assertThat((String)wrapper.getEncoding()).isEqualTo("\"base64\"");
        LifecycleUtil.dispose((Object)wrapper);
    }

    @Test
    public void saveChangesShouldPreserveMessageId() throws Exception {
        String messageId = "<5436@ab.com>";
        MimeMessage message = MimeMessageUtil.mimeMessageFromString((String)("Message-ID: " + messageId + "\r\nSubject: test\r\n\r\nContent!"));
        MimeMessageWrapper mimeMessageWrapper = new MimeMessageWrapper(message);
        mimeMessageWrapper.saveChanges();
        Assertions.assertThat((String)mimeMessageWrapper.getMessageID()).isEqualTo(messageId);
        LifecycleUtil.dispose((Object)mimeMessageWrapper);
    }

    @Test
    public void getMessageSizeShouldBeAccurateWhenHeadersAreModified() throws Exception {
        MimeMessageWrapper wrapper = new MimeMessageWrapper((MimeMessageSource)MimeMessageInputStreamSource.create((String)MailImpl.getId(), (InputStream)ClassLoaderUtils.getSystemResourceAsSharedStream((String)"JAMES-1593.eml")));
        wrapper.setHeader("header", "vss");
        Assertions.assertThat((long)wrapper.getMessageSize()).isEqualTo(IOUtils.consume((InputStream)wrapper.getMessageInputStream()));
        LifecycleUtil.dispose((Object)wrapper);
    }

    @Test
    public void getMessageSizeShouldBeAccurateWhenHeadersAreModifiedAndOtherEncoding() throws Exception {
        MimeMessageWrapper wrapper = new MimeMessageWrapper((MimeMessageSource)MimeMessageInputStreamSource.create((String)MailImpl.getId(), (InputStream)ClassLoaderUtils.getSystemResourceAsSharedStream((String)"mail-containing-unicode-characters.eml")));
        wrapper.setHeader("header", "vss");
        Assertions.assertThat((long)wrapper.getMessageSize()).isEqualTo(IOUtils.consume((InputStream)wrapper.getMessageInputStream()));
        LifecycleUtil.dispose((Object)wrapper);
    }

    private final class TestableMimeMessageWrapper
    extends MimeMessageWrapper {
        boolean messageLoadable = true;
        boolean headersLoadable = true;

        private TestableMimeMessageWrapper(MimeMessageWrapperTest mimeMessageWrapperTest, MimeMessageSource source) throws MessagingException {
            super(source);
        }

        public boolean messageParsed() {
            return this.messageParsed;
        }

        public void setHeadersLoadable(boolean headersLoadable) {
            this.headersLoadable = headersLoadable;
        }

        protected synchronized void loadHeaders() throws MessagingException {
            if (!this.headersLoadable) {
                throw new IllegalStateException("headersLoadable disabled");
            }
            super.loadHeaders();
        }

        public synchronized void loadMessage() throws MessagingException {
            if (!this.messageLoadable) {
                throw new IllegalStateException("messageLoadable disabled");
            }
            super.loadMessage();
        }
    }
}

