/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.transport.mailets;

import java.time.Clock;
import java.time.ZonedDateTime;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import org.apache.james.transport.mailets.Expires;
import org.apache.mailet.Mail;
import org.apache.mailet.Mailet;
import org.apache.mailet.MailetConfig;
import org.apache.mailet.base.DateFormats;
import org.apache.mailet.base.test.FakeMail;
import org.apache.mailet.base.test.FakeMailetConfig;
import org.apache.mailet.base.test.MailUtil;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

public class ExpiresTest {
    private final ZonedDateTime NOW = ZonedDateTime.parse("2021-12-14T16:36:47Z");
    private Mailet mailet;

    @BeforeEach
    void setUp() {
        Clock clock = Clock.fixed(this.NOW.toInstant(), this.NOW.getZone());
        this.mailet = new Expires(clock);
    }

    @Test
    void shouldThrowWhenNoConfiguration() {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").build();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.mailet.init((MailetConfig)mailetConfig)).isInstanceOf(MessagingException.class)).hasMessage("Please configure at least one of minAge, maxAge, defaultAge");
    }

    @Test
    void shouldThrowWhenMinAgeAfterMaxAge() {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("minAge", "3d").setProperty("maxAge", "1h").build();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.mailet.init((MailetConfig)mailetConfig)).isInstanceOf(MessagingException.class)).hasMessage("minAge must be before maxAge");
    }

    @Test
    void shouldThrowWhenDefaultAgeAfterMaxAge() {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("defaultAge", "3d").setProperty("maxAge", "1h").build();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.mailet.init((MailetConfig)mailetConfig)).isInstanceOf(MessagingException.class)).hasMessage("defaultAge must be before maxAge");
    }

    @Test
    void shouldThrowWhenDefaultAgeBeforeMinAge() {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("defaultAge", "1h").setProperty("minAge", "3d").build();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.mailet.init((MailetConfig)mailetConfig)).isInstanceOf(MessagingException.class)).hasMessage("minAge must be before defaultAge");
    }

    @Test
    void shouldThrowOnMessagingException() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("defaultAge", "1d").build();
        this.mailet.init((MailetConfig)mailetConfig);
        Mail mail = (Mail)Mockito.mock(Mail.class);
        Mockito.when((Object)mail.getMessage()).thenThrow(new Throwable[]{new MessagingException()});
        Assertions.assertThatThrownBy(() -> this.mailet.service(mail)).isInstanceOf(MessagingException.class);
    }

    @Test
    void shouldSetHeaderOnMessage() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("defaultAge", "1h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).containsExactly((Object[])new String[]{ExpiresTest.asDateTime(this.NOW.plusHours(1L))});
    }

    @Test
    void shouldKeepHeaderWhenAlreadyPresent() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("defaultAge", "1h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        mimeMessage.addHeader("Expires", ExpiresTest.asDateTime(this.NOW.plusHours(2L)));
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).containsExactly((Object[])new String[]{ExpiresTest.asDateTime(this.NOW.plusHours(2L))});
    }

    @Test
    void shouldReplaceHeaderWhenBelowMinimum() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("minAge", "3h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        mimeMessage.addHeader("Expires", ExpiresTest.asDateTime(this.NOW.plusHours(1L)));
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).containsExactly((Object[])new String[]{ExpiresTest.asDateTime(this.NOW.plusHours(3L))});
    }

    @Test
    void shouldKeepHeaderWhenAboveMinimum() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("minAge", "1h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        mimeMessage.addHeader("Expires", ExpiresTest.asDateTime(this.NOW.plusHours(2L)));
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).containsExactly((Object[])new String[]{ExpiresTest.asDateTime(this.NOW.plusHours(2L))});
    }

    @Test
    void shouldReplaceHeaderWhenAboveMaximum() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("maxAge", "3h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        mimeMessage.addHeader("Expires", ExpiresTest.asDateTime(this.NOW.plusHours(5L)));
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).containsExactly((Object[])new String[]{ExpiresTest.asDateTime(this.NOW.plusHours(3L))});
    }

    @Test
    void shouldKeepHeaderWhenBelowMaximum() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("maxAge", "5h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        mimeMessage.addHeader("Expires", ExpiresTest.asDateTime(this.NOW.plusHours(3L)));
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).containsExactly((Object[])new String[]{ExpiresTest.asDateTime(this.NOW.plusHours(3L))});
    }

    @Test
    void shouldKeepHeaderWhenInRange() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("minAge", "1h").setProperty("maxAge", "5h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        mimeMessage.addHeader("Expires", ExpiresTest.asDateTime(this.NOW.plusHours(3L)));
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).containsExactly((Object[])new String[]{ExpiresTest.asDateTime(this.NOW.plusHours(3L))});
    }

    @Test
    void shouldIgnoreRangeWhenNoHeaderPresent() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("minAge", "1h").setProperty("maxAge", "5h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).isNullOrEmpty();
    }

    @Test
    void shouldSetHeaderForFixedRange() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("minAge", "3h").setProperty("defaultAge", "3h").setProperty("maxAge", "3h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).containsExactly((Object[])new String[]{ExpiresTest.asDateTime(this.NOW.plusHours(3L))});
    }

    @Test
    void shouldReplaceHeaderForFixedRange() throws MessagingException {
        FakeMailetConfig mailetConfig = FakeMailetConfig.builder().mailetName("Test").setProperty("minAge", "3h").setProperty("defaultAge", "3h").setProperty("maxAge", "3h").build();
        this.mailet.init((MailetConfig)mailetConfig);
        MimeMessage mimeMessage = MailUtil.createMimeMessage();
        mimeMessage.addHeader("Expires", ExpiresTest.asDateTime(this.NOW.plusHours(5L)));
        FakeMail mail = MailUtil.createMockMail2Recipients((MimeMessage)mimeMessage);
        this.mailet.service((Mail)mail);
        Assertions.assertThat((Object[])mail.getMessage().getHeader("Expires")).containsExactly((Object[])new String[]{ExpiresTest.asDateTime(this.NOW.plusHours(3L))});
    }

    private static String asDateTime(ZonedDateTime when) {
        return DateFormats.RFC822_DATE_FORMAT.format(when);
    }
}

