/*
 * Tigase Unified Archive Component - Extension of implementation of Message Archiving component for Tigase XMPP Server
 * Copyright (C) 2015 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.archive.unified.db;

import org.junit.*;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.MethodSorters;
import org.junit.runners.model.Statement;
import tigase.archive.unified.QueryCriteria;
import tigase.archive.unified.modules.FileQueryModule;
import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.db.AbstractDataSourceAwareTestCase;
import tigase.db.DBInitException;
import tigase.db.DataRepository;
import tigase.db.DataSourceAware;
import tigase.xml.Element;
import tigase.xml.XMLUtils;
import tigase.xmpp.StanzaType;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;
import tigase.xmpp.rsm.RSM;

import java.security.SecureRandom;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public class JDBCUnifiedArchiveRepository_FileMetadataTest
		extends AbstractDataSourceAwareTestCase<DataRepository, UnifiedArchiveRepository> {

	private final static SimpleDateFormat formatter2 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");

	private static final String PROJECT_ID = "unified-archive";
	private static final String VERSION = "3.0.0-SNAPSHOT";
	protected static String emoji = "\uD83D\uDE97\uD83D\uDCA9\uD83D\uDE21";
	private static JID[] buddies = null;
	private static JID owner = null;
	// this is static to pass date from first test to next one
	private static Date testStart = null;

	private static List<TestItem> allFiles = new ArrayList<>();

	@ClassRule
	public static TestRule rule = new TestRule() {
		@Override
		public Statement apply(Statement stmnt, Description d) {
			if (uri == null || !uri.startsWith("jdbc:")) {
				return new Statement() {
					@Override
					public void evaluate() throws Throwable {
						Assume.assumeTrue("Ignored due to not passed DB URI!", false);
					}
				};
			}
			return stmnt;
		}
	};

	@BeforeClass
	public static void prepareTest() throws DBInitException {
		loadSchema(PROJECT_ID, VERSION, Collections.singleton("unified-archive"));

		owner = JID.jidInstanceNS("ua-fm-" + UUID.randomUUID(), "test", "tigase-1");
		buddies = IntStream.of(0, 1, 2)
				.mapToObj(x -> JID.jidInstanceNS("ua-fm-" + UUID.randomUUID(), "test1", "tigase-" + x))
				.toArray(JID[]::new);
	}

	static {
		formatter2.setTimeZone(TimeZone.getTimeZone("UTC"));
	}

	private UnifiedArchiveRepository<DataRepository> repo;

	@Before
	public void setup() throws RepositoryException, InstantiationException, IllegalAccessException, SQLException,
							   ClassNotFoundException {
		repo = getDataSourceAware();

	}

	@Override
	protected Class<? extends DataSourceAware> getDataSourceAwareIfc() {
		return UnifiedArchiveRepository.class;
	}

	private static final SecureRandom random = new SecureRandom();
	private static final Set<String> forbidden = Set.of("'", "\"");

	protected String generateString(int lenght) {
		return random.ints(lenght, 33, 127)
				.mapToObj(x -> String.valueOf((char) x))
				.filter(x -> !forbidden.contains(x))
				.collect(Collectors.joining());
	}

	protected String generateFilename(String suffix) {
		return generateString(10) + "." + suffix;
	}

	private static final String[] filenameExtensions = new String[]{"jpg", "png", "avi", "mov", "mp4", "doc", "docx",
																	"zip", "7zip", "gz", "hvec"};

	protected String generateFilename() {
		return generateFilename(filenameExtensions[random.nextInt(filenameExtensions.length)]);
	}

	protected String generateURL(String filename) {
		return "https://example.com:443/" + UUID.randomUUID().toString() + "/file/" + filename;
	}

	protected String generateURL() {
		return generateURL(generateFilename());
	}

	private static final String[] mediaTypes = new String[]{"image/png", "image/jpeg", "video/avi", "video/mpeg",
															"unknown"};

	protected String generateMediaType() {
		return mediaTypes[random.nextInt(mediaTypes.length)];
	}

	protected UnifiedArchiveRepository.FileMetadata generateFileMetadata() {
		String filename = generateFilename();
		if (random.nextBoolean()) {
			return new JDBCUnifiedArchiveRepository.FileMetadata(XMLUtils.escape(generateURL(filename)),
																 generateMediaType(), XMLUtils.escape(filename),
																 "Interesting file " + UUID.randomUUID(),
																 Math.abs(random.nextLong()));
		} else {
			return new JDBCUnifiedArchiveRepository.FileMetadata(XMLUtils.escape(generateURL(filename)));
		}
	}

	private class TestItem {

		private final BareJID owner;
		private final JID buddy;
		private final Date timestamp;
		private final String id;
		private final UnifiedArchiveRepository.FileMetadata metadata;

		public TestItem(BareJID owner, JID buddy, Date timestamp, String id,
						UnifiedArchiveRepository.FileMetadata metadata) {
			this.owner = owner;
			this.buddy = buddy;
			this.timestamp = timestamp;
			this.metadata = metadata;
			this.id = id;
		}

		public String getId() {
			return id;
		}

		public BareJID getOwner() {
			return owner;
		}

		public JID getBuddy() {
			return buddy;
		}

		public Date getTimestamp() {
			return timestamp;
		}

		public UnifiedArchiveRepository.FileMetadata getMetadata() {
			return metadata;
		}
	}

	private static int minimalTimestampOffset = 1;

	protected Date nextTimstamp() {
		int offset = minimalTimestampOffset;
		minimalTimestampOffset += random.nextInt(20);
		if (uri.contains(":sqlserver:")) {
			long tmp = ((long)(System.currentTimeMillis() / 10)) * 10;
			return new Date(tmp + (offset * 100));
		} else {
			return new Date(System.currentTimeMillis() + (offset * 100));
		}
	}

	protected TestItem genereteFileItem() {
		return new TestItem(owner.getBareJID(), buddies[random.nextInt(buddies.length)], nextTimstamp(),
							UUID.randomUUID().toString(), generateFileMetadata());
	}

	protected Element packetFromItem(TestItem item) {
		Element msg = new Element("message", new String[]{"from", "to", "type"},
								  new String[]{item.getOwner().toString(), item.getBuddy().toString(),
											   StanzaType.chat.name()});
		msg.withElement("oob", "jabber:iq:oob", oob -> oob.withElement("url", null, item.getMetadata().getURL()));
		msg.withElement("body", null, item.getMetadata().getURL());

		if (item.getMetadata().getName() != null) {
			Element fileEl = new Element("file");
			fileEl.setXMLNS("urn:xmpp:jingle:apps:file-transfer:5");

			fileEl.addChild(new Element("media-type", item.getMetadata().getMediaType()));
			fileEl.addChild(new Element("name", item.getMetadata().getName()));
			fileEl.addChild(new Element("desc", item.getMetadata().getDescription()));
			fileEl.addChild(new Element("size", String.valueOf(item.getMetadata().getSize())));

			Element referenceEl = new Element("reference");
			referenceEl.setXMLNS("urn:xmpp:reference:0");
			referenceEl.setAttribute("type", "data");
			referenceEl.setAttribute("uri", item.getMetadata().getURL());

			Element mediaSharing = new Element("media-sharing");
			mediaSharing.setXMLNS("urn:xmpp:sims:1");
			referenceEl.addChild(mediaSharing);

			mediaSharing.addChild(fileEl);

			msg.addChild(referenceEl);
		}

		return msg;
	}

	@Test
	public void test1_archiveFiles() throws RepositoryException, ComponentException {
		if (uri == null) {
			return;
		}
		Date date = new Date();
		testStart = date;

		for (int i = 0; i < 20; i++) {
			TestItem item = genereteFileItem();

			Element msg = packetFromItem(item);
			repo.archiveMessage(item.getOwner(), item.getBuddy(), item.getTimestamp(), msg, item.getId(), null);

			allFiles.add(item);
		}

		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setStart(date);
		crit.getRsm().setMax(1);

		repo.queryItems(crit, (query, item) -> {
			return;
		});

		Assert.assertEquals("Invalid count of archived messages", 20, (int) crit.getRsm().getCount());
	}

	@Test
	public void test2_queryAllFiles() throws RepositoryException, ComponentException {
		FileQueryModule.FileQuery query = new FileQueryModule.FileQuery(BareJID.bareJIDInstanceNS(owner.getDomain()),
																		owner.getBareJID(), null, null, null, null,
																		null, null, null);
		RSM rsm = new RSM(100);

		List<UnifiedArchiveRepository.FileItem> expected = allFiles.stream()
				.map(this::mapToFileItem)
				.collect(Collectors.toList());
		List<UnifiedArchiveRepository.FileItem> items = repo.queryFiles(query, rsm).collect(Collectors.toList());

		for (int i = 0; i < expected.size(); i++) {
			UnifiedArchiveRepository.FileItem i1 = expected.get(i);
			UnifiedArchiveRepository.FileItem i2 = items.get(i);

			Assert.assertEquals(i1.getId(), i2.getId());
			Assert.assertEquals(i1.getURL(), i2.getURL());
			Assert.assertEquals(i1.getName(), i2.getName());
			Assert.assertEquals(i1.getDescription(), i2.getDescription());
			Assert.assertEquals(i1.getMediaType(), i2.getMediaType());
			Assert.assertEquals(i1.getSize(), i2.getSize());
		}

		//Assert.assertEquals(expected, items);
		Assert.assertEquals(20, (int) rsm.getCount());
	}

	@Test
	public void test3_queryFirst10Files() throws RepositoryException, ComponentException {
		FileQueryModule.FileQuery query = new FileQueryModule.FileQuery(BareJID.bareJIDInstanceNS(owner.getDomain()),
																		owner.getBareJID(), null, null, null, null,
																		null, null, null);
		RSM rsm = new RSM(10);

		List<UnifiedArchiveRepository.FileItem> expected = allFiles.stream()
				.limit(10)
				.map(this::mapToFileItem)
				.collect(Collectors.toList());
		List<UnifiedArchiveRepository.FileItem> items = repo.queryFiles(query, rsm).collect(Collectors.toList());

		Assert.assertEquals(expected, items);
		Assert.assertEquals(20, (int) rsm.getCount());
	}

	@Test
	public void test4_queryAfterFiles() throws RepositoryException, ComponentException {
		FileQueryModule.FileQuery query = new FileQueryModule.FileQuery(BareJID.bareJIDInstanceNS(owner.getDomain()),
																		owner.getBareJID(), null, null, null, null,
																		null, null, null);
		RSM rsm = new RSM(10);
		rsm.setAfter(allFiles.get(9).id);

		List<UnifiedArchiveRepository.FileItem> expected = allFiles.stream()
				.skip(10)
				.map(this::mapToFileItem)
				.collect(Collectors.toList());
		List<UnifiedArchiveRepository.FileItem> items = repo.queryFiles(query, rsm).collect(Collectors.toList());

		Assert.assertEquals(expected, items);
		Assert.assertEquals(20, (int) rsm.getCount());
	}

	@Test
	public void test5_queryBuddy1Files() throws RepositoryException, ComponentException {
		FileQueryModule.FileQuery query = new FileQueryModule.FileQuery(BareJID.bareJIDInstanceNS(owner.getDomain()),
																		owner.getBareJID(), buddies[0].getBareJID(),
																		null, null, null, null, null, null);
		RSM rsm = new RSM(100);

		List<UnifiedArchiveRepository.FileItem> expected = allFiles.stream()
				.filter(item -> buddies[0].equals(item.getBuddy()))
				.map(this::mapToFileItem)
				.collect(Collectors.toList());
		List<UnifiedArchiveRepository.FileItem> items = repo.queryFiles(query, rsm).collect(Collectors.toList());

		Assert.assertEquals(expected, items);
		Assert.assertEquals(expected.size(), (int) rsm.getCount());
	}

	@Test
	public void test5_queryDomainFiles() throws RepositoryException, ComponentException {
		FileQueryModule.FileQuery query = new FileQueryModule.FileQuery(BareJID.bareJIDInstanceNS(owner.getDomain()),
																		null, null,
																		testStart, null, null, null, null, null);
		RSM rsm = new RSM(100);

		List<UnifiedArchiveRepository.FileItem> expected = allFiles.stream()
				.map(this::mapToFileItem)
				.collect(Collectors.toList());
		List<UnifiedArchiveRepository.FileItem> items = repo.queryFiles(query, rsm).collect(Collectors.toList());

		Assert.assertEquals(expected, items);
		Assert.assertEquals(expected.size(), (int) rsm.getCount());
	}

	@Test
	public void test6_queryFilesByMediaType() throws RepositoryException, ComponentException {
		Optional<String> mediaType = allFiles.stream()
				.map(TestItem::getMetadata)
				.map(UnifiedArchiveRepository.FileMetadata::getMediaType)
				.filter(Objects::nonNull)
				.findFirst();
		if (mediaType.isPresent()) {
			FileQueryModule.FileQuery query = new FileQueryModule.FileQuery(
					BareJID.bareJIDInstanceNS(owner.getDomain()), owner.getBareJID(), null, null, null, null,
					mediaType.get(), null, null);
			RSM rsm = new RSM(100);

			List<UnifiedArchiveRepository.FileItem> expected = allFiles.stream()
					.filter(item -> mediaType.get().equals(item.getMetadata().getMediaType()))
					.map(this::mapToFileItem)
					.collect(Collectors.toList());
			List<UnifiedArchiveRepository.FileItem> items = repo.queryFiles(query, rsm).collect(Collectors.toList());

			Assert.assertEquals(expected, items);
			Assert.assertEquals(expected.size(), (int) rsm.getCount());
		}
		;
	}

	@Test
	public void test7_queryFilesByContains() throws RepositoryException, ComponentException {
		Optional<String> contains = allFiles.stream()
				.map(TestItem::getMetadata)
				.map(UnifiedArchiveRepository.FileMetadata::getName)
				.filter(Objects::nonNull)
				.findFirst();
		if (contains.isPresent()) {
			FileQueryModule.FileQuery query = new FileQueryModule.FileQuery(
					BareJID.bareJIDInstanceNS(owner.getDomain()), owner.getBareJID(), null, null, null, contains.get(),
					null, null, null);
			RSM rsm = new RSM(100);

			List<UnifiedArchiveRepository.FileItem> expected = allFiles.stream()
					.filter(item -> contains.get().equals(item.getMetadata().getName()))
					.limit(1)
					.map(this::mapToFileItem)
					.collect(Collectors.toList());
			List<UnifiedArchiveRepository.FileItem> items = repo.queryFiles(query, rsm).collect(Collectors.toList());

			Assert.assertEquals(expected, items);
			Assert.assertEquals(expected.size(), (int) rsm.getCount());
		}
	}

	@Test
	public void test8_removeItems() throws RepositoryException, ComponentException {
		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setStart(testStart);

		List<UnifiedArchiveRepository.Item> msgs = new ArrayList<>();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertNotEquals("No messages in repository to execute test - we should have some already", 0,
							   msgs.size());
		repo.removeItems(owner.getBareJID(), null, testStart, new Date(System.currentTimeMillis() + 1000 * 1000));

		msgs.clear();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Still some messages, while in this duration all should be deleted", 0, msgs.size());
	}

	protected UnifiedArchiveRepository.FileItem mapToFileItem(TestItem item) {
		return new JDBCUnifiedArchiveRepository.FileItem(item.getOwner(), item.getBuddy().getBareJID(),
														 item.getTimestamp(), item.getId(), item.getMetadata().getURL(),
														 item.getMetadata().getMediaType(),
														 item.getMetadata().getName(),
														 item.getMetadata().getDescription(),
														 item.getMetadata().getSize());
	}
}
