/*
 * Tigase Unified Archive Component - Extension of implementation of Message Archiving component for Tigase XMPP Server
 * Copyright (C) 2015 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.archive.unified.db;

import org.junit.*;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.model.Statement;
import tigase.archive.db.MessageArchiveRepository;
import tigase.component.exceptions.RepositoryException;
import tigase.db.*;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.Element;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;
import tigase.xmpp.rsm.RSM;

import java.sql.SQLException;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * Created by andrzej on 19.02.2017.
 */

public class JDBCUnifiedArchiveRepositoryWithRecentsTest2
		extends AbstractDataSourceAwareTestCase<DataRepository,UnifiedArchiveRepository> {

	private static final String PROJECT_ID = "unified-archive";
	private static final String VERSION = "3.0.0-SNAPSHOT";

	@ClassRule
	public static TestRule rule = new TestRule() {
		@Override
		public Statement apply(Statement stmnt, Description d) {
			if (uri == null || !uri.startsWith("jdbc:")) {
				return new Statement() {
					@Override
					public void evaluate() throws Throwable {
						Assume.assumeTrue("Ignored due to not passed DB URI!", false);
					}
				};
			}
			return stmnt;
		}
	};
	
	private List<JID> buddies = new ArrayList<>();
	private BareJID owner;
	private UnifiedArchiveRepository<DataRepository> repo;
	private Date start;
	private Date end;
	private List<Date> newestMessageDate = new ArrayList<>();

	@BeforeClass
	public static void prepareTest() throws DBInitException {
		loadSchema(PROJECT_ID, VERSION, Collections.singleton("unified-archive"));
	}

	@Before
	public void setup()
			throws RepositoryException, InstantiationException, IllegalAccessException, TigaseStringprepException,
				   InterruptedException, SQLException, ClassNotFoundException {
		repo = getDataSourceAware();

		owner = BareJID.bareJIDInstance("owner-" + UUID.randomUUID().toString(), "test.com");

		start = new Date(System.currentTimeMillis() - 10 * 1000);
		end = new Date(System.currentTimeMillis() + 10 * 1000);
		Thread.sleep(1000);
		buddies.add(JID.jidInstance("chat-" + UUID.randomUUID().toString(), "test.com", "res-1"));
		buddies.add(JID.jidInstance("chat-" + UUID.randomUUID().toString(), "test.com", "res-1"));

		long diff = (end.getTime() - start.getTime()) / 5;

		for (int i=0; i<2; i++) {
			JID buddy = buddies.get(i);
			long startX = start.getTime() - (1000 * i);
			Date date = new Date(startX + (diff * 3));
			Element elem = createGroupchat(owner, buddy, MessageArchiveRepository.Direction.incoming, UUID.randomUUID().toString());
			repo.archiveMessage(owner, buddy, date, elem, UUID.randomUUID().toString(),null);
			date = new Date(startX + (diff * 4));
			elem = createGroupchat(owner, buddy, MessageArchiveRepository.Direction.incoming, UUID.randomUUID().toString());
			repo.archiveMessage(owner, buddy, date, elem, UUID.randomUUID().toString(), null);
			if (i == 1) {
				elem = createGroupchat(owner, buddy, MessageArchiveRepository.Direction.incoming, UUID.randomUUID().toString());
				repo.archiveMessage(owner, buddy, date, elem, UUID.randomUUID().toString(), null);
			}
			newestMessageDate.add(date);
			date = new Date(startX + (diff * 2));
			elem = createGroupchat(owner, buddy, MessageArchiveRepository.Direction.incoming, UUID.randomUUID().toString());
			repo.archiveMessage(owner, buddy, date, elem, UUID.randomUUID().toString(),null);
			date = new Date(startX + (diff * 1));
			elem = createGroupchat(owner, buddy, MessageArchiveRepository.Direction.incoming, UUID.randomUUID().toString());
			repo.archiveMessage(owner, buddy, date, elem, UUID.randomUUID().toString(),null);
		}
	}

	@After
	public void tearDown() throws TigaseDBException {
		for (JID buddy : buddies) {
			repo.removeItems(owner, buddy.getBareJID().toString(), start, end);
		}
		repo.destroy();
	}

	@Test
	public void testQueryRecentsAll() throws TigaseDBException {
		try {
			assertRecentsQueryAll(null);
		} catch (Exception ex) {
			ex.printStackTrace();
			throw ex;
		}
	}

	@Test
	public void testQueryRecentsAllSince() throws TigaseDBException {
		assertRecentsQueryAll(start);
	}


	@Test
	public void testQueryRecentGroupchats() throws TigaseDBException {
		List<UnifiedArchiveRepository.Item> results = repo.queryRecents(owner, start, null, EnumSet.of(
				UnifiedArchiveRepository.Type.groupchat), EnumSet.allOf(UnifiedArchiveRepository.CallCondition.class),
																		new RSM()).collect(Collectors.toList());

		List<BareJID> expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());

		assertEquals(2, results.size());
		assertItem(results, expBuddies, newestMessageDate, 0, UnifiedArchiveRepository.ItemType.groupchat);
		assertItem(results, expBuddies, newestMessageDate,1, UnifiedArchiveRepository.ItemType.groupchat);
	}
	
	@Override
	protected Class<? extends DataSourceAware> getDataSourceAwareIfc() {
		return UnifiedArchiveRepository.class;
	}

	private void assertRecentsQueryAll(Date start) throws TigaseDBException {
		List<UnifiedArchiveRepository.Item> results = repo.queryRecents(owner, start, null, EnumSet.allOf(
				UnifiedArchiveRepository.Type.class), EnumSet.allOf(UnifiedArchiveRepository.CallCondition.class),
																		new RSM()).collect(Collectors.toList());

		List<BareJID> expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());

		assertEquals(2, results.size());
		assertItem(results, expBuddies, newestMessageDate, 0, UnifiedArchiveRepository.ItemType.groupchat);
		assertItem(results, expBuddies, newestMessageDate, 1, UnifiedArchiveRepository.ItemType.groupchat);
	}

	private void assertItem(List<UnifiedArchiveRepository.Item> results, List<BareJID> expBuddies, List<Date> expDates, int pos,
							UnifiedArchiveRepository.ItemType itemType) {
		assertItem(results, expBuddies, expDates, pos, itemType, null);
	}

	private void assertItem(List<UnifiedArchiveRepository.Item> results, List<BareJID> expBuddies, List<Date> expDates, int pos,
							UnifiedArchiveRepository.ItemType itemType, Predicate<UnifiedArchiveRepository.Item> itemPredicate) {
		UnifiedArchiveRepository.Item item = results.get(pos);
		assertEquals(expBuddies.get(pos).toString(), item.getWith());
		assertEquals(itemType, item.getItemType());
		assertTrue(expDates.get(pos).getTime()/1000 == item.getTimestamp().getTime()/1000);
		if (itemPredicate != null) {
			assertTrue(itemPredicate.test(item));
		}
	}
	
	private Element createGroupchat(BareJID owner, JID buddy, MessageArchiveRepository.Direction direction,
									String body) {
		Element message = new Element("message", new String[]{"from", "to"},
									  direction == MessageArchiveRepository.Direction.incoming
									  ? new String[]{buddy.toString(), owner.toString()}
									  : new String[]{owner.toString(), buddy.toString()});
		message.setAttribute("type", "groupchat");
		message.addChild(new Element("body", body));
		return message;

	}
}
