/*
 * Tigase Unified Archive Component - Extension of implementation of Message Archiving component for Tigase XMPP Server
 * Copyright (C) 2015 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.archive.unified.db;

import org.junit.*;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.model.Statement;
import tigase.archive.db.MessageArchiveRepository;
import tigase.component.exceptions.RepositoryException;
import tigase.db.*;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.Element;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;
import tigase.xmpp.rsm.RSM;

import java.sql.SQLException;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * Created by andrzej on 19.02.2017.
 */

public class JDBCUnifiedArchiveRepositoryWithRecentsTest extends AbstractDataSourceAwareTestCase<DataRepository,UnifiedArchiveRepository> {

	private static final String PROJECT_ID = "unified-archive";
	private static final String VERSION = "3.0.0-SNAPSHOT";

	@ClassRule
	public static TestRule rule = new TestRule() {
		@Override
		public Statement apply(Statement stmnt, Description d) {
			if (uri == null || !uri.startsWith("jdbc:")) {
				return new Statement() {
					@Override
					public void evaluate() throws Throwable {
						Assume.assumeTrue("Ignored due to not passed DB URI!", false);
					}
				};
			}
			return stmnt;
		}
	};
	
	private List<JID> buddies = new ArrayList<>();
	private BareJID owner;
	private UnifiedArchiveRepository<DataRepository> repo;
	private Date start;

	@BeforeClass
	public static void prepareTest() throws DBInitException {
		loadSchema(PROJECT_ID, VERSION, Collections.singleton("unified-archive"));
	}

	@Before
	public void setup()
			throws RepositoryException, InstantiationException, IllegalAccessException, TigaseStringprepException,
				   InterruptedException, SQLException, ClassNotFoundException {
		repo = getDataSourceAware();

		owner = BareJID.bareJIDInstance("owner-" + UUID.randomUUID().toString(), "test.com");

		start = new Date();
		Thread.sleep(1000);

		for (int i = 0; i < 5; i++) {
			JID buddy = JID.jidInstance("chat-" + UUID.randomUUID().toString(), "test.com", "res-1");
			buddies.add(buddy);
			switch (i) {
				case 0: {
					Element elem = createGroupchat(owner, buddy, MessageArchiveRepository.Direction.incoming,
												   UUID.randomUUID().toString());
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);

					elem = createJingle(owner, buddy, MessageArchiveRepository.Direction.incoming, "session-initiate",
										null);
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);

					elem = createChat(owner, buddy, MessageArchiveRepository.Direction.outgoing,
									  UUID.randomUUID().toString());
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);
					break;
				}
				case 1: {
					Element elem = createGroupchat(owner, buddy, MessageArchiveRepository.Direction.incoming,
												   UUID.randomUUID().toString());
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);

					elem = createChat(owner, buddy, MessageArchiveRepository.Direction.outgoing,
									  UUID.randomUUID().toString());
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);

					elem = createJingle(owner, buddy, MessageArchiveRepository.Direction.incoming, "session-terminate",
										"success");
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);

					elem = new Element("message", new String[]{"from", "to"},
									   new String[]{buddy.toString(), owner.toString()});
					elem.addChild(new Element("x", new Element[]{
							new Element("invite", new String[]{"from"}, new String[]{buddy.toString()})},
											  new String[]{"xmlns"},
											  new String[]{"http://jabber.org/protocol/muc#user"}));
					elem.setAttribute("tigase-offline", "true");
					elem.setAttribute("tigase-offline-only", "true");
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);

					Thread.sleep(1000);
					break;
				}
				case 2: {
					Element elem = createChat(owner, buddy, MessageArchiveRepository.Direction.outgoing,
											  UUID.randomUUID().toString());
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);

					elem = createJingle(owner, buddy, MessageArchiveRepository.Direction.outgoing, "session-terminate",
										"cancel");
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);
					break;
				}
				case 3: {
					Element elem = createGroupchat(owner, buddy, MessageArchiveRepository.Direction.incoming,
												   UUID.randomUUID().toString());
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);

					elem = createJingle(owner, buddy, MessageArchiveRepository.Direction.incoming, "session-initiate",
										null);
					elem.getChild("jingle").setAttribute("offline", "true");
					elem.setAttribute("tigase-offline", "true");
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);
					break;
				}
				case 4: {
					Element elem = createChat(owner, buddy, MessageArchiveRepository.Direction.outgoing,
											  UUID.randomUUID().toString());
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);

					elem = createGroupchat(owner, buddy, MessageArchiveRepository.Direction.incoming,
										   UUID.randomUUID().toString());
					repo.archiveMessage(owner, buddy, new Date(), elem,
										UUID.randomUUID().toString(), null);
					Thread.sleep(1000);
					break;
				}
			}
		}
	}

	@After
	public void tearDown() {
		buddies.forEach(buddy -> {
			try {
				repo.removeItems(owner, buddy.getBareJID().toString(), start, new Date());
			} catch (Exception ex) {
				ex.printStackTrace();
			}
		});
		repo.destroy();
	}

	@Test
	public void testQueryRecentsAll() throws TigaseDBException {
		try {
			assertRecentsQueryAll(null);
		} catch (Exception ex) {
			ex.printStackTrace();
			throw ex;
		}
	}

	@Test
	public void testQueryRecentsAllSince() throws TigaseDBException {
		assertRecentsQueryAll(start);
	}

	@Test
	public void testQueryRecentsAllPaginated() throws TigaseDBException {
		RSM rsm = new RSM();
		rsm.setMax(2);
		List<UnifiedArchiveRepository.Item> results = repo.queryRecents(owner, start, null, EnumSet.allOf(
				UnifiedArchiveRepository.Type.class), EnumSet.allOf(UnifiedArchiveRepository.CallCondition.class), rsm)
				.collect(Collectors.toList());

		List<BareJID> expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());
		Collections.reverse(expBuddies);
		expBuddies = expBuddies.subList(0, 2);
		assertItem(results, expBuddies, 0, UnifiedArchiveRepository.ItemType.groupchat);
		assertItem(results, expBuddies, 1, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionMissed);

		rsm = new RSM();
		rsm.setMax(2);
		rsm.setIndex(2);

		results = repo.queryRecents(owner, start, null, EnumSet.allOf(UnifiedArchiveRepository.Type.class),
									EnumSet.allOf(UnifiedArchiveRepository.CallCondition.class), rsm)
				.collect(Collectors.toList());

		expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());
		Collections.reverse(expBuddies);
		expBuddies = expBuddies.subList(2, 4);
		assertItem(results, expBuddies, 0, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionCancel);
		assertItem(results, expBuddies, 1, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionSuccess);
	}

	@Test
	public void testQueryRecentChats() throws TigaseDBException {
		List<UnifiedArchiveRepository.Item> results = repo.queryRecents(owner, start, null,
																		EnumSet.of(UnifiedArchiveRepository.Type.chat),
																		EnumSet.allOf(
																				UnifiedArchiveRepository.CallCondition.class),
																		new RSM()).collect(Collectors.toList());

		List<BareJID> expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());
		expBuddies.remove(3);
		Collections.reverse(expBuddies);
		assertItem(results, expBuddies, 0, UnifiedArchiveRepository.ItemType.chat);
		assertItem(results, expBuddies, 1, UnifiedArchiveRepository.ItemType.chat);
		assertItem(results, expBuddies, 2, UnifiedArchiveRepository.ItemType.chat);
		assertItem(results, expBuddies, 3, UnifiedArchiveRepository.ItemType.chat);
	}

	@Test
	public void testQueryRecentGroupchats() throws TigaseDBException {
		List<UnifiedArchiveRepository.Item> results = repo.queryRecents(owner, start, null, EnumSet.of(
				UnifiedArchiveRepository.Type.groupchat), EnumSet.allOf(UnifiedArchiveRepository.CallCondition.class),
																		new RSM()).collect(Collectors.toList());

		List<BareJID> expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());
		expBuddies.remove(2);
		Collections.reverse(expBuddies);
		assertItem(results, expBuddies, 0, UnifiedArchiveRepository.ItemType.groupchat);
		assertItem(results, expBuddies, 1, UnifiedArchiveRepository.ItemType.groupchat);
		assertItem(results, expBuddies, 2, UnifiedArchiveRepository.ItemType.groupchat);
		assertItem(results, expBuddies, 3, UnifiedArchiveRepository.ItemType.groupchat);
	}

	@Test
	public void testQueryRecentCalls() throws TigaseDBException {
		List<UnifiedArchiveRepository.Item> results = repo.queryRecents(owner, start, null,
																		EnumSet.of(UnifiedArchiveRepository.Type.call),
																		EnumSet.allOf(
																				UnifiedArchiveRepository.CallCondition.class),
																		new RSM()).collect(Collectors.toList());

		List<BareJID> expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());
		expBuddies.remove(4);
		Collections.reverse(expBuddies);
		assertItem(results, expBuddies, 0, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionMissed);
		assertItem(results, expBuddies, 1, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionCancel);
		assertItem(results, expBuddies, 2, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionSuccess);
		assertItem(results, expBuddies, 3, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionCancel);
	}

	@Test
	public void testQueryRecentCallsMissed() throws TigaseDBException {
		List<UnifiedArchiveRepository.Item> results = repo.queryRecents(owner, start, null,
																		EnumSet.of(UnifiedArchiveRepository.Type.call),
																		EnumSet.of(
																				UnifiedArchiveRepository.CallCondition.missed),
																		new RSM()).collect(Collectors.toList());

		List<BareJID> expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());
		Collections.reverse(expBuddies);
		expBuddies = Arrays.asList(expBuddies.get(1));
		assertItem(results, expBuddies, 0, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionMissed);
	}

	@Test
	public void testQueryRecentCallsSuccessAndMissed() throws TigaseDBException {
		List<UnifiedArchiveRepository.Item> results = repo.queryRecents(owner, start, null,
																		EnumSet.of(UnifiedArchiveRepository.Type.call),
																		EnumSet.of(
																				UnifiedArchiveRepository.CallCondition.success,
																				UnifiedArchiveRepository.CallCondition.missed),
																		new RSM()).collect(Collectors.toList());

		List<BareJID> expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());
		Collections.reverse(expBuddies);
		expBuddies = Arrays.asList(expBuddies.get(1), expBuddies.get(3));
		assertItem(results, expBuddies, 0, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionMissed);
		assertItem(results, expBuddies, 1, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionSuccess);
	}

	@Override
	protected Class<? extends DataSourceAware> getDataSourceAwareIfc() {
		return UnifiedArchiveRepository.class;
	}

	private void assertRecentsQueryAll(Date start) throws TigaseDBException {
		List<UnifiedArchiveRepository.Item> results = repo.queryRecents(owner, start, null, EnumSet.allOf(
				UnifiedArchiveRepository.Type.class), EnumSet.allOf(UnifiedArchiveRepository.CallCondition.class),
																		new RSM()).collect(Collectors.toList());

		List<BareJID> expBuddies = buddies.stream().map(buddy -> buddy.getBareJID()).collect(Collectors.toList());
		Collections.reverse(expBuddies);

		assertItem(results, expBuddies, 0, UnifiedArchiveRepository.ItemType.groupchat);
		assertItem(results, expBuddies, 1, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionMissed);
		assertItem(results, expBuddies, 2, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionCancel);
		assertItem(results, expBuddies, 3, UnifiedArchiveRepository.ItemType.call, this::assertCallConditionSuccess);
		assertItem(results, expBuddies, 4, UnifiedArchiveRepository.ItemType.chat);
	}

	private void assertItem(List<UnifiedArchiveRepository.Item> results, List<BareJID> expBuddies, int pos,
							UnifiedArchiveRepository.ItemType itemType) {
		assertItem(results, expBuddies, pos, itemType, null);
	}

	private void assertItem(List<UnifiedArchiveRepository.Item> results, List<BareJID> expBuddies, int pos,
							UnifiedArchiveRepository.ItemType itemType, Predicate<UnifiedArchiveRepository.Item> itemPredicate) {
		UnifiedArchiveRepository.Item item = results.get(pos);
		assertEquals(expBuddies.get(pos).toString(), item.getWith());
		assertEquals(itemType, item.getItemType());
		if (itemPredicate != null) {
			assertTrue(itemPredicate.test(item));
		}
	}

	private boolean assertCallConditionSuccess(UnifiedArchiveRepository.Item item) {
		Element jingle = item.getMessage().getChild("jingle");
		String action = jingle.getAttributeStaticStr("action");
		assertTrue("session-accept".equals(action) || "session-terminate".equals(action));
		Element reason = jingle.getChild("reason");
		assertTrue(reason == null || reason.getChild("cancel") == null);
		return true;
	}

	private boolean assertCallConditionCancel(UnifiedArchiveRepository.Item item) {
		Element jingle = item.getMessage().getChild("jingle");
		String action = jingle.getAttributeStaticStr("action");
		if ("session-initiate".equals(action)) {
			assertTrue(jingle.getAttributeStaticStr("offline") == null);
		} else {
			assertEquals("session-terminate", action);
			Element reason = jingle.getChild("reason");
			assertTrue(reason != null && reason.getChild("cancel") != null);
		}
		return true;
	}

	private boolean assertCallConditionMissed(UnifiedArchiveRepository.Item item) {
		Element jingle = item.getMessage().getChild("jingle");
		String action = jingle.getAttributeStaticStr("action");
		assertEquals("session-initiate", action);
		assertTrue(jingle.getAttributeStaticStr("offline") != null);
		return true;
	}

	private Element createJingle(BareJID owner, JID buddy, MessageArchiveRepository.Direction direction, String action,
								 String reason) {
		Element iq = new Element("iq", new String[]{"from", "to"},
								 direction == MessageArchiveRepository.Direction.incoming
								 ? new String[]{buddy.toString(), owner.toString()}
								 : new String[]{owner.toString(), buddy.toString()});
		iq.setAttribute("id", UUID.randomUUID().toString());
		iq.setAttribute("item-type", "call");
		Element jingle = new Element("jingle", new String[]{"sid", "action"},
									 new String[]{UUID.randomUUID().toString(), action});
		if (reason != null) {
			Element reasonEl = new Element("reason");
			reasonEl.addChild(new Element(reason));
			jingle.addChild(reasonEl);
		}
		iq.addChild(jingle);

		return iq;
	}

	private Element createChat(BareJID owner, JID buddy, MessageArchiveRepository.Direction direction, String body) {
		Element message = new Element("message", new String[]{"from", "to"},
									  direction == MessageArchiveRepository.Direction.incoming
									  ? new String[]{buddy.toString(), owner.toString()}
									  : new String[]{owner.toString(), buddy.toString()});
		message.addChild(new Element("body", body));
		return message;
	}

	private Element createGroupchat(BareJID owner, JID buddy, MessageArchiveRepository.Direction direction,
									String body) {
		Element message = new Element("message", new String[]{"from", "to"},
									  direction == MessageArchiveRepository.Direction.incoming
									  ? new String[]{buddy.toString(), owner.toString()}
									  : new String[]{owner.toString(), buddy.toString()});
		message.setAttribute("type", "groupchat");
		message.addChild(new Element("body", body));
		return message;

	}
}
