/*
 * Tigase Unified Archive Component - Extension of implementation of Message Archiving component for Tigase XMPP Server
 * Copyright (C) 2015 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.archive.unified.db;

import org.junit.*;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.MethodSorters;
import org.junit.runners.model.Statement;
import tigase.archive.db.MessageArchiveRepository;
import tigase.archive.unified.QueryCriteria;
import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.db.*;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.Element;
import tigase.xmpp.StanzaType;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;

/**
 * @author andrzej
 */
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public class JDBCUnifiedArchiveRepositoryTest extends AbstractDataSourceAwareTestCase<DataRepository,UnifiedArchiveRepository> {

	private final static SimpleDateFormat formatter2 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");

	private static final String PROJECT_ID = "unified-archive";
	private static final String VERSION = "3.0.0-SNAPSHOT";
	protected static String emoji = "\uD83D\uDE97\uD83D\uDCA9\uD83D\uDE21";
	private static JID buddy = null;
	private static JID owner = null;
	// this is static to pass date from first test to next one
	private static Date testStart = null;

	@ClassRule
	public static TestRule rule = new TestRule() {
		@Override
		public Statement apply(Statement stmnt, Description d) {
			if (uri == null || !uri.startsWith("jdbc:")) {
				return new Statement() {
					@Override
					public void evaluate() throws Throwable {
						Assume.assumeTrue("Ignored due to not passed DB URI!", false);
					}
				};
			}
			return stmnt;
		}
	};

	@BeforeClass
	public static void prepareTest() throws DBInitException {
		loadSchema(PROJECT_ID, VERSION, Collections.singleton("unified-archive"));

		owner = JID.jidInstanceNS("ua-" + UUID.randomUUID(), "test", "tigase-1");
		buddy = JID.jidInstanceNS("ua-" + UUID.randomUUID(), "test", "tigase-2");
	}
	static {
		formatter2.setTimeZone(TimeZone.getTimeZone("UTC"));
	}

	protected boolean checkEmoji = true;
	private UnifiedArchiveRepository<DataRepository> repo;

	@Before
	public void setup() throws RepositoryException, InstantiationException, IllegalAccessException, SQLException,
							   ClassNotFoundException {
		repo = getDataSourceAware();
		
		if (uri.startsWith("jdbc:derby:") || uri.startsWith("jdbc:jtds:sqlserver") ||
				uri.startsWith("jdbc:sqlserver")) {
			checkEmoji = false;
		}
	}

	@Override
	protected Class<? extends DataSourceAware> getDataSourceAwareIfc() {
		return UnifiedArchiveRepository.class;
	}
	
	// This test to work requires change in archiveMessage method to throw Exception - by default exception is catched in this method
//	@Test
//	@Ignore
//	public void testDeadlocksOnInsert() throws InterruptedException {
//		try {
//			Map<String,String> params = new HashMap<String,String>();
//			params.put(RepositoryFactory.DATA_REPO_POOL_SIZE_PROP_KEY, "90");
//			JDBCUnifiedArchiveRepository repo = new JDBCUnifiedArchiveRepository();
//			repo.initRepository(uri, params);
//
//			Queue<Thread> threads = new ArrayDeque<Thread>();
//			for (int i=0; i<160; i++) {
//				final int ti = i;
//				final JID jid = JID.jidInstanceNS("user-"+i+"@test-domain.com/res-1");
//				Thread t = new Thread() {
//					@Override
//					public void run() {
//						try {
//						for (int j=0; j<1000; j++) {
//							Element message = new Element("message", new String[] { "from", "to", "type", "id"}, new String[] { jid.toString(), jid.getBareJID().toString(), "set", UUID.randomUUID().toString() });
//							message.addChild(new Element("bind", UUID.randomUUID().toString(), new String[] { "action" }, new String[] { "login" }));
//							repo.archiveMessage(jid.getBareJID(), jid, MessageArchiveRepository.Direction.incoming, new Date(), message, new HashSet<String>());
//						}
//							System.out.println("executed last insert for thread " + ti);
//						} catch (Exception ex) {
//							ex.printStackTrace();
//						}
//					}
//				};
//				threads.offer(t);
//				t.start();
//			}
//			Thread t = null;
//			while ((t = threads.poll()) != null) {
//				t.join();
//			}
//		} catch (DBInitException ex) {
//			ex.printStackTrace();
//		}
//	}

	@Test
	public void test1_archiveMessage1() throws RepositoryException, ComponentException {
		if (uri == null) {
			return;
		}
		Date date = new Date();
		testStart = date;
		String body = "Test 1";
		if (checkEmoji) {
			body += emoji;
		}
		Element msg = new Element("message", new String[]{"from", "to", "type"},
								  new String[]{owner.toString(), buddy.toString(), StanzaType.chat.name()});
		msg.addChild(new Element("body", body));
		repo.archiveMessage(owner.getBareJID(), buddy, date, msg, UUID.randomUUID().toString(), null);

		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(date);
		crit.getRsm().setMax(1);
		List<UnifiedArchiveRepository.Item> msgs = new ArrayList<>();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Incorrect number of message", 1, msgs.size());

		UnifiedArchiveRepository.Item res = msgs.get(0);
		Assert.assertEquals("Incorrect direction", MessageArchiveRepository.Direction.outgoing, res.getDirection());
		Assert.assertEquals("Incorrect message body", body, res.getMessage().getChildCData("message/body"));
	}

	@Test
	public void test2_archiveMessage2withTags() throws InterruptedException, RepositoryException, ComponentException {
		Thread.sleep(2000);
		Date date = new Date();
		String body = "Test 2 with #Test123";
		if (checkEmoji) {
			body += emoji;
		}
		Element msg = new Element("message", new String[]{"from", "to", "type"},
								  new String[]{buddy.toString(), owner.toString(), StanzaType.chat.name()});
		msg.addChild(new Element("body", body));
		Set<String> tags = new HashSet<String>();
		if (checkEmoji) {
			tags.add("#Test123" + emoji);
		} else {
			tags.add("#Test123");
		}
		repo.archiveMessage(owner.getBareJID(), buddy, date, msg, UUID.randomUUID().toString(), tags);

		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(date);
		crit.getRsm().setMax(1);

		List<UnifiedArchiveRepository.Item> items = new ArrayList<>();
		repo.queryItems(crit, (query, item) -> items.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Incorrect number of message", 1, items.size());

		UnifiedArchiveRepository.Item res = items.get(0);
		Assert.assertEquals("Incorrect direction", MessageArchiveRepository.Direction.incoming, res.getDirection());
		Assert.assertEquals("Incorrect message body", body, res.getMessage().getChildCData("message/body"));
	}

	@Test
	public void test3_getCollections() throws TigaseDBException {
		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(testStart);

		List<ColItem> chats = new ArrayList<>();
		repo.queryCollections(crit, (query, col) -> chats.add(new ColItem(col)));
		Assert.assertEquals("Incorrect number of collections", 1, chats.size());

		ColItem chat = chats.get(0);
		Assert.assertEquals("Incorrect buddy", buddy.getBareJID().toString(), chat.with);
		Assert.assertEquals("Incorrect timestamp", testStart.getTime() / 1000, chat.start.getTime() / 1000);
	}

	@Test
	public void test3_getCollectionsByTag() throws TigaseDBException {
		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(testStart);
		if (checkEmoji) {
			crit.addTag("#Test123" + emoji);
		} else {
			crit.addTag("#Test123");
		}
		List<ColItem> chats = new ArrayList<>();
		repo.queryCollections(crit, (query, col) -> chats.add(new ColItem(col)));
		Assert.assertEquals("Incorrect number of collections", 1, chats.size());
		ColItem chat = chats.get(0);
		Assert.assertEquals("Incorrect buddy", buddy.getBareJID().toString(), chat.with);
	}

	@Test
	public void test4_getItems() throws InterruptedException, RepositoryException, ComponentException {
		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(testStart);

		List<UnifiedArchiveRepository.Item> msgs = new ArrayList<>();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Incorrect number of message", 2, msgs.size());

		UnifiedArchiveRepository.Item res = msgs.get(0);
		Assert.assertEquals("Incorrect direction", MessageArchiveRepository.Direction.outgoing, res.getDirection());
		Assert.assertEquals("Incorrect message body", "Test 1" + (checkEmoji ? emoji : ""),
							res.getMessage().getChildCData("message/body"));

		res = msgs.get(1);
		Assert.assertEquals("Incorrect direction", MessageArchiveRepository.Direction.incoming, res.getDirection());
		Assert.assertEquals("Incorrect message body", "Test 2 with #Test123" + (checkEmoji ? emoji : ""),
							res.getMessage().getChildCData("message/body"));

		crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(testStart);

		msgs.clear();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertTrue("Incorrect number of message", msgs.size() >= 1);
	}

	@Test
	public void test4_getItemsWithTag() throws InterruptedException, RepositoryException, ComponentException {
		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(testStart);
		if (checkEmoji) {
			crit.addTag("#Test123" + emoji);
		} else {
			crit.addTag("#Test123");
		}

		List<UnifiedArchiveRepository.Item> msgs = new ArrayList<>();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Incorrect number of message", 1, msgs.size());

		UnifiedArchiveRepository.Item res = msgs.get(0);
		Assert.assertEquals("Incorrect direction", MessageArchiveRepository.Direction.incoming, res.getDirection());
		Assert.assertEquals("Incorrect message body", "Test 2 with #Test123" + (checkEmoji ? emoji : ""),
							res.getMessage().getChildCData("message/body"));

		crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setStart(testStart);
		if (checkEmoji) {
			crit.addTag("#Test123" + emoji);
		} else {
			crit.addTag("#Test123");
		}

		msgs.clear();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertTrue("Incorrect number of message", msgs.size() >= 1);
	}

	@Test
	public void test5_getCollectionsContains() throws TigaseDBException {
		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(testStart);
		crit.addContains("Test 1");

		List<ColItem> chats = new ArrayList<>();
		repo.queryCollections(crit, (query, col) -> chats.add(new ColItem(col)));
		Assert.assertEquals("Incorrect number of collections", 1, chats.size());

		ColItem chat = chats.get(0);
		Assert.assertEquals("Incorrect buddy", buddy.getBareJID().toString(), chat.with);
		Assert.assertEquals("Incorrect timestamp", testStart.getTime() / 1000, chat.start.getTime() / 1000);

		crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(testStart);
		crit.addContains("Test 123");

		chats.clear();
		repo.queryCollections(crit, (query, col) -> chats.add(new ColItem(col)));
		Assert.assertEquals("Incorrect number of collections", 0, chats.size());
	}

	@Test
	public void test6_getItems() throws InterruptedException, RepositoryException, ComponentException {
		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(testStart);
		crit.addContains("Test 1");

		List<UnifiedArchiveRepository.Item> msgs = new ArrayList<>();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Incorrect number of message", 1, msgs.size());

		UnifiedArchiveRepository.Item res = msgs.get(0);
		Assert.assertEquals("Incorrect direction", MessageArchiveRepository.Direction.outgoing, res.getDirection());
		Assert.assertEquals("Incorrect message body", "Test 1" + (checkEmoji ? emoji : ""),
							res.getMessage().getChildCData("message/body"));
	}

	@Test
	public void test7_removeItems() throws RepositoryException, ComponentException {
		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(testStart);

		List<UnifiedArchiveRepository.Item> msgs = new ArrayList<>();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertNotEquals("No messages in repository to execute test - we should have some already", 0,
							   msgs.size());
		repo.removeItems(owner.getBareJID(), buddy.getBareJID().toString(), testStart, new Date());

		msgs.clear();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Still some messages, while in this duration all should be deleted", 0, msgs.size());
	}

	@Test
	public void test8_removeExpiredItems() throws RepositoryException, TigaseStringprepException, ComponentException {
		Date date = new Date();
		String uuid = UUID.randomUUID().toString();
		testStart = date;
		String body = "Test 1 " + uuid;
		if (checkEmoji) {
			body += emoji;
		}
		Element msg = new Element("message", new String[]{"from", "to", "type"},
								  new String[]{owner.toString(), buddy.toString(), StanzaType.chat.name()});
		msg.addChild(new Element("body", body));
		Element delay = new Element("delay");
		LocalDateTime time = LocalDateTime.now().minusDays(1).minusHours(1);
		Date originalTime = new Date(time.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli());
		delay.setAttribute("stamp", formatter2.format(originalTime));
		msg.addChild(delay);
		repo.archiveMessage(owner.getBareJID(), buddy, originalTime, msg,
							UUID.randomUUID().toString(), null);

		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.addContains(uuid);
		crit.getRsm().setMax(1);
		List<UnifiedArchiveRepository.Item> msgs = new ArrayList<>();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Incorrect number of messages", 1, msgs.size());

		crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.setStart(date);
		crit.addContains(uuid);
		crit.getRsm().setMax(1);
		msgs.clear();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Incorrect number of messages", 0, msgs.size());

		LocalDateTime before = LocalDateTime.now().minusDays(1);
		repo.deleteExpiredMessages(BareJID.bareJIDInstance(owner.getDomain()), before);

		crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		crit.addContains(uuid);
		crit.getRsm().setMax(1);
		msgs.clear();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Incorrect number of messages", 0, msgs.size());
	}

	@Test
	public void testOfflineMessagesExcludedFromArchive() throws ComponentException, RepositoryException {
		Element elem = new Element("message", new String[]{"from", "to"},
								   new String[]{buddy.toString(), owner.toString()});
		elem.addChild(new Element("x", new Element[]{
				new Element("invite", new String[]{"from"}, new String[]{buddy.toString()})}, new String[]{"xmlns"},
								  new String[]{"http://jabber.org/protocol/muc#user"}));
		elem.setAttribute("tigase-offline", "true");
		elem.setAttribute("tigase-offline-only", "true");
		repo.archiveMessage(owner.getBareJID(), buddy, new Date(), elem,
							UUID.randomUUID().toString(), null);

		QueryCriteria crit = repo.newQuery();
		crit.setQuestionerJID(owner);
		crit.setWith(JID.jidInstance(buddy.getBareJID()));
		List<UnifiedArchiveRepository.Item> msgs = new ArrayList<>();
		repo.queryItems(crit, (query, item) -> msgs.add((UnifiedArchiveRepository.Item) item));
		Assert.assertEquals("Messages which should not be availble, are available!", 0, msgs.size());
	}

	class ColItem {

		Date start;
		String type;
		String with;

		public ColItem(MessageArchiveRepository.Collection col) {
			this.with = col.getWith();
			this.start = col.getStartTs();
			this.type = ((JDBCUnifiedArchiveRepository.Collection) col).getType();
		}
	}
}
