/*
 * Tigase Unified Archive Component - Extension of implementation of Message Archiving component for Tigase XMPP Server
 * Copyright (C) 2015 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.archive.unified.db;

import org.junit.*;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.MethodSorters;
import org.junit.runners.model.Statement;
import tigase.archive.processors.MessageArchivePlugin;
import tigase.archive.unified.QueryCriteria;
import tigase.archive.unified.db.JDBCFlexibleOfflineMessageRetrievalRepository.MSG_TYPES;
import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.db.DBInitException;
import tigase.db.DataRepository;
import tigase.db.DataSourceAware;
import tigase.server.xmppsession.SessionManagerHandler;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.Element;
import tigase.xmpp.NotAuthorizedException;
import tigase.xmpp.StanzaType;
import tigase.xmpp.XMPPResourceConnection;
import tigase.xmpp.XMPPSession;
import tigase.xmpp.impl.AbstractProcessorWithDataSourceAwareTestCase;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

import static org.junit.Assert.*;

/**
 * @author andrzej
 */
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public class JDBCFlexibleOfflineMessageRespositoryTest
		extends AbstractProcessorWithDataSourceAwareTestCase<DataRepository,UnifiedArchiveRepository> {

	private static final Logger log = Logger.getLogger(
			JDBCFlexibleOfflineMessageRespositoryTest.class.getCanonicalName());

	private final static SimpleDateFormat formatter2 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");

	private static final String PROJECT_ID = "unified-archive";
	private static final String VERSION = "3.0.0";
	@ClassRule
	public static TestRule rule = new TestRule() {
		@Override
		public Statement apply(Statement stmnt, Description d) {
			if (uri == null || !uri.startsWith("jdbc:")) {
				return new Statement() {
					@Override
					public void evaluate() throws Throwable {
						Assume.assumeTrue("Ignored due to not passed DB URI!", false);
					}
				};
			}
			return stmnt;
		}
	};

	@BeforeClass
	public static void prepareTest() throws DBInitException {
		loadSchema(PROJECT_ID, VERSION, Collections.singleton("unified-archive"));
	}

	static {
		formatter2.setTimeZone(TimeZone.getTimeZone("UTC"));
	}

	private JID buddy = null;
	private JID owner = null;
	private JID ownerConnId = JID.jidInstanceNS("c2s@zeus/14ds34d-sersrs");
	private JDBCFlexibleOfflineMessageRetrievalRepository repo;
	// this is static to pass date from first test to next one
	private Date testStart = null;
	
	@Before
	public void setup() throws Exception {
		getKernel().registerBean(MessageArchivePlugin.class).setActive(true).exec();
		getKernel().registerBean(JDBCFlexibleOfflineMessageRetrievalRepository.class).exportable().setActive(true).exec();

		repo = getKernel().getInstance(JDBCFlexibleOfflineMessageRetrievalRepository.class);
		repo.setDataSource(dataSource);

		owner = JID.jidInstanceNS("ua-" + UUID.randomUUID(), "test", "tigase-1");
		buddy = JID.jidInstanceNS("ua-" + UUID.randomUUID(), "test", "tigase-2");
		getUserRepository().addUser(owner.getBareJID());
		getUserRepository().addUser(buddy.getBareJID());
	}
	
	@After
	public void tearDown() throws Exception {
		repo = null;
	}

	@Test
	public void testOfflineMessageRetrieval()
			throws RepositoryException, TigaseStringprepException, NotAuthorizedException, ComponentException {
		XMPPResourceConnection conn = getXMPPResourceConnection(ownerConnId, owner);
		repo.deleteMessagesToJID(null, conn);

		Date date = new Date();
		String body = "Test 1";
		Element msg = new Element("message", new String[]{"from", "to", "type"},
								  new String[]{buddy.toString(), owner.toString(), StanzaType.chat.name()});
		msg.addChild(new Element("body", body));
		repo.storeMessage(buddy, owner, date, msg, null);
		body = "Test 2 with #Test123";
		msg = new Element("message", new String[]{"from", "to", "type"},
						  new String[]{buddy.toString(), owner.toString(), StanzaType.chat.name()});
		msg.addChild(new Element("body", body));
		repo.storeMessage(buddy, owner, date, msg, null);

		Element presence = new Element("presence", new String[]{"from", "to", "type"},
									   new String[]{buddy.toString(), owner.toString(), StanzaType.subscribe.name()});
		repo.storeMessage(buddy, owner, date, presence, null);

		msg = new Element("message", new String[]{"from", "to"}, new String[]{buddy.toString(), owner.toString()});
		msg.addChild(new Element("x", new Element[]{
				new Element("invite", new String[]{"from"}, new String[]{buddy.toString()})}, new String[]{"xmlns"},
								 new String[]{"http://jabber.org/protocol/muc#user"}));
		assertTrue(repo.storeMessage(buddy, owner, new Date(), msg, null));

		Map<Enum, Long> counts = repo.getMessagesCount(owner);
		assertNotNull(counts);
		assertEquals(2, counts.size());
		assertEquals("Wrong number of messages for user " + owner, 3, (long) counts.get(MSG_TYPES.message));
		assertEquals(1, (long) counts.get(MSG_TYPES.presence));

		Queue<Element> msgs = repo.loadMessagesToJID(conn, false);
		assertEquals(4, msgs.size());

		List<Element> items = repo.getMessagesList(owner);
		assertEquals(4, items.size());

		String msgId = items.get(0).getAttribute("node");
		msgs = repo.loadMessagesToJID(Collections.singletonList(msgId), conn, true, null);
		assertEquals(1, msgs.size());

		msgs = repo.loadMessagesToJID(conn, false);
		assertEquals(3, msgs.size());

		repo.deleteMessagesToJID(null, conn);

		// there should be no messages left in MA
		UnifiedArchiveRepository ua_repo = getInstance(UnifiedArchiveRepository.class);
		QueryCriteria crit = (QueryCriteria) ua_repo.newQuery();
		crit.setQuestionerJID(owner);
		ua_repo.queryItems(crit, (query, item) -> assertNotNull(item.getMessage().getChild("body")));
	}

	@Override
	protected Class<? extends DataSourceAware> getDataSourceAwareIfc() {
		return UnifiedArchiveRepository.class;
	}

	protected XMPPResourceConnection getXMPPResourceConnection(JID connId, JID owner)
			throws TigaseStringprepException, NotAuthorizedException {
		return super.getSession(connId, owner);
	}

	private class SessionManagerHandlerImpl
			implements SessionManagerHandler {

		Map<BareJID, XMPPSession> sessions = new HashMap<BareJID, XMPPSession>();

		public SessionManagerHandlerImpl() {
		}

		@Override
		public JID getComponentId() {
			return JID.jidInstanceNS("sess-man@localhost");
		}

		@Override
		public void handleLogin(BareJID userId, XMPPResourceConnection conn) {
			XMPPSession session = sessions.get(userId);
			if (session == null) {
				session = new XMPPSession(userId.getLocalpart());
				sessions.put(userId, session);
			}
			try {
				session.addResourceConnection(conn);
			} catch (TigaseStringprepException ex) {
				log.log(Level.SEVERE, null, ex);
			}
		}

		@Override
		public void handleLogout(BareJID userId, XMPPResourceConnection conn) {
			XMPPSession session = sessions.get(conn);
			if (session != null) {
				session.removeResourceConnection(conn);
				if (session.getActiveResourcesSize() == 0) {
					sessions.remove(userId);
				}
			}
		}

		@Override
		public void handlePresenceSet(XMPPResourceConnection conn) {
		}

		@Override
		public void handleResourceBind(XMPPResourceConnection conn) {
		}

		@Override
		public boolean isLocalDomain(String domain, boolean includeComponents) {
			return !domain.contains("-ext");
		}

		@Override
		public void handleDomainChange(String domain, XMPPResourceConnection conn) {
		}
	}

}
