/*
 * Tigase XMPP Server - The instant messaging server
 * Copyright (C) 2004 Tigase, Inc. (office@tigase.com)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 */
package tigase.server.websocket;

import junit.framework.TestCase;
import org.junit.Assert;
import org.junit.Test;
import tigase.util.Base64;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * @author andrzej
 */
public class WebSocketHybiTest
		extends TestCase {

	private WebSocketHybi impl;

	@Test
	public void testCalculateWsAcceptKey() throws Exception {
		// From RFC 6455
		assertEquals("s3pPLMBiTxaQ9kYGzzhZRbK+xOo=", WebSocketHybi.calculateWsAcceptKey("dGhlIHNhbXBsZSBub25jZQ=="));

		// Responses generated by echo.websocket.org
		assertEquals("3YUhy9uOn3SwgBgA9kDld2AH2r8=", WebSocketHybi.calculateWsAcceptKey("Ox0pU/Y2qstxzCqKHwtJcQ=="));
		assertEquals("4vvUf3Izc4cQeL8fr4oZX5qxxFs=", WebSocketHybi.calculateWsAcceptKey("tsmEcS/j84lYkxw4SyfCYg=="));
	}

	@Test
	public void testFrameEncodingDecoding() throws IOException {
		String input = "<test-data><subdata/></test-data>";
		ByteBuffer buf = ByteBuffer.wrap(input.getBytes());
		final ByteBuffer tmp = ByteBuffer.allocate(1024);
		WebSocketXMPPIOService<Object> io = new WebSocketXMPPIOService<Object>(
				new WebSocketProtocolIfc[]{new WebSocketHybi()}) {

			@Override
			protected void writeBytes(ByteBuffer data) {
				tmp.put(data);
			}

		};
		io.maskingKey = new byte[4];
		impl.encodeFrameAndWrite(io, buf);
		tmp.flip();
		ByteBuffer tmp1 = maskFrame(tmp);
		ByteBuffer decoded = impl.decodeFrame(io, tmp1);
		Assert.assertArrayEquals("Data before encoding do not match data after decoding", input.getBytes(),
								 decoded.array());
	}

	@Test
	public void testHandshakeFail() throws NoSuchAlgorithmException, IOException {
		final ByteBuffer tmp = ByteBuffer.allocate(2048);
		WebSocketXMPPIOService<Object> io = new WebSocketXMPPIOService<Object>(
				new WebSocketProtocolIfc[]{new WebSocketHybi()}) {

			@Override
			public int getLocalPort() {
				return 80;
			}

			@Override
			protected void writeBytes(ByteBuffer data) {
				tmp.put(data);
			}

		};
		Map<String, String> params = new HashMap<String, String>();
		params.put("Sec-WebSocket-Key1".toUpperCase(), "1C2J899_05  6  !  M 9    ^4");
		params.put("Sec-WebSocket-Key2".toUpperCase(), "23 2ff0M_E0#.454X23");
		params.put("Sec-WebSocket-Protocol".toUpperCase(), "xmpp");
		byte[] bytes = new byte[10];
		bytes[0] = '\r';
		bytes[1] = '\n';
		Assert.assertFalse("Handshake succeeded", impl.handshake(io, params, bytes));
	}

	@Test
	public void testHandshakeOK() throws NoSuchAlgorithmException, IOException {
		final ByteBuffer tmp = ByteBuffer.allocate(2048);
		final StringBuilder sb = new StringBuilder();
		WebSocketXMPPIOService<Object> io = new WebSocketXMPPIOService<Object>(
				new WebSocketProtocolIfc[]{new WebSocketHybi()}) {

			@Override
			public int getLocalPort() {
				return 80;
			}

			@Override
			protected void writeBytes(ByteBuffer data) {
				tmp.put(data);
			}

			protected void writeData(String data) {
				sb.append(data);
			}

		};
		Map<String, String> params = new HashMap<String, String>();
		params.put("Sec-WebSocket-Version".toUpperCase(), "13");
		params.put("Sec-WebSocket-Key".toUpperCase(), "some random data as a key");
		params.put("Sec-WebSocket-Protocol".toUpperCase(), "xmpp");
		byte[] bytes = new byte[10];
		bytes[0] = '\r';
		bytes[1] = '\n';
		Assert.assertTrue("Handshake failed", impl.handshake(io, params, bytes));
		tmp.flip();
		byte[] read = new byte[tmp.remaining()];
//		Arrays.stream(sb.toString().split("\n")).forEach(System.out::println);
		Optional<String> secWebSocketAccept = Arrays.stream(sb.toString().split("\n"))
				.map(line -> line.split(":"))
				.filter(line -> "Sec-WebSocket-Accept".equalsIgnoreCase(line[0].trim()))
				.map(line -> line[1])
				.map(String::trim)
				.findFirst();

		String expSecWebSocketAccept = Base64.encode(MessageDigest.getInstance("SHA-1")
															 .digest(("some random data as a key" +
																	 "258EAFA5-E914-47DA-95CA-C5AB0DC85B11").getBytes(
																	 Charset.forName("UTF-8"))));
		assertEquals(expSecWebSocketAccept, secWebSocketAccept.get());
	}

	@Test
	public void testTwoWebSocketTextFramesInSingleTcpFrame() throws Exception {
		String input1 = "<test-data><subdata/></test-data>";
		String input2 = "<test2/>";
		ByteBuffer frame1 = generateIncomingFrame(input1);
		ByteBuffer frame2 = generateIncomingFrame(input2);

		ByteBuffer tmp = ByteBuffer.allocate(frame1.remaining() + frame2.remaining());
		tmp.put(frame1);
		tmp.put(frame2);
		tmp.flip();

		WebSocketXMPPIOService<Object> io = new WebSocketXMPPIOService<Object>(
				new WebSocketProtocolIfc[]{new WebSocketHybi()});
		io.maskingKey = new byte[4];
		ByteBuffer decoded = impl.decodeFrame(io, tmp);
		Assert.assertArrayEquals("Data of first frame before encoding do not match data after decoding",
								 input1.getBytes(), decoded.array());
		decoded = impl.decodeFrame(io, tmp);
		Assert.assertArrayEquals("Data of second frame before encoding do not match data after decoding",
								 input2.getBytes(), decoded.array());
	}

	@Test
	public void testTwoWebSocketFramesPingAndTextFrameInSingleTcpFrame() throws Exception {
		String input2 = "<test-data><subdata/></test-data>";
		ByteBuffer frame1 = ByteBuffer.allocate(20);
		frame1.put((byte) 0x89);
		frame1.put((byte) 0x04);
		frame1.put(new byte[]{0x00, 0x00, 0x00, 0x00});
		frame1.flip();
		frame1 = maskFrame(frame1);
		ByteBuffer frame2 = generateIncomingFrame(input2);

		ByteBuffer tmp = ByteBuffer.allocate(frame1.remaining() + frame2.remaining());
		tmp.put(frame1);
		tmp.put(frame2);
		tmp.flip();

		ByteBuffer tmp2 = ByteBuffer.allocate(1024);
		WebSocketXMPPIOService<Object> io = new WebSocketXMPPIOService<Object>(
				new WebSocketProtocolIfc[]{new WebSocketHybi()}) {
			@Override
			protected void writeBytes(ByteBuffer data) {
				tmp2.put(data);
			}
		};
		io.maskingKey = new byte[4];
		ByteBuffer decoded = impl.decodeFrame(io, tmp);
		Assert.assertNotNull(decoded);
		Assert.assertArrayEquals("Data of first frame before encoding do not match data after decoding", new byte[0],
								 decoded.array());
		tmp2.flip();
		Assert.assertNotEquals("PONG frame not sent!", 0, tmp2.remaining());
		assertEquals("PONG frame not sent!", (byte) 0x8A, tmp2.get(0));

		decoded = impl.decodeFrame(io, tmp);
		Assert.assertArrayEquals("Data of second frame before encoding do not match data after decoding",
								 input2.getBytes(), decoded.array());

	}

	@Override
	protected void setUp() throws Exception {
		impl = new WebSocketHybi();
	}

	@Override
	protected void tearDown() throws Exception {
		impl = null;
	}

	private ByteBuffer maskFrame(ByteBuffer data) {
		ByteBuffer tmp = ByteBuffer.allocate(1024);
		byte[] header = new byte[2];
		data.get(header);
		header[header.length - 1] = (byte) (header[header.length - 1] | 0x80);
		tmp.put(header);
		byte[] mask = {0x00, 0x00, 0x00, 0x00};
		tmp.put(mask);
		byte b;
		while (data.hasRemaining()) {
			b = data.get();
			b = (byte) (b ^ 0x00);
			tmp.put(b);
		}
		tmp.flip();
		return tmp;
	}

	private ByteBuffer generateIncomingFrame(String input) throws IOException {
		ByteBuffer buf = ByteBuffer.wrap(input.getBytes());
		final ByteBuffer tmp = ByteBuffer.allocate(1024);
		WebSocketXMPPIOService<Object> io = new WebSocketXMPPIOService<Object>(
				new WebSocketProtocolIfc[]{new WebSocketHybi()}) {

			@Override
			protected void writeBytes(ByteBuffer data) {
				tmp.put(data);
			}

		};
		io.maskingKey = new byte[4];
		impl.encodeFrameAndWrite(io, buf);
		tmp.flip();
		return maskFrame(tmp);
	}
}
