/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.repositories;

import org.junit.After;
import org.junit.Before;
import org.junit.FixMethodOrder;
import org.junit.Test;
import org.junit.runners.MethodSorters;
import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.db.AbstractDataSourceAwareTestCase;
import tigase.db.DBInitException;
import tigase.db.DataSource;
import tigase.db.DataSourceAware;
import tigase.push.api.IPushRepository;
import tigase.push.api.IPushSettings;
import tigase.xmpp.Authorization;
import tigase.xmpp.jid.BareJID;

import java.util.UUID;
import java.util.stream.Stream;

import static org.junit.Assert.*;

/**
 * Created by andrzej on 03.01.2017.
 */
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public abstract class AbstractIPushRepositoryTest<DS extends DataSource> extends AbstractDataSourceAwareTestCase<DS, IPushRepository> {

	private static String[] devices = new String[]{"device-1", "device-2"};
	private static String[] nodes = new String[devices.length];
	private static BareJID serviceJid = BareJID.bareJIDInstanceNS("push.example.com");
	private static BareJID user2Jid = BareJID.bareJIDInstanceNS("user2-" + UUID.randomUUID() + "@example.com");
	private static BareJID userJid = BareJID.bareJIDInstanceNS("user-" + UUID.randomUUID() + "@example.com");
	protected IPushRepository repository;

	@Before
	public void setup() throws RepositoryException, DBInitException, IllegalAccessException, InstantiationException {
		repository = getDataSourceAware();
	}

	@After
	public void cleanUp() {
		repository = null;
	}

	@Test
	public void test01_testMissingNode() throws Exception {
		assertNull(repository.getNodeSettings(serviceJid, "some-node"));
	}

	@Test
	public void test02_testDeviceRegistration1() throws Exception {
		IPushSettings settings = repository.registerDevice(serviceJid, userJid, "dummy", devices[0], null);
		assertNotNull(settings);
		assertNotNull(settings.getNode());
		nodes[0] = settings.getNode();
	}

	@Test
	public void test03_checkNode() throws Exception {
		IPushSettings settings = repository.getNodeSettings(serviceJid, nodes[0]);
		assertNotNull(settings);
		assertEquals(nodes[0], settings.getNode());
		assertArrayEquals(new String[]{devices[0]}, settings.getDevices()
				.stream()
				.map(device -> device.getDeviceId())
				.toArray(size -> new String[size]));
		assertEquals(userJid, settings.getOwnerJid());

//		settings = repository.getNodeSettings(userJid, "dummy", devices[0]);
//		assertNotNull(settings);
//		assertEquals(nodes[0], settings.getNode());
//		assertArrayEquals(new String[]{devices[0]}, settings.getDevices()
//				.stream()
//				.map(device -> device.getDeviceId())
//				.toArray(size -> new String[size]));
//		assertEquals(userJid, settings.getOwnerJid());
	}

	@Test
	public void test04_testDeviceRegistration2() throws Exception {
		IPushSettings settings = repository.registerDevice(serviceJid, userJid, "dummy", devices[1], null);
		assertNotNull(settings);
		assertNotNull(settings.getNode());
		nodes[1] = settings.getNode();
	}

	@Test
	public void test05_checkNode() throws Exception {
		IPushSettings settings = repository.getNodeSettings(serviceJid, nodes[1]);
		assertNotNull(settings);
		assertEquals(nodes[1], settings.getNode());
		assertEquals(userJid, settings.getOwnerJid());
		if (nodes[0].equals(nodes[1])) {
			assertArrayEquals(devices, settings.getDevices()
					.stream()
					.map(device -> device.getDeviceId())
					.sorted()
					.toArray(size -> new String[size]));
		} else {
			assertArrayEquals(new String[]{devices[1]}, settings.getDevices()
					.stream()
					.map(device -> device.getDeviceId())
					.sorted()
					.toArray(size -> new String[size]));
		}

//		settings = repository.getNodeSettings(userJid, "dummy", devices[1]);
//		assertNotNull(settings);
//		assertEquals(nodes[1], settings.getNode());
//		assertEquals(userJid, settings.getOwnerJid());
//		if (nodes[0].equals(nodes[1])) {
//			assertArrayEquals(devices, settings.getDevices()
//					.stream()
//					.map(device -> device.getDeviceId())
//					.sorted()
//					.toArray(size -> new String[size]));
//		} else {
//			assertArrayEquals(new String[]{devices[1]}, settings.getDevices()
//					.stream()
//					.map(device -> device.getDeviceId())
//					.sorted()
//					.toArray(size -> new String[size]));
//		}
	}

	@Test
	public void test06_testNodeForDifferentUser() throws Exception {
		IPushSettings settings = repository.getNodeSettings(serviceJid, nodes[0]);
		assertNotNull(settings);
		assertFalse(settings.isOwner(user2Jid));
	}

	@Test
	public void test07_testDeviceUnregisrationByDifferentUser() throws Exception {
		try {
			IPushSettings settings = repository.unregisterDevice(serviceJid, user2Jid, "dummy", devices[0]);
			assertFalse(true);
		} catch (ComponentException ex) {
			assertEquals(Authorization.ITEM_NOT_FOUND, ex.getErrorCondition());
		}

		IPushSettings settings = repository.getNodeSettings(serviceJid, nodes[1]);
		assertNotNull(settings);
		assertEquals(nodes[1], settings.getNode());
		assertEquals(userJid, settings.getOwnerJid());
		if (nodes[0].equals(nodes[1])) {
			assertArrayEquals(devices, settings.getDevices()
					.stream()
					.map(device -> device.getDeviceId())
					.sorted()
					.toArray(size -> new String[size]));
		} else {
			assertArrayEquals(new String[]{devices[1]}, settings.getDevices()
					.stream()
					.map(device -> device.getDeviceId())
					.sorted()
					.toArray(size -> new String[size]));
		}
	}

	@Test
	public void test08_testDeviceUnregiraton1() throws Exception {
		IPushSettings settings = repository.unregisterDevice(serviceJid, userJid, "dummy", devices[0]);
		assertNotNull(settings);
	}

	@Test
	public void test09_checkNode() throws Exception {
		IPushSettings settings = repository.getNodeSettings(serviceJid, nodes[1]);
		assertNotNull(settings);
		assertEquals(nodes[1], settings.getNode());
		assertArrayEquals(new String[]{devices[1]}, settings.getDevices()
				.stream()
				.map(device -> device.getDeviceId())
				.toArray(size -> new String[size]));
		assertEquals(userJid, settings.getOwnerJid());
	}

	@Test
	public void test10_testDeviceUnregiratonWihtoutServiceJidAndUserJid() throws Exception {
		IPushSettings settings = ((Stream<IPushSettings>) repository.getNodeSettings("dummy", devices[1])).findAny()
				.orElse(null);
		assertNotNull(settings);
		settings = repository.unregisterDevice(settings.getServiceJid(), settings.getOwnerJid(), "dummy", devices[1]);
		assertNotNull(settings);
	}

	@Test
	public void test11_ensureNodeIsRemoved() throws Exception {
		assertNull(repository.getNodeSettings(serviceJid, nodes[0]));
		assertNull(repository.getNodeSettings(serviceJid, nodes[1]));
	}

	@Override
	protected Class<? extends DataSourceAware> getDataSourceAwareIfc() {
		return IPushRepository.class;
	}
}
