/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.modules;

import org.junit.Test;
import tigase.component.PacketWriter;
import tigase.component.exceptions.ComponentException;
import tigase.component.modules.AbstractModule;
import tigase.component.responses.AsyncCallback;
import tigase.push.api.INotification;
import tigase.push.api.IPlainNotification;
import tigase.push.api.IPushProvider;
import tigase.push.api.IPushSettings;
import tigase.push.apns.APNsBinaryApiProvider;
import tigase.push.repositories.InMemoryPushRepository;
import tigase.server.Packet;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.DomBuilderHandler;
import tigase.xml.Element;
import tigase.xml.SingletonFactory;
import tigase.xmpp.Authorization;
import tigase.xmpp.StanzaType;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.CompletableFuture;

import static org.junit.Assert.*;
import static tigase.xmpp.impl.push.PushNotificationHelper.createPlainNotification;
import static tigase.xmpp.impl.push.PushNotificationHelper.createPushNotification;

/**
 * Created by andrzej on 03.01.2017.
 */
public class PublishNotificationModuleTest {

	private PublishNotificationModule module = new PublishNotificationModule();

	@Test
	public void test_getFeatures() {
		assertTrue(Arrays.asList(module.getFeatures()).contains("urn:xmpp:push:0"));
	}

	@Test
	public void test_criteria() throws Exception {
		Packet pushNotification = createPushNotification(JID.jidInstance("push.example.com"),
														 JID.jidInstance("user@example.com"), "some-node",
														 createPlainNotification(10, JID.jidInstance("user2@example.com"),
																			"Some body of a message"));

		assertTrue(module.getModuleCriteria().match(pushNotification.getElement()));
		assertFalse(module.getModuleCriteria().match(new Element("message")));
		assertFalse(module.getModuleCriteria().match(new Element("iq")));
		assertFalse(module.getModuleCriteria().match(new Element("presence")));
	}

	@Test
	public void test_process_success() throws Exception {
		JID serviceJid = JID.jidInstance("push.example.com");
		JID userJid = JID.jidInstance("user@example.com");
		String deviceId = UUID.randomUUID().toString();

		InMemoryPushRepository reposistory = new InMemoryPushRepository();
		IPushSettings pushSettings = reposistory.registerDevice(serviceJid.getBareJID(), userJid.getBareJID(), "dummy",
																deviceId, null);
		String node = pushSettings.getNode();

		Field f = PublishNotificationModule.class.getDeclaredField("repository");
		f.setAccessible(true);
		f.set(module, reposistory);

		final List<Packet> packetQueue = new ArrayList<>();
		f = AbstractModule.class.getDeclaredField("writer");
		f.setAccessible(true);
		f.set(module, new PacketWriter() {
			@Override
			public void write(Collection<Packet> packets) {
				packets.forEach(this::write);
			}

			@Override
			public void write(Packet packet) {
				packetQueue.add(packet);
			}

			@Override
			public void write(Packet packet, AsyncCallback callback) {
				write(packet);
			}
		});

		final List<String> deviceQueue = new ArrayList<>();
		final List<INotification> notificationQueue = new ArrayList<>();
		module.setPushProviders(Collections.singletonList(new IPushProvider() {

			@Override
			public String getName() {
				return "dummy";
			}

			@Override
			public String getDescription() {
				return "Dummy provider";
			}

			@Override
			public CompletableFuture<String> pushNotification(IPushSettings.IDevice device, INotification notification) {
				deviceQueue.add(device.getDeviceId());
				notificationQueue.add(notification);
				return CompletableFuture.completedFuture(UUID.randomUUID().toString());
			}
		}));

		Packet pushNotification = createPushNotification(serviceJid, userJid, node,
														 createPlainNotification(10, JID.jidInstance("user2@example.com"),
																			"Some body of a message"));

		module.process(pushNotification);

		assertEquals(1, deviceQueue.size());
		assertEquals(1, notificationQueue.size());
		assertEquals(1, packetQueue.size());

		assertEquals(deviceId, deviceQueue.get(0));
		IPlainNotification notification = (IPlainNotification) notificationQueue.get(0);
		assertEquals(10, notification.getMessageCount().longValue());
		assertEquals(JID.jidInstance("user2@example.com"), notification.getLastMessageSender());
		assertEquals("Some body of a message", notification.getLastMessageBody());
		assertEquals(StanzaType.result, packetQueue.get(0).getType());
		assertEquals("iq", packetQueue.get(0).getElemName());
		assertEquals(JID.jidInstance("example.com"), packetQueue.get(0).getStanzaTo());
		assertEquals(serviceJid, packetQueue.get(0).getStanzaFrom());
		assertEquals(INotification.Priority.high,notification.getPriority());
	}

	@Test
	public void test_process_success_low() throws Exception {
		JID serviceJid = JID.jidInstance("push.example.com");
		JID userJid = JID.jidInstance("user@example.com");
		String deviceId = UUID.randomUUID().toString();

		InMemoryPushRepository reposistory = new InMemoryPushRepository();
		IPushSettings pushSettings = reposistory.registerDevice(serviceJid.getBareJID(), userJid.getBareJID(), "dummy",
																deviceId, null);
		String node = pushSettings.getNode();

		Field f = PublishNotificationModule.class.getDeclaredField("repository");
		f.setAccessible(true);
		f.set(module, reposistory);

		final List<Packet> packetQueue = new ArrayList<>();
		f = AbstractModule.class.getDeclaredField("writer");
		f.setAccessible(true);
		f.set(module, new PacketWriter() {
			@Override
			public void write(Collection<Packet> packets) {
				packets.forEach(this::write);
			}

			@Override
			public void write(Packet packet) {
				packetQueue.add(packet);
			}

			@Override
			public void write(Packet packet, AsyncCallback callback) {
				write(packet);
			}
		});

		final List<String> deviceQueue = new ArrayList<>();
		final List<INotification> notificationQueue = new ArrayList<>();
		module.setPushProviders(Collections.singletonList(new IPushProvider() {

			@Override
			public String getName() {
				return "dummy";
			}

			@Override
			public String getDescription() {
				return "Dummy provider";
			}

			@Override
			public CompletableFuture<String> pushNotification(IPushSettings.IDevice device, INotification notification) {
				deviceQueue.add(device.getDeviceId());
				notificationQueue.add(notification);
				return CompletableFuture.completedFuture(UUID.randomUUID().toString());
			}
		}));

		Element notificationEl = createPlainNotification(10, JID.jidInstance("user2@example.com"),
														 "Some body of a message");
		Element priorityEl = new Element("priority", "low");
		priorityEl.setXMLNS("tigase:push:priority:0");
		notificationEl.addChild(priorityEl);
		Packet pushNotification = createPushNotification(serviceJid, userJid, node,
														 notificationEl);

		module.process(pushNotification);

		assertEquals(1, deviceQueue.size());
		assertEquals(1, notificationQueue.size());
		assertEquals(1, packetQueue.size());

		assertEquals(deviceId, deviceQueue.get(0));
		IPlainNotification notification = (IPlainNotification) notificationQueue.get(0);
		assertEquals(10, notification.getMessageCount().longValue());
		assertEquals(JID.jidInstance("user2@example.com"), notification.getLastMessageSender());
		assertEquals("Some body of a message", notification.getLastMessageBody());
		assertEquals(StanzaType.result, packetQueue.get(0).getType());
		assertEquals("iq", packetQueue.get(0).getElemName());
		assertEquals(JID.jidInstance("example.com"), packetQueue.get(0).getStanzaTo());
		assertEquals(serviceJid, packetQueue.get(0).getStanzaFrom());
		assertEquals(INotification.Priority.low,notification.getPriority());
	}
	 
	@Test
	public void test_process_failures() throws Exception {
		JID serviceJid = JID.jidInstance("push.example.com");
		JID userJid = JID.jidInstance("user@example.com");
		String deviceId = UUID.randomUUID().toString();

		InMemoryPushRepository reposistory = new InMemoryPushRepository();
		IPushSettings pushSettings = reposistory.registerDevice(serviceJid.getBareJID(), userJid.getBareJID(), "dummy",
																deviceId, null);
		String node = pushSettings.getNode();

		Field f = PublishNotificationModule.class.getDeclaredField("repository");
		f.setAccessible(true);
		f.set(module, reposistory);

		final List<Packet> packetQueue = new ArrayList<>();
		f = AbstractModule.class.getDeclaredField("writer");
		f.setAccessible(true);
		f.set(module, new PacketWriter() {
			@Override
			public void write(Collection<Packet> packets) {
				packets.forEach(this::write);
			}

			@Override
			public void write(Packet packet) {
				packetQueue.add(packet);
			}

			@Override
			public void write(Packet packet, AsyncCallback callback) {
				write(packet);
			}
		});

		final List<String> deviceQueue = new ArrayList<>();
		final List<INotification> notificationQueue = new ArrayList<>();
		module.setPushProviders(Collections.singletonList(new IPushProvider() {

			@Override
			public String getName() {
				return "dummy";
			}

			@Override
			public String getDescription() {
				return "Dummy provider";
			}

			@Override
			public CompletableFuture<String> pushNotification(IPushSettings.IDevice device, INotification notification) {
				deviceQueue.add(device.getDeviceId());
				notificationQueue.add(notification);
				return CompletableFuture.completedFuture(UUID.randomUUID().toString());
			}
		}));

		Packet pushNotification = createPushNotification(serviceJid, userJid, node + "%%",
														 createPlainNotification(10, JID.jidInstance("user2@example.com"),
																			"Some body of a message"));

		try {
			module.process(pushNotification);
			assertFalse(true);
		} catch (ComponentException ex) {
			assertEquals(Authorization.ITEM_NOT_FOUND, ex.getErrorCondition());
		}
		
	}

	// ejabberd LOW
	private static final String TEST_PAYLOAD_NO_X = "<notification xmlns='urn:xmpp:push:0'></notification>";
	// ejabberd HIGH (without message)
	private static final String TEST_PAYLOAD_X = "<notification xmlns='urn:xmpp:push:0'><x type='submit' xmlns='jabber:x:data'><field var='FORM_TYPE' type='hidden'><value>urn:xmpp:push:summary</value></field></x></notification>";
	// ejabberd HIGH (with message)
	private static final String TEST_PAYLOAD_X_MSG = "<notification xmlns='urn:xmpp:push:0'><x type='submit' xmlns='jabber:x:data'><field var='FORM_TYPE' type='hidden'><value>urn:xmpp:push:summary</value></field><field var='last-message-body' type='text-single' label='The body text of the last received message'><value>New message</value></field></x></notification>";
	// Prosody LOW
	private static final String TEST_PAYLOAD_X_COUNT_NO_MSG = "<notification xmlns='urn:xmpp:push:0'><x type='submit' xmlns='jabber:x:data'><field var='FORM_TYPE' type='hidden'><value>urn:xmpp:push:summary</value></field><field var='message-count'><value>1</value></field></x></notification>";
	// Prosody HIGH (always has message or static message)
	private static final String TEST_PAYLOAD_X_COUNT_AND_MSG = "<notification xmlns='urn:xmpp:push:0'><x type='submit' xmlns='jabber:x:data'><field var='FORM_TYPE' type='hidden'><value>urn:xmpp:push:summary</value></field><field var='message-count'><value>1</value></field><field var='last-message-body' type='text-single' label='The body text of the last received message'><value>New message</value></field></x></notification>";

	@Test
	public void testPayloadParsing() throws TigaseStringprepException, ComponentException {
		// ejabberd LOW
		INotification notification = parsePayload(TEST_PAYLOAD_NO_X);
		assertNotNull(notification);
		assertEquals(INotification.Priority.low, notification.getPriority());

		// ejabberd HIGH (without message)
		notification = parsePayload(TEST_PAYLOAD_X);
		assertNotNull(notification);
		assertEquals(INotification.Priority.high, notification.getPriority());

		// ejabberd HIGH (with message)
		notification = parsePayload(TEST_PAYLOAD_X_MSG);
		assertNotNull(notification);
		assertEquals(INotification.Priority.high, notification.getPriority());

		// Prosody LOW
		notification = parsePayload(TEST_PAYLOAD_X_COUNT_NO_MSG);
		assertNotNull(notification);
		assertEquals(INotification.Priority.low, notification.getPriority());

		// Prosody HIGH (always has message or static message)
		notification = parsePayload(TEST_PAYLOAD_X_COUNT_AND_MSG);
		assertNotNull(notification);
		assertEquals(INotification.Priority.high, notification.getPriority());
	}

	private INotification parsePayload(String payloadString) throws TigaseStringprepException, ComponentException {
		DomBuilderHandler handler = new DomBuilderHandler();
		APNsBinaryApiProvider provider = new APNsBinaryApiProvider();
		SingletonFactory.getParserInstance().parse(handler, payloadString);
		return module.parseNotification(BareJID.bareJIDInstanceNS("example.net"), handler.getParsedElements().poll());
	}

}
