/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.adhoc;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import tigase.component.adhoc.AdHocCommand;
import tigase.component.adhoc.AdHocCommandException;
import tigase.component.adhoc.AdHocCommandManager;
import tigase.push.api.INotification;
import tigase.push.api.IPushProvider;
import tigase.push.api.IPushSettings;
import tigase.push.repositories.InMemoryPushRepository;
import tigase.server.Command;
import tigase.server.Packet;
import tigase.xml.Element;
import tigase.xmpp.StanzaType;
import tigase.xmpp.jid.JID;

import java.lang.reflect.Field;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import static org.junit.Assert.*;

/**
 * Created by andrzej on 04.01.2017.
 */
public class RegisterDeviceTest {

	private RegisterDevice adhoc;
	private AdHocCommandManager manager;
	private DummyPushProvider pushProvider;
	private JID pushServiceJid = JID.jidInstanceNS("push.example.com");
	private InMemoryPushRepository repository;
	private JID userJid;

	@Before
	public void setUp() throws Exception {
		manager = new AdHocCommandManager();
		adhoc = new RegisterDevice();
		pushProvider = new DummyPushProvider();
		repository = new InMemoryPushRepository();

		Field f = RegisterDevice.class.getDeclaredField("pushProviders");
		f.setAccessible(true);
		f.set(adhoc, Arrays.asList(pushProvider));
		f = RegisterDevice.class.getDeclaredField("repository");
		f.setAccessible(true);
		f.set(adhoc, repository);
		manager.setAllCommands(new AdHocCommand[]{adhoc});

		userJid = JID.jidInstance("user-" + UUID.randomUUID() + "@example.com/res-1");
	}

	@After
	public void tearDown() {
		adhoc = null;
		manager = null;
	}

	@Test
	public void test_formRetrieval() throws Exception {
		Packet packet = Packet.packetInstance(new Element("iq", new Element[]{
				new Element("command", new String[]{"xmlns", "action", "node"},
							new String[]{"http://jabber.org/protocol/commands", "execute", "register-device"})},
														  new String[]{"type"}, new String[]{"set"}));
		Packet result = process(packet);
		assertEquals(StanzaType.result, result.getType());
	}

	@Test
	public void test_formSubmit() throws Exception {
		String deviceToken = UUID.randomUUID().toString();
		Packet packet = Packet.packetInstance(new Element("iq", new Element[]{new Element("command", new Element[]{
													  new Element("x", new Element[]{
															  new Element("field", new Element[]{new Element("value", pushProvider.getName())},
																		  new String[]{"var"}, new String[]{"provider"}),
															  new Element("field", new Element[]{new Element("value", deviceToken)}, new String[]{"var"},
																		  new String[]{"device-token"})}, new String[]{"xmlns", "type"},
																  new String[]{"jabber:x:data", "submit"})}, new String[]{"xmlns", "action", "node"},
																						  new String[]{
																								  "http://jabber.org/protocol/commands",
																								  "execute",
																								  "register-device"})},
														  new String[]{"type"}, new String[]{"set"}), userJid,
											  pushServiceJid);
		
		Packet result = process(packet);
		assertEquals(StanzaType.result, result.getType());

		String node = Command.getFieldValue(result, "node");
		assertNotNull(node);

		IPushSettings settings = repository.getNodeSettings(pushServiceJid.getBareJID(), node);
		assertNotNull(settings);
		assertTrue(settings.isOwner(userJid.getBareJID()));
		assertEquals(1, settings.getDevices().size());
		assertEquals(deviceToken, settings.getDevices().get(0).getDeviceId());
	}

	@Test
	public void test_reregistration() throws Exception {
		String deviceToken = UUID.randomUUID().toString();
		Packet packet = Packet.packetInstance(new Element("iq", new Element[]{new Element("command", new Element[]{
													  new Element("x", new Element[]{
															  new Element("field", new Element[]{new Element("value", pushProvider.getName())},
																		  new String[]{"var"}, new String[]{"provider"}),
															  new Element("field", new Element[]{new Element("value", deviceToken)}, new String[]{"var"},
																		  new String[]{"device-token"})}, new String[]{"xmlns", "type"},
																  new String[]{"jabber:x:data", "submit"})}, new String[]{"xmlns", "action", "node"},
																						  new String[]{
																								  "http://jabber.org/protocol/commands",
																								  "execute",
																								  "register-device"})},
														  new String[]{"type"}, new String[]{"set"}), userJid,
											  pushServiceJid);
		Packet result = process(packet);
		assertEquals(StanzaType.result, result.getType());

		String node = Command.getFieldValue(result, "node");
		assertNotNull(node);

		IPushSettings settings = repository.getNodeSettings(pushServiceJid.getBareJID(), node);
		assertNotNull(settings);
		assertTrue(settings.isOwner(userJid.getBareJID()));
		assertEquals(1, settings.getDevices().size());
		assertEquals(deviceToken, settings.getDevices().get(0).getDeviceId());
		
		result = process(packet);
		assertEquals(StanzaType.result, result.getType());

		node = Command.getFieldValue(result, "node");
		assertNotNull(node);

		settings = repository.getNodeSettings(pushServiceJid.getBareJID(), node);
		assertNotNull(settings);
		assertTrue(settings.isOwner(userJid.getBareJID()));
		assertEquals(1, settings.getDevices().size());
		assertEquals(deviceToken, settings.getDevices().get(0).getDeviceId());
	}

	protected Packet process(Packet packet) throws AdHocCommandException {
		Queue<Packet> queue = new ArrayDeque<>();
		manager.process(packet, queue::offer);
		return queue.poll();
	}

	public static class DummyPushProvider
			implements IPushProvider {

		@Override
		public String getName() {
			return "dummy";
		}

		@Override
		public String getDescription() {
			return "Dummy Push Provider";
		}

		@Override
		public CompletableFuture<String> pushNotification(IPushSettings.IDevice device, INotification notification) {
			return CompletableFuture.completedFuture(UUID.randomUUID().toString());
		}
	}

}
