/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.repositories;

import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.db.DBInitException;
import tigase.db.DataSource;
import tigase.db.DataSourceHelper;
import tigase.db.beans.MDRepositoryBean;
import tigase.db.beans.MDRepositoryBeanWithStatistics;
import tigase.kernel.beans.Bean;
import tigase.push.PushNotificationsComponent;
import tigase.push.api.IPushRepository;
import tigase.push.api.IPushSettings;
import tigase.xmpp.jid.BareJID;

import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;

/**
 * Created by andrzej on 05.01.2017.
 */
@Bean(name = "pushRepository", parent = PushNotificationsComponent.class, active = true)
public class PushRepositoryMDBean
		extends MDRepositoryBeanWithStatistics<IPushRepository>
		implements IPushRepository {

	public PushRepositoryMDBean() {
		super(IPushRepository.class);
	}

	@Override
	public Class<?> getDefaultBeanClass() {
		return PushRepositoryConfigBean.class;
	}

	@Override
	public IPushSettings registerDevice(BareJID serviceJid, BareJID userJid, String provider, String deviceId, String deviceSecondId)
			throws RepositoryException {
		return getRepository(serviceJid.getDomain()).registerDevice(serviceJid, userJid, provider, deviceId, deviceSecondId);
	}

	@Override
	public IPushSettings unregisterDevice(BareJID serviceJid, BareJID userJid, String provider, String deviceId)
			throws RepositoryException, ComponentException {
		return getRepository(serviceJid.getDomain()).unregisterDevice(serviceJid, userJid, provider, deviceId);
	}

	@Override
	public IPushSettings getNodeSettings(BareJID serviceJid, String node) throws RepositoryException {
		return getRepository(serviceJid.getDomain()).getNodeSettings(serviceJid, node);
	}

	@Override
	public Stream<IPushSettings> getNodeSettings(BareJID userJid, String provider, String deviceId) throws RepositoryException {
		return getNodeSettings(provider, deviceId).filter(it -> it.getOwnerJid().equals(userJid));
	}

	@Override
	public Stream<IPushSettings> getNodeSettings(String provider, String deviceId) throws RepositoryException {
		return getRepositories().values().stream().flatMap((IPushRepository repo) -> {
			Stream<IPushSettings> result;
			try {
				result = repo.getNodeSettings(provider, deviceId);
			} catch (RepositoryException ex) {
				result = Stream.empty();
			}
			return result;
		});
	}

	@Override
	public Map<String, Statistics> getStatistics() throws RepositoryException {
		Map<String, Statistics> result = new HashMap<>();
		for (IPushRepository repo : getRepositories().values()) {
			result.putAll(repo.getStatistics());
		}
		return result;
	}

	@Override
	public void setDataSource(DataSource dataSource) {
		// nothing to do in MD bean
	}

	@Override
	protected Class<? extends IPushRepository> findClassForDataSource(DataSource dataSource) throws DBInitException {
		return DataSourceHelper.getDefaultClass(IPushRepository.class, dataSource.getResourceUri());
	}

	public static class PushRepositoryConfigBean
			extends MDRepositoryBean.MDRepositoryConfigBean<IPushRepository> {

		@Override
		protected Class<?> getRepositoryClassName() throws DBInitException, ClassNotFoundException {
			String cls = getCls();
			if (cls == null) {
				cls = "default";
			}
			switch (cls) {
				case "memory":
					return InMemoryPushRepository.class;
				default:
					return super.getRepositoryClassName();
			}
		}
	}
}
