/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.repositories;

import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.db.DataRepository;
import tigase.db.Repository;
import tigase.db.util.RepositoryVersionAware;
import tigase.kernel.beans.config.ConfigField;
import tigase.push.Device;
import tigase.push.PushSettings;
import tigase.push.api.IPushSettings;
import tigase.xmpp.Authorization;
import tigase.xmpp.jid.BareJID;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;
import java.util.logging.Logger;
import java.util.stream.Stream;

/**
 * Created by andrzej on 05.01.2017.
 */
@Repository.Meta(supportedUris = {"jdbc:.*"})
@Repository.SchemaId(id = Schema.PUSH_SCHEMA_ID, name = Schema.PUSH_SCHEMA_NAME)
public class JDBCPushRepository
		extends AbstractPushRepository<DataRepository>
		implements RepositoryVersionAware {

	private static final Logger log = Logger.getLogger(JDBCPushRepository.class.getCanonicalName());
	protected DataRepository repo;
	@ConfigField(desc = "Query to retrieve settings for a node with provider and device id", alias = "get-node-settings-by-device-query")
	private String getNodeSettingsByDeviceQuery = "{ call Tig_Push_GetNodeSettingsByDevice(?,?) }";
	@ConfigField(desc = "Query to retrieve settings for a node with service jid and node", alias = "get-node-settings-by-node-query")
	private String getNodeSettingsByNodeQuery = "{ call Tig_Push_GetNodeSettingsByNode(?,?) }";
	@ConfigField(desc = "Query to register device", alias = "register-device-query")
	private String registerDeviceQuery = "{ call Tig_Push_RegisterDevice(?,?,?,?,?,?) }";
	@ConfigField(desc = "Query to unregister device", alias = "unregister-device-query")
	private String unregisterDeviceQuery = "{ call Tig_Push_UnregisterDevice(?,?,?,?,?) }";
	@ConfigField(desc = "Query to retrieve statistics", alias = "stats-query")
	private String statsQuery = "{ call Tig_Push_Stats() }";

	@Override
	public IPushSettings registerDevice(BareJID serviceJid, BareJID userJid, String provider, String deviceId, String deviceSecondId)
			throws RepositoryException {
		String node = calculateNode(serviceJid, userJid, deviceId);
		try {
			PreparedStatement ps = repo.getPreparedStatement(userJid, registerDeviceQuery);
			synchronized (ps) {
				ps.setString(1, serviceJid.toString());
				ps.setString(2, userJid.toString());
				ps.setString(3, node);
				ps.setString(4, provider);
				ps.setString(5, deviceId);
				ps.setString(6, deviceSecondId);

				ps.executeUpdate();
			}
		} catch (SQLException ex) {
			throw new RepositoryException("Could not register device", ex);
		}
		return getNodeSettings(serviceJid, node);
	}

	@Override
	public IPushSettings unregisterDevice(BareJID serviceJid, BareJID userJid, String provider, String deviceId)
			throws RepositoryException, ComponentException {
		String node = calculateNode(serviceJid, userJid, deviceId);
		IPushSettings pushSettings = getNodeSettings(serviceJid, node);
		if (pushSettings == null) {
			return unregisterDeviceOld(serviceJid, userJid,provider, deviceId);
		}
		try {
			PreparedStatement ps = repo.getPreparedStatement(userJid, unregisterDeviceQuery);
			synchronized (ps) {
				ps.setString(1, serviceJid.toString());
				ps.setString(2, userJid.toString());
				ps.setString(3, node);
				ps.setString(4, provider);
				ps.setString(5, deviceId);

				ps.executeUpdate();
			}
		} catch (SQLException ex) {
			throw new RepositoryException("Could not unregister device", ex);
		}
		return pushSettings;
	}
	
	public IPushSettings unregisterDeviceOld(BareJID serviceJid, BareJID userJid, String provider, String deviceId)
			throws RepositoryException, ComponentException {
		String node = calculateNode(serviceJid, userJid);
		IPushSettings pushSettings = getNodeSettings(serviceJid, node);
		if (pushSettings != null) {
			IPushSettings.IDevice device = new Device(provider, deviceId, null);
			pushSettings = pushSettings.removeDevice(device);
		}
		if (pushSettings == null) {
			throw new ComponentException(Authorization.ITEM_NOT_FOUND, "Device is not registered");
		}
		try {
			PreparedStatement ps = repo.getPreparedStatement(userJid, unregisterDeviceQuery);
			synchronized (ps) {
				ps.setString(1, serviceJid.toString());
				ps.setString(2, userJid.toString());
				ps.setString(3, node);
				ps.setString(4, provider);
				ps.setString(5, deviceId);

				ps.executeUpdate();
			}
		} catch (SQLException ex) {
			throw new RepositoryException("Could not unregister device", ex);
		}
		return pushSettings;
	}

	@Override
	public IPushSettings getNodeSettings(BareJID serviceJid, String node) throws RepositoryException {
		try {
			PreparedStatement ps = repo.getPreparedStatement(node.hashCode(), getNodeSettingsByNodeQuery);
			synchronized (ps) {
				ResultSet rs = null;
				try {
					ps.setString(1, serviceJid.toString());
					ps.setString(2, node);

					rs = ps.executeQuery();

					if (!rs.next()) {
						return null;
					}

					if (node.charAt(2) == '#') {
						int version = Integer.parseInt(node.substring(0, 2));
						BareJID ownerJid = BareJID.bareJIDInstanceNS(rs.getString(1));
						PushSettings.IDevice device = readDevice(rs, 2);
						return new PushSettings(version, serviceJid, node, ownerJid, List.of(device));
					} else {
						int i = 1;
						BareJID ownerJid = BareJID.bareJIDInstanceNS(rs.getString(i++));
						List<IPushSettings.IDevice> devices = new ArrayList<>();
						do {
							devices.add(readDevice(rs, i));
						} while (rs.next());

						return new PushSettings(0, serviceJid, node, ownerJid, devices);
					}
				} finally {
					repo.release(null, rs);
				}
			}
		} catch (SQLException ex) {
			throw new RepositoryException("Could not retrieve setting by service jid and node", ex);
		}
	}

	@Override
	public Stream<IPushSettings> getNodeSettings(String provider, String deviceId) throws RepositoryException {
		try {
			PreparedStatement ps = repo.getPreparedStatement(deviceId.hashCode(), getNodeSettingsByDeviceQuery);
			synchronized (ps) {
				ResultSet rs = null;
				try {
					ps.setString(1, provider);
					ps.setString(2, deviceId);

					rs = ps.executeQuery();

					if (!rs.next()) {
						return null;
					}

					List<IPushSettings> settings = new ArrayList<>();

					do {
						BareJID serviceJid = BareJID.bareJIDInstanceNS(rs.getString(1));
						BareJID ownerJid = BareJID.bareJIDInstanceNS(rs.getString(2));
						String node = rs.getString(3);

						if (node.charAt(2) == '#') {
							int version = Integer.parseInt(node.substring(0, 2));
							IPushSettings.IDevice device = readDevice(rs, 4);
							if (settings.stream()
									.noneMatch(it -> it.getServiceJid().equals(serviceJid) &&
											it.getOwnerJid().equals(ownerJid) && it.getNode().equals(node))) {
								settings.add(new PushSettings(version, serviceJid, node, ownerJid, List.of(device)));
							}
						} else {
							IPushSettings entry = settings.stream()
									.filter(it -> it.getServiceJid().equals(serviceJid) &&
											it.getOwnerJid().equals(ownerJid) && it.getNode().equals(node))
									.findFirst()
									.orElseGet(() -> new PushSettings(0, serviceJid, node, ownerJid, Collections.emptyList()));

							settings.remove(entry);
							settings.add(entry.addDevice(readDevice(rs, 4)));
						}
					} while (rs.next());

					return settings.stream();
				} finally {
					repo.release(null, rs);
				}
			}
		} catch (SQLException ex) {
			throw new RepositoryException("Could not retrieve setting by provider and device id", ex);
		}
	}

	@Override
	public Map<String, Statistics> getStatistics() throws RepositoryException {
		try {
			Map<String, Statistics> results = new HashMap<>();
			PreparedStatement stmt = repo.getPreparedStatement(0, statsQuery);
			synchronized (stmt) {
				ResultSet rs = stmt.executeQuery();
				try {
					while (rs.next()) {
						String provider = rs.getString(1);
						StatisticsImpl stats = (StatisticsImpl) results.computeIfAbsent(provider, (k) -> new StatisticsImpl(k));
						stats.setCounterValue(rs.getString(2), rs.getInt(3));
					}
				} finally {
					repo.release(null, rs);
				}
			}
			return results;
		} catch (SQLException ex) {
			throw new RepositoryException("Could not retrieve statistics from repository", ex);
		}
	}

	@Override
	public void setDataSource(DataRepository dataSource) {
		try {
			initRepo(dataSource);
			repo = dataSource;
		} catch (SQLException ex) {
			throw new RuntimeException("Failed to initialize access to SQL database for JDBCPushRepository", ex);
		}
	}

	protected IPushSettings.IDevice readDevice(ResultSet rs, int i) throws SQLException {
		String provider = rs.getString(i++);
		String deviceId = rs.getString(i++);
		String deviceSecondId = rs.getString(i++);

		return new Device(provider, deviceId, deviceSecondId);
	}

	protected void initRepo(DataRepository repo) throws SQLException {
		repo.initPreparedStatement(registerDeviceQuery, registerDeviceQuery);
		repo.initPreparedStatement(unregisterDeviceQuery, unregisterDeviceQuery);
		repo.initPreparedStatement(getNodeSettingsByNodeQuery, getNodeSettingsByNodeQuery);
		repo.initPreparedStatement(getNodeSettingsByDeviceQuery, getNodeSettingsByDeviceQuery);
		repo.initPreparedStatement(statsQuery, statsQuery);
	}
}
