/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.repositories;

import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.db.DataSource;
import tigase.db.util.SchemaManager;
import tigase.push.Device;
import tigase.push.PushSettings;
import tigase.push.api.IPushSettings;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xmpp.Authorization;
import tigase.xmpp.jid.BareJID;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Created by andrzej on 02.01.2017.
 */
public class InMemoryPushRepository
		extends AbstractPushRepository {

	private final ConcurrentHashMap<Key, IPushSettings> cache = new ConcurrentHashMap<>();

	@Override
	public IPushSettings registerDevice(BareJID serviceJid, BareJID userJid, String provider, String deviceId, String deviceSecondId)
			throws RepositoryException {
		String node = calculateNode(serviceJid, userJid, deviceId);
		Key key = new Key(serviceJid, node);
		IPushSettings.IDevice device = new Device(provider, deviceId, deviceSecondId);

		return cache.compute(key, (k,v) -> new PushSettings(1, serviceJid, node, userJid, List.of(device)));
	}

	@Override
	public IPushSettings unregisterDevice(BareJID serviceJid, BareJID userJid, String provider, String deviceId)
			throws RepositoryException, ComponentException {
		String node = calculateNode(serviceJid, userJid, deviceId);
		Key key = new Key(serviceJid, node);

		IPushSettings settings = cache.remove(key);
		if (settings == null) {
			return unregisterDeviceOld(serviceJid, userJid, provider, deviceId);
		}
		return settings;
	}

	public IPushSettings unregisterDeviceOld(BareJID serviceJid, BareJID userJid, String provider, String deviceId)
			throws RepositoryException, ComponentException {
		IPushSettings.IDevice device = new Device(provider, deviceId, null);
		String node = calculateNode(serviceJid, userJid);
		Key key = new Key(serviceJid, node);

		IPushSettings settings = cache.computeIfPresent(key, (key1, settings1) -> settings1.removeDevice(device));
		if (settings == null) {
			throw new ComponentException(Authorization.ITEM_NOT_FOUND, "Device is not registered");
		}
		if (settings.getDevices().isEmpty()) {
			cache.remove(key, settings);
		}
		return settings;
	}

	@Override
	public IPushSettings getNodeSettings(BareJID serviceJid, String node) {
		return cache.get(new Key(serviceJid, node));
	}

	@Override
	public Stream<IPushSettings> getNodeSettings(String provider, String deviceId) throws RepositoryException {
		IPushSettings.IDevice device = new Device(provider, deviceId, null);
		return cache.values().stream().filter(s -> s.getDevices().contains(device));
	}

	@Override
	public Map<String, Statistics> getStatistics() throws RepositoryException {
		Map<String, Statistics> results = new HashMap<>();
		List<String> providers = cache.values()
				.stream()
				.flatMap(s -> s.getDevices().stream())
				.map(IPushSettings.IDevice::getProviderName)
				.distinct()
				.collect(Collectors.toList());
		for (String provider : providers) {
			StatisticsImpl stats = (StatisticsImpl) results.computeIfAbsent(provider, (k) -> new StatisticsImpl(k));
			stats.setCounterValue("all", (int) cache.values()
					.stream()
					.flatMap(s -> s.getDevices().stream())
					.map(IPushSettings.IDevice::getProviderName)
					.filter(provider::equals)
					.count());
			stats.setCounterValue("accounts", (int) cache.values()
					.stream()
					.flatMap(s -> s.getDevices()
							.stream()
							.map(device -> new SchemaManager.Pair(device.getProviderName(), s.getOwnerJid())))
					.filter(pair -> provider.equals(pair.getKey()))
					.map(SchemaManager.Pair::getValue)
					.distinct()
					.count());
		}
		return results;
	}

	@Override
	public void setDataSource(DataSource dataSource) {
		// nothing to do here
	}

	public static class Key {

		private final String node;
		private final BareJID serviceJid;

		// as node is hex representation of hash of userjid and servicejid, do we still need serviceJid in hex?
		public Key(BareJID serviceJid, String node) {
			this.serviceJid = serviceJid;
			this.node = node;
		}

		@Override
		public int hashCode() {
			return serviceJid.hashCode() * 31 + node.hashCode();
		}

		@Override
		public boolean equals(Object obj) {
			if (!(obj instanceof Key)) {
				return false;
			}
			return serviceJid.equals(((Key) obj).serviceJid) && node.equals(((Key) obj).node);
		}

		@Override
		public String toString() {
			return "NodeKey[serviceJid = " + serviceJid.toString() + ", node = " + node + "]";
		}

	}

	public static void main(String[] args) throws TigaseStringprepException, RepositoryException {
		InMemoryPushRepository repo = new InMemoryPushRepository();
		System.out.println(repo.calculateNode(BareJID.bareJIDInstance("push.hi-low.eu"), BareJID.bareJIDInstance("andrzej@hi-low.eu"), "97C5A1222F29F247E800F5B57761E85B29974EAFF72EB0954B5436AC98E51E67"));
	}
}
