/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.repositories;

import tigase.component.exceptions.RepositoryException;
import tigase.db.DataSource;
import tigase.push.api.IPushRepository;
import tigase.push.api.IPushSettings;
import tigase.xmpp.jid.BareJID;

import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * Created by andrzej on 05.01.2017.
 */
public abstract class AbstractPushRepository<DS extends DataSource>
		implements IPushRepository<DS> {

	private static final Charset UTF8 = Charset.forName("UTF-8");
	private static final char[] hexArray = "0123456789ABCDEF".toCharArray();

	@Override
	public Stream<IPushSettings> getNodeSettings(BareJID userJid, String provider, String deviceId) throws RepositoryException {
		return getNodeSettings(provider, deviceId).filter(it -> it.getOwnerJid().equals(userJid));
	}

	protected String calculateNode(BareJID serviceJid, BareJID userJid) throws RepositoryException {
		try {
			MessageDigest md = MessageDigest.getInstance("SHA-256");
			md.update(userJid.toString().toLowerCase().getBytes(UTF8));
			md.update((byte) 0x00);
			byte[] data = md.digest(serviceJid.toString().toLowerCase().getBytes(UTF8));
			return toHex(data, "");
		} catch (NoSuchAlgorithmException ex) {
			throw new RepositoryException(ex);
		}
	}

	protected String calculateNode(BareJID serviceJid, BareJID userJid, String deviceId) throws RepositoryException {
		try {
			MessageDigest md = MessageDigest.getInstance("SHA-256");
			md.update(userJid.toString().toLowerCase().getBytes(UTF8));
			md.update(deviceId.toUpperCase().getBytes(UTF8));
			md.update((byte) 0x00);
			byte[] data = md.digest(serviceJid.toString().toLowerCase().getBytes(UTF8));
			return toHex(data, "01#");
		} catch (NoSuchAlgorithmException ex) {
			throw new RepositoryException(ex);
		}
	}

	protected String toHex(byte[] data, String prefix) {
		StringBuilder sb = new StringBuilder((data.length * 2) + prefix.length());
		sb.append(prefix);
		for (int i = 0; i < data.length; i++) {
			int v = data[i] & 0xFF;
			sb.append(hexArray[v >>> 4]);
			sb.append(hexArray[v & 0x0F]);
		}
		return sb.toString();
	}

	protected static class StatisticsImpl implements Statistics {
		private final String provider;
		private final Map<String,Integer> counters = new HashMap<>();

		protected StatisticsImpl(String provider) {
			this.provider = provider;
		}

		public Map<String,Integer> getCounters() {
			return counters;
		}

		public Collection<String> getCounterNames() {
			return counters.keySet();
		}

		public int getCounterValue(String name) {
			return Optional.ofNullable(counters.get(name)).orElse(0);
		}

		protected void setCounterValue(String name, int value) {
			counters.put(name, value);
		}
	}
}
