/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.monitor;

import tigase.eventbus.EventBus;
import tigase.form.Field;
import tigase.form.Form;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.monitor.MonitorComponent;
import tigase.monitor.tasks.AbstractConfigurableTimerTask;
import tigase.monitor.tasks.TasksEvent;
import tigase.util.datetime.TimestampHelper;
import tigase.xml.XMLUtils;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

@Bean(name = "push-ssl-certificate-expiration-checker-task", parent = MonitorComponent.class, active = true)
public class MonitorSSLCertificatesExpirationCheckerTask
		extends AbstractConfigurableTimerTask {

	private static final Logger log = Logger.getLogger(MonitorSSLCertificatesExpirationCheckerTask.class.getCanonicalName());

	private static final String EVENT_NAME = "tigase.monitor.tasks.push.MonitorSSLCertificatesExpirationTask";
	private static final String STILL_VALID_IN_VAR = "still-valid-in";
	private static final TimestampHelper dtf = new TimestampHelper();

	@Inject
	private EventBus eventBus;
	@Inject
	private MonitorComponent component;
	@Inject(nullAllowed = true)
	private List<SSLCertificateExpirationAware> containers = Collections.emptyList();

	private int stillValidInDays = 7;

	private ConcurrentHashMap<String,Long> triggeredEvents = new ConcurrentHashMap<>();

	public MonitorSSLCertificatesExpirationCheckerTask() {
		setPeriod(1000 * 60 * 60);
	}

	public List<SSLCertificateExpirationAware> getContainers() {
		return containers;
	}

	public void setContainers(List<SSLCertificateExpirationAware> containers) {
		this.containers = Optional.ofNullable(containers).orElse(Collections.emptyList());
	}

	@Override
	public Form getCurrentConfiguration() {
		Form f = super.getCurrentConfiguration();

		f.addField(Field.fieldTextSingle(STILL_VALID_IN_VAR, "" + stillValidInDays, "Check if certificates are still valid in [days]"));

		return f;
	}

	@Override
	public void setNewConfiguration(Form form) {
		Field f = form.get(STILL_VALID_IN_VAR);
		if (f != null) {
			stillValidInDays = Integer.parseInt(f.getValue());
		}

		super.setNewConfiguration(form);
	}

	@Override
	protected void run() {
		checkPushCertificatesValidation();
	}

	protected void checkPushCertificatesValidation() {
		if (log.isLoggable(Level.FINE)) {
			log.log(Level.FINE, "Checking SSL certificates for " + containers);
		}
		List<SSLCertificateExpirationAware.Result> results = containers.stream()
				.flatMap(SSLCertificateExpirationAware::getSSLCertificatesValidPeriod)
				.collect(Collectors.toList());

		for (SSLCertificateExpirationAware.Result result : results) {
			if (result.isValid(LocalDateTime.now()) && result.isValid(LocalDateTime.now().plusDays(stillValidInDays))) {
				triggeredEvents.remove(result.getName());
			} else {
				final APNSCertificatesExpirationEvent event = new APNSCertificatesExpirationEvent(result);
				long daysToExpire = LocalDateTime.now().until(result.getValidTo(), ChronoUnit.DAYS);

				Consumer<Level> logResult = level -> {
					if (!result.isValid(LocalDateTime.now())) {
						log.log(level, () -> "Certificate " + result.getName() + " is not valid!");
					} else {
						log.log(level, () -> "Certificate " + result.getName() + " will expire in " + daysToExpire + " days");
					}
				};

				long prevDaysToExpire = Optional.ofNullable(triggeredEvents.put(result.getName(), daysToExpire)).orElse(Long.MAX_VALUE);
				if (daysToExpire != prevDaysToExpire) {
					logResult.accept(Level.WARNING);
					if (log.isLoggable(Level.FINEST)) {
						log.log(Level.FINEST, "Firing event: " + event);
					}
					eventBus.fire(event);
				} else {
					logResult.accept(Level.FINEST);
				}
			}
		}
	}

	static class APNSCertificatesExpirationEvent
			extends TasksEvent {

		private final SSLCertificateExpirationAware.Result result;
		private final String validFrom;
		private final String validTo;
		private final String message;

		public APNSCertificatesExpirationEvent(SSLCertificateExpirationAware.Result result) {
			super("APNSCertificatesExpirationEvent: " + result.getName(),
				  "Fired when APNS Push Certificate reaches validity expiration");
			this.result = result;

			this.validFrom = result.getValidFrom().format(DateTimeFormatter.ISO_DATE_TIME);
			this.validTo = result.getValidTo().format(DateTimeFormatter.ISO_DATE_TIME);
			if (!result.isValid(LocalDateTime.now())) {
				message = XMLUtils.escape("Certificate " + result.getName() + " is not valid!");
			} else {
				long daysToExpire = LocalDateTime.now().until(result.getValidTo(), ChronoUnit.DAYS);
				message = XMLUtils.escape(
						"Certificate " + result.getName() + " will expire in " + daysToExpire + " days");
			}
		}

		@Override
		public Map<String, String> getAdditionalData() {
			// @formatter:off
			return Map.of(
					"valid-from", validFrom,
					"valid-to", validTo,
					"message", message
			);
			// @formatter:onn
		}
	}
}
