/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.modules;

import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.component.modules.AbstractModule;
import tigase.criteria.Criteria;
import tigase.criteria.ElementCriteria;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.kernel.beans.config.ConfigField;
import tigase.push.EncryptedNotification;
import tigase.push.PlainNotification;
import tigase.push.PushNotificationsComponent;
import tigase.push.api.*;
import tigase.server.DataForm;
import tigase.server.Iq;
import tigase.server.Packet;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.Element;
import tigase.xml.XMLUtils;
import tigase.xmpp.Authorization;
import tigase.xmpp.PacketErrorTypeException;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Created by andrzej on 02.01.2017.
 */
@Bean(name = "publish-notification", parent = PushNotificationsComponent.class, active = true)
public class PublishNotificationModule
		extends AbstractModule {

	private static final String XMLNS = "urn:xmpp:push:0";
	private static final String PUBSUB_XMLNS = "http://jabber.org/protocol/pubsub";
	private static final Criteria CRITERIA = ElementCriteria.nameType(Iq.ELEM_NAME, "set")
			.add(ElementCriteria.name("pubsub", PUBSUB_XMLNS))
			.add(ElementCriteria.name("publish"));
	private String[] FEATURES = {XMLNS};

	@Inject(nullAllowed = true)
	private List<IPushProvider> pushProviders;
	private Map<String, IPushProvider> pushProvidersMap;

	@Inject
	private IPushRepository repository;

	@ConfigField(desc = "Default priority of notifications", alias = "def-priority")
	private INotification.Priority defPriority = INotification.Priority.high;

	private static void throwNodeMissing(BareJID serviceJid, String node) throws ComponentException {
		throw new ComponentException(Authorization.ITEM_NOT_FOUND, "Node " + node + " not found at " + serviceJid);
	}

	@Override
	public String[] getFeatures() {
		return FEATURES;
	}

	@Override
	public Criteria getModuleCriteria() {
		return CRITERIA;
	}

	@Override
	public void process(Packet packet) throws ComponentException, TigaseStringprepException {
		Element publish = packet.getElement().getChild("pubsub").getChild("publish");
		String node = publish.getAttributeStaticStr("node");
		if (node == null) {
			throw new ComponentException(Authorization.BAD_REQUEST, "Missing node attribute");
		}

		Element item = publish.getChild("item");
		if (item == null) {
			throw new ComponentException(Authorization.BAD_REQUEST, "Missing item to push");
		}

		Element notification = item.getChild("notification", XMLNS);
		if (notification == null) {
			throw new ComponentException(Authorization.BAD_REQUEST, "Missing notification to push");
		}

		try {
			pushNotification(packet.getStanzaTo().getBareJID(), packet.getStanzaFrom().getBareJID(), node,
							 notification).thenAccept(notificationId -> {
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "push notification for request from " + packet.getStanzaFrom() + " with id " +
							packet.getStanzaId() + " sent in notification with id " + notificationId);
				}
				write(packet.okResult((Element) null, 0));
			}).exceptionally(ex -> {
				handlePushNotificationException(packet, ex);
				return null;
			});
		} catch (RepositoryException ex) {
			throw new RuntimeException(ex);
		}
	}

	public void handlePushNotificationException(Packet packet, Throwable ex) {
		if (ex instanceof CompletionException) {
			handlePushNotificationException(packet, ex.getCause());
		} else {
			if (log.isLoggable(Level.FINE)) {
				log.log(Level.FINE, "push notification for request from " + packet.getStanzaFrom() + " with id " +
						packet.getStanzaId() + " failed", ex);
			}
			try {
				if (ex instanceof ComponentException) {
					write(((ComponentException) ex).getErrorCondition().getResponseMessage(packet, ex.getMessage(), false));
				} else {
					write(Authorization.INTERNAL_SERVER_ERROR.getResponseMessage(packet, null, false));
				}
			} catch (PacketErrorTypeException e) {
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.WARNING, "could not send response", e);
				}
			}
		}
	}

	public void setPushProviders(List<IPushProvider> pushProviders) {
		if (pushProviders == null) {
			pushProviders = Collections.emptyList();
		}
		this.pushProviders = pushProviders;
		this.pushProvidersMap = pushProviders.stream()
				.collect(Collectors.toConcurrentMap(provider -> provider.getName(), Function.identity()));
		FEATURES = Stream.concat(Stream.of(XMLNS), this.pushProviders.stream().map(provider -> provider.getName()))
				.toArray(x -> new String[x]);
	}

	protected IPushProvider getProvider(IPushSettings.IDevice device) {
		return pushProvidersMap.get(device.getProviderName());
	}

	protected CompletableFuture<String> pushNotification(BareJID serviceJid, BareJID senderJid, String node, Element notificationElem)
			throws ComponentException, TigaseStringprepException, RepositoryException {
		IPushSettings pushSettings = repository.getNodeSettings(serviceJid, node,
																PublishNotificationModule::throwNodeMissing);

		if ((!pushSettings.isOwner(senderJid)) && (!pushSettings.getOwnerJid().getDomain().equals(senderJid.getDomain()))) {
			throw new ComponentException(Authorization.FORBIDDEN, "Cannot publish item - you are not node owner");
		}

		INotification notification = parseNotification(pushSettings.getOwnerJid(), notificationElem);

		if (pushSettings.getVersion() == 0) {
			// old version, we should force clients to reregister and drop those entries!
			CompletableFuture<String> future = new CompletableFuture<>();
			for (IPushSettings.IDevice device : pushSettings.getDevices()) {
				IPushProvider provider = getProvider(device);
				if (provider == null) {
					log.log(Level.FINE, "Could not send push notification to provider " + device.getProviderName() +
							" - missing push provider!");
					continue;
				}

				provider.pushNotification(device, notification).thenAccept(result -> future.complete(result)).exceptionally(ex -> {
					future.completeExceptionally(ex);
					return null;
				});
			}
			return future;
		} else {
			// new version can have only one device per node
			IPushSettings.IDevice device = pushSettings.getDevices().get(0);
			IPushProvider provider = getProvider(device);
			if (provider == null) {
				return CompletableFuture.failedFuture(new ComponentException(Authorization.SERVICE_UNAVAILABLE, "Push provider not available"));
			}

			return provider.pushNotification(device, notification);
		}
	}

	// This method do the priority guessing, hopefully it will guess correctly..
	// based on mod_push and mod_cloud_notify source code
	// Anyone else what to do that in a different manner???
	protected Optional<INotification.Priority> guessPriority(Element notificationElem) {
		return Optional.ofNullable(notificationElem.getChild("x", "jabber:x:data")).map(x -> {
			// if `last-message-body` is NOT NULL then it is high priority
			if (DataForm.getFieldValue(x, "last-message-body") != null) {
				return INotification.Priority.high;
			}
			// if we do not have last-message-body lets check `message-count` if non-null this is Prosody low priority push
			if (DataForm.getFieldValue(x, "message-count") != null) {
				return INotification.Priority.low;
			}
			// in other case this is ejabberd high priority push (because it has `x` element)
			return INotification.Priority.high;
		});
	}

	protected INotification.Priority parsePriority(Element notificationElem) {
		return Optional.ofNullable(notificationElem.getChild("priority", "tigase:push:priority:0"))
				.map(Element::getCData)
				.map(INotification.Priority::valueOf)
				.or(() -> guessPriority(notificationElem))
				.orElse(INotification.Priority.low);
	}

	protected INotification parseNotification(BareJID userJid, Element notificationElem)
			throws ComponentException, TigaseStringprepException {
		INotification.Priority priority = parsePriority(notificationElem);
		Element encrypted = notificationElem.getChild("encrypted", "tigase:push:encrypt:0");
		if (encrypted == null) {
			Long messageCount = parseLong(notificationElem, "message-count");
			JID lastMessageSender = parseJID(notificationElem, "last-message-sender");
			String body = DataForm.getFieldValue(notificationElem, "last-message-body");

			String nickname = null;
			Element groupchat = notificationElem.getChild("groupchat", "http://tigase.org/protocol/muc#offline");
			if (groupchat != null) {
				Element nicknameEl = groupchat.getChild("nickname");
				if (nicknameEl != null) {
					nickname = XMLUtils.unescape(nicknameEl.getCData());
				}
			}

			return new PlainNotification(userJid, priority, messageCount, lastMessageSender, body, nickname);
		} else {
			// how to pass device id??
			INotification.Type type = "voip".equals(encrypted.getAttributeStaticStr("type")) ? INotification.Type.voip : INotification.Type.normal;
			return new EncryptedNotification(userJid, priority, type, encrypted.getCData(), encrypted.getAttributeStaticStr("iv"));
		}
	}

	protected Long parseLong(Element elem, String field) {
		String fieldVal = DataForm.getFieldValue(elem, field);
		if (fieldVal == null || fieldVal.isEmpty()) {
			return null;
		}
		return Long.parseLong(fieldVal);
	}

	protected JID parseJID(Element elem, String field) throws ComponentException, TigaseStringprepException {
		String fieldVal = DataForm.getFieldValue(elem, field);
		if (fieldVal == null || fieldVal.isEmpty()) {
			return null;
		}
		return JID.jidInstance(fieldVal);
	}

}
