/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.fcm;

import groovy.json.JsonBuilder;
import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.jaxmpp.core.client.exceptions.JaxmppException;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.kernel.beans.UnregisterAware;
import tigase.kernel.beans.config.ConfigField;
import tigase.kernel.beans.config.ConfigFieldType;
import tigase.kernel.beans.config.ConfigurationChangedAware;
import tigase.pubsub.Affiliation;
import tigase.push.PushNotificationsComponent;
import tigase.push.api.*;
import tigase.push.modules.AffiliationChangedModule;
import tigase.stats.ComponentStatisticsProvider;
import tigase.stats.Counter;
import tigase.stats.StatisticsList;

import java.util.*;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

/**
 * Created by andrzej on 05.01.2017.
 */
@Bean(name = "fcm-xmpp-api", parent = PushNotificationsComponent.class, active = false)
public class FcmXmppApiProvider
		implements IPushProvider, ConfigurationChangedAware, UnregisterAware, FcmProvider, ComponentStatisticsProvider {

	private static final Logger log = Logger.getLogger(FcmXmppApiProvider.class.getCanonicalName());
	@Inject
	private AffiliationChangedModule affiliationChangedModule;
	private List<FcmConnection> connectons = new ArrayList<>();
	@ConfigField(desc = "Provider description")
	private String description = "Push provider for FCM - XMPP";
	@ConfigField(desc = "Provider name")
	private String name = "fcm-xmpp-api";
	private BlockingDeque<FcmConnection> pool = new LinkedBlockingDeque<>();
	@ConfigField(desc = "Connections pool size", alias = "pool-size")
	private int poolSize = 2;
	@Inject
	private IPushRepository repository;
	@ConfigField(desc = "Sender ID", alias = "sender-id")
	private String senderId;
	@ConfigField(desc = "Server key", alias = "server-key", type = ConfigFieldType.Password)
	private String serverKey;

	private final Counter numberOfPushes = new Counter("Number of push notifications sent", Level.FINE);
	private final Counter numberOfErrors = new Counter("Number of failed push notifications", Level.FINE);
	
	@Override
	public String getName() {
		return name;
	}

	@Override
	public String getDescription() {
		return description;
	}

	@Override
	public CompletableFuture<String> pushNotification(IPushSettings.IDevice device, INotification notification) {
		CompletableFuture<String> future = new CompletableFuture<>();
		Map<String, Object> payload = preparePayload(notification);
		payload.put("to", device.getDeviceId());
		String payloadStr = new JsonBuilder(payload).toString();
		int i = 0;
		numberOfPushes.inc();
		while (i <= 3 && !future.isDone()) {
			try {
				if (sendNotification(payloadStr)) {
					numberOfPushes.inc();
					future.complete((String) payload.get("message_id"));
				}
			} catch (JaxmppException ex) {
				log.log(Level.WARNING, "Failed to send push notification, retry " + i, ex);
				if (i == 3) {
					future.completeExceptionally(ex);
				}
			} catch (Throwable ex) {
				log.log(Level.WARNING, "Failed to send push notification, retry " + i, ex);
				if (i == 3) {
					future.completeExceptionally(ex);
				}
			}
			i++;
		}
		if (future.isCompletedExceptionally()) {
			numberOfErrors.inc();
		}
		return future;
	}

	@Override
	public void beforeUnregister() {
		for (FcmConnection conn : connectons) {
			conn.setProvider(null);
			conn.stop();
		}

		connectons.clear();
	}

	@Override
	public void beanConfigurationChanged(Collection<String> changedFields) {
		if (changedFields.contains("serverKey") || changedFields.contains("senderId") ||
				changedFields.contains("poolSize")) {

			for (FcmConnection conn : connectons) {
				conn.setProvider(null);
				conn.stop();
			}

			connectons.clear();

			for (int i = 0; i < poolSize; i++) {
				FcmConnection conn = new FcmConnection(serverKey, senderId, false);
				conn.setProvider(this);
				conn.start();
				connectons.add(conn);
			}
		}
	}

	public void connected(FcmConnection conn) {
		this.pool.offer(conn);
	}

	public void disconnected(FcmConnection conn) {
		this.pool.remove(conn);
	}

	public void unregisterDevice(String deviceId) {
		try {
			Stream<IPushSettings> settingsStream = repository.getNodeSettings(getName(), deviceId);
			settingsStream.forEach(settings -> {
				// pushSettings is not equal to settings!
				try {
					IPushSettings pushSettings = repository.unregisterDevice(settings.getServiceJid(),
																			 settings.getOwnerJid(), getName(),
																			 deviceId);
					if (pushSettings != null && (pushSettings.getDevices().isEmpty() || pushSettings.getVersion() > 0)) {
						affiliationChangedModule.notifyAffiliationChanged(pushSettings.getServiceJid(),
																		  pushSettings.getOwnerJid(),
																		  pushSettings.getNode(), Affiliation.none);
					}
				} catch (RepositoryException | ComponentException ex) {
					log.log(Level.WARNING, getName() + ", failed to unregister device = " + deviceId, ex);
				}
			});
		} catch (RepositoryException ex) {
			log.log(Level.WARNING, getName() + ", failed to unregister device = " + deviceId, ex);
		}
	}

	@Override
	public void everyHour() {
		numberOfPushes.everyHour();
		numberOfErrors.everyHour();
	}

	@Override
	public void everyMinute() {
		numberOfPushes.everyMinute();
		numberOfErrors.everyMinute();
	}

	@Override
	public void everySecond() {
		numberOfPushes.everySecond();
		numberOfErrors.everySecond();
	}

	@Override
	public void getStatistics(String compName, StatisticsList list) {
		numberOfPushes.getStatistics(compName + "/" + getName(), list);
		numberOfErrors.getStatistics(compName + "/" + getName(), list);
	}

	protected Map<String, Object> preparePayload(INotification notification) {
		Map<String, Object> payload = new HashMap<>();

		payload.put("message_id", "m-" + UUID.randomUUID().toString());
		payload.put("priority", "high");
		//payload.put("dry_run", "true");

		if (notification instanceof IPlainNotification) {
			IPlainNotification plainNotification = (IPlainNotification) notification;
			Map<String, Object> data = new HashMap<>();
			data.put("account", notification.getAccount().toString());

			plainNotification.ifMessageCount(count -> data.put("unread-messages", count));
			plainNotification.ifLastMessageSender(sender -> data.put("sender", sender.toString()));
			plainNotification.ifGroupchatSenderNickname(nickname -> data.put("nickname", nickname));
			plainNotification.ifLastMessageBody(body -> {
				if (body.length() > 512) {
					body = body.substring(0, 500) + "...";
				}
				data.put("body", body);
			});
			payload.put("data", data);
		} else if (notification instanceof IEncryptedNotification) {
			IEncryptedNotification encryptedNotification = (IEncryptedNotification) notification;
			Map<String, Object> data = new HashMap<>();
			data.put("account", notification.getAccount().toString());
			data.put("encrypted", encryptedNotification.getEncrypted());
			payload.put("data", data);
		}

		return payload;
	}

	public void pushNotificationFailed(String error) {
		numberOfErrors.inc();
	}

	private boolean sendNotification(String payload) throws Throwable {
		FcmConnection conn = null;
		try {
			conn = pool.poll(5, TimeUnit.SECONDS);
			if (conn != null) {
				conn.sendNotification(payload);
				pool.offer(conn);
			}
		} finally {
			if (conn != null) {
				pool.offer(conn);
			}
		}
		return false;
	}

}
