/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.fcm;

import groovy.json.JsonSlurper;
import tigase.jaxmpp.core.client.*;
import tigase.jaxmpp.core.client.criteria.Criteria;
import tigase.jaxmpp.core.client.criteria.ElementCriteria;
import tigase.jaxmpp.core.client.exceptions.JaxmppException;
import tigase.jaxmpp.core.client.xml.Element;
import tigase.jaxmpp.core.client.xml.ElementFactory;
import tigase.jaxmpp.core.client.xml.XMLException;
import tigase.jaxmpp.core.client.xmpp.modules.auth.AuthModule;
import tigase.jaxmpp.core.client.xmpp.stanzas.Message;
import tigase.jaxmpp.j2se.Jaxmpp;
import tigase.jaxmpp.j2se.connectors.socket.SocketConnector;

import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import static tigase.jaxmpp.core.client.xmpp.modules.auth.SaslMechanism.FORCE_AUTHZID;

/**
 * Created by andrzej on 20.05.2017.
 */
public class FcmConnection
		implements Jaxmpp.LoggedInHandler, Jaxmpp.LoggedOutHandler {

	private static final Logger log = Logger.getLogger(FcmConnection.class.getCanonicalName());

	private static final String FCM_SERVER = "fcm-xmpp.googleapis.com";

	private Jaxmpp jaxmpp;

	private FcmProvider provider = null;

	public FcmConnection(String serverKey, String senderId, boolean testing) {
		jaxmpp = new Jaxmpp();
		jaxmpp.getModulesManager().register(new FcmModule(this));

		jaxmpp.getEventBus().addHandler(JaxmppCore.LoggedInHandler.LoggedInEvent.class, this);
		jaxmpp.getEventBus().addHandler(JaxmppCore.LoggedOutHandler.LoggedOutEvent.class, this);

		jaxmpp.getConnectionConfiguration().setUserJID(BareJID.bareJIDInstance(senderId, FCM_SERVER));
		jaxmpp.getConnectionConfiguration().setUserPassword(serverKey);
		jaxmpp.getConnectionConfiguration().setServer(FCM_SERVER);
		jaxmpp.getConnectionConfiguration().setPort(testing ? 5236 : 5235);

		jaxmpp.getSessionObject().setUserProperty(SocketConnector.USE_PLAIN_SSL_KEY, true);
		jaxmpp.getSessionObject().setUserProperty(AuthModule.LOGIN_USER_NAME_KEY, senderId + "@gcm.googleapis.com");
		jaxmpp.getSessionObject().setUserProperty(FORCE_AUTHZID, false);
	}

	public void setProvider(FcmProvider handler) {
		this.provider = handler;
	}

	public void start() {
		try {
			synchronized (jaxmpp) {
				if (jaxmpp.getConnector().getState() == Connector.State.disconnected) {
					jaxmpp.login();
				}
			}
		} catch (JaxmppException ex) {
			jaxmpp.getEventBus().fire(new JaxmppCore.LoggedOutHandler.LoggedOutEvent(jaxmpp.getSessionObject()));
		}
	}

	public void stop() {
		try {
			jaxmpp.disconnect();
		} catch (JaxmppException ex) {
			log.log(Level.FINEST, "Could not stop connection to FCM", ex);
		}
	}

	@Override
	public void onLoggedIn(SessionObject sessionObject) {
		if (this.provider != null) {
			provider.connected(this);
		}
	}

	@Override
	public void onLoggedOut(SessionObject sessionObject) {
		if (this.provider != null) {
			provider.disconnected(this);
			start();
		}
	}

	public void sendNotification(String payload) throws JaxmppException {
		if (!jaxmpp.isConnected()) {
			start();
			throw new JaxmppException("Connection to FCM server not ready");
		}
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "sending notification, data = " + payload);
		}
		Element gcm = ElementFactory.create("gcm", payload, "google:mobile:data");
		Message message = Message.createMessage();
		message.addChild(gcm);
		jaxmpp.send(message);
	}

	protected void pushNotificationFailed(String error) {
		if (provider != null) {
			provider.pushNotificationFailed(error);
		}
	}

	private void deviceUnregistered(String deviceId) {
		if (provider != null) {
			provider.unregisterDevice(deviceId);
		}
	}

	public static class FcmModule
			implements XmppModule {

		private static final Criteria CRIT = ElementCriteria.name("message");

		private FcmConnection conn;

		public FcmModule(FcmConnection conn) {
			this.conn = conn;
		}

		@Override
		public Criteria getCriteria() {
			return CRIT;
		}

		@Override
		public String[] getFeatures() {
			return new String[0];
		}

		@Override
		public void process(Element message) throws XMPPException, XMLException, JaxmppException {
			Element gcm = message.getFirstChild("gcm");
			if (gcm == null || !"google:mobile:data".equals(gcm.getXMLNS())) {
				return;
			}

			String json = gcm.getValue();
			try {
				Map<String, Object> payload = (Map<String, Object>) new JsonSlurper().parse(json.getBytes("UTF-8"));

				Object type = payload.get("message_type");
				if (type == null) {
					log.log(Level.FINEST, "Received upstream message, ignoring...");
					return;
				}

				switch (type.toString()) {
					case "ack":
						handleAck(payload);
						break;
					case "nack":
						handleNack(payload);
						break;
					case "receipt":
						log.log(Level.FINEST, "Received upstream message of type receipt, data = " + json);
						break;
					case "control":
						handleControl(payload);
					default:
						log.log(Level.FINER,
								"Received upstream message of unknown type, type " + type + ", data = " + json);
				}
			} catch (Exception e) {
				log.log(Level.WARNING, "Exception processing message received from FCM, data = " + json, e);
			}
		}

		private void handleAck(Map<String, Object> payload) {
			// nothing to do for now
		}

		private void handleNack(Map<String, Object> payload) {
			String error = (String) payload.get("error");

			if (error == null) {
				log.log(Level.FINEST, "Received NACK without an error code!");
				return;
			}

			conn.pushNotificationFailed(error);

			switch (error) {
				case "BAD_REGISTRATION":
				case "DEVICE_UNREGISTERED":
					String deviceId = (String) payload.get("from");
					if (log.isLoggable(Level.FINEST)) {
						log.log(Level.FINEST,
								"Received NACK with type " + error + ", unregistering device id " + deviceId);
					}
					conn.deviceUnregistered(deviceId);
					break;
				default:
					log.log(Level.FINER,
							"Received NACK with error = " + error + ", message = " + payload.get("error_description"));
			}
		}

		private void handleControl(Map<String, Object> payload) {
			String type = (String) payload.get("control_type");

			if ("CONNECTION_DRAINING".equals(type)) {
				conn.stop();
			} else {
				log.log(Level.FINEST, "Received control packet with unknown type = " + type + ", data = " + payload);
			}
		}
	}

}
