/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.apns;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class ApnsPayload {

	public static ApnsPayload.Builder newBuilder() {
		return new ApnsPayload.Builder();
	}

	private final Map<String, Object> aps;
	private final Map<String, Object> alert;
	private final Map<String, Object> customFields;

	private ApnsPayload(Builder builder) {
		this.aps = builder.aps;
		this.alert = builder.alert;
		this.customFields = builder.customFields;
	}
	
	public String toPayloadString() {
		if (!alert.isEmpty()) {
			aps.put("alert", alert);
		}
		if (!aps.isEmpty()) {
			this.customFields.put("aps", aps);
		}
		return valueToString(this.customFields);
	}

	private static String valueToString(Object value) {
		if (value instanceof Number) {
			return value.toString();
		} else if (value instanceof String) {
			return escapeValue(value.toString());
		} else if (value instanceof List) {
			return "[" + ((List) value).stream().map(ApnsPayload::valueToString).collect(Collectors.joining(", ")) + "]";
		} else if (value instanceof Map) {
			return "{" + ((Stream<Map.Entry>) ((Map) value).entrySet().stream()).map(
					(Map.Entry e) -> "\"" + e.getKey() + "\" : " + valueToString(e.getValue()))
					.collect(Collectors.joining(", ")) + "}";
		} else {
			return "null";
		}
	}

	private static String escapeValue(String in) {
		return "\"" + in.replace("\"", "\\\"") + "\"";
	}

	public static class Builder {

		private Map<String, Object> aps = new HashMap<>();
		private Map<String, Object> alert = new HashMap<>();
		private Map<String, Object> customFields = new HashMap<>();

		private Builder() {
		}
		
		public Builder badge(Integer badge) {
			this.set(aps,"badge", badge);
			return this;
		}

		public Builder sound(String sound) {
			this.set(aps, "sound", sound);
			return this;
		}

		public Builder title(String title) {
			return this.set(alert,"title", title);
		}

		public Builder subtitle(String subtitle) {
			return this.set(alert, "subtitle", subtitle);
		}

		public Builder body(String body) {
			return this.set(alert, "body", body);
		}

		public Builder category(String category) {
			return this.set(aps, "category", category);
		}

		public Builder threadId(String threadId) {
			return this.set(aps, "thread-id", threadId);
		}

		public Builder instantDeliveryOrSilentNotification() {
			return this.contentAvailable(1);
		}

		public Builder contentAvailable(Integer value) {
			return this.set(aps, "content-available", value);
		}

		public Builder mutableContent(Integer value) {
			return this.set(aps, "mutable-content", value);
		}

		public Builder customField(String name, Object value) {
			return this.set(customFields, name, value);
		}

		public ApnsPayload build() {
			return new ApnsPayload(this);
		}

		private Builder set(Map<String, Object> map, String key, Object value) {
			if (value != null) {
				map.put(key, value);
			} else {
				map.remove(key);
			}
			return this;
		}
	}

}
