/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.apns;

import java.util.UUID;

public class ApnsNotification {

	private final String id;
	private final String deviceId;
	private final PushType pushType;
	private final ApnsPayload payload;
	private final String collapseId;
	private final String topic;

	private ApnsNotification(Builder builder) {
		this.payload = builder.payload;
		this.pushType = builder.pushType;
		this.deviceId = builder.deviceId;
		this.collapseId = builder.collapseId;
		this.topic = builder.topic;
		if (this.deviceId == null) {
			throw new IllegalArgumentException("DeviceID is required!");
		}
		if (this.pushType == null) {
			throw new IllegalArgumentException("Push Type can not be null!");
		}
		if (this.payload == null) {
			throw new IllegalArgumentException("Payload is required!");
		}

		this.id = UUID.randomUUID().toString();
	}

	public static Builder newBuilder() {
		return new Builder();
	}

	public String getDeviceId() {
		return deviceId;
	}

	public String getId() {
		return id;
	}

	public PushType getPushType() {
		return pushType;
	}

	public int getPriority() {
		return pushType == PushType.background ? 5 : 10;
	}

	public ApnsPayload getPayload() {
		return payload;
	}

	public String getCollapseId() {
		return collapseId;
	}

	public String getTopic() {
		return topic;
	}

	@Override
	public String toString() {
		return "[id: " + getId() + ", deviceId: " + getDeviceId() + ", type: " + getPushType().name() +
				", collapseId: " + getCollapseId() + ", payload: " + getPayload().toPayloadString() + " ]";
	}

	public static class Builder {

		private ApnsPayload payload;
		private PushType pushType;
		private String deviceId;
		private String collapseId;
		private String topic;

		private Builder() {
		}

		public Builder collapseId(String collapseId) {
			this.collapseId = collapseId;
			return this;
		}

		public Builder deviceId(String deviceId) {
			this.deviceId = deviceId;
			return this;
		}

		public Builder pushType(PushType pushType) {
			this.pushType = pushType;
			return this;
		}

		public Builder payload(ApnsPayload payload) {
			this.payload = payload;
			return this;
		}

		public Builder topic(String topic) {
			this.topic = topic;
			return this;
		}

		public ApnsNotification build() {
			return new ApnsNotification(this);
		}
	}

	public enum PushType {
		alert,
		background,
		voip
	}

}
