/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.apns;

import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.db.TigaseDBException;
import tigase.db.UserRepository;
import tigase.eventbus.EventBus;
import tigase.eventbus.EventBusFactory;
import tigase.eventbus.HandleEvent;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Initializable;
import tigase.kernel.beans.Inject;
import tigase.kernel.beans.UnregisterAware;
import tigase.kernel.beans.config.ConfigField;
import tigase.kernel.beans.config.ConfigFieldType;
import tigase.kernel.beans.config.ConfigurationChangedAware;
import tigase.pubsub.Affiliation;
import tigase.push.AbstractProvider;
import tigase.push.PushNotificationsComponent;
import tigase.push.api.*;
import tigase.push.modules.AffiliationChangedModule;
import tigase.push.monitor.SSLCertificateExpirationAware;
import tigase.stats.ComponentStatisticsProvider;
import tigase.stats.Counter;
import tigase.stats.StatisticsList;
import tigase.xmpp.Authorization;

import java.io.IOException;
import java.io.Serializable;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

import static tigase.push.PushNotificationsComponent.COMPONENT_ID;

/**
 * Created by andrzej on 05.01.2017.
 */
@Bean(name = "apns-binary-api", parent = PushNotificationsComponent.class, active = false)
public class APNsBinaryApiProvider extends AbstractProvider
		implements IPushProvider, ConfigurationChangedAware, Initializable, UnregisterAware, ComponentStatisticsProvider,
				   SSLCertificateExpirationAware {

	private static final Logger log = Logger.getLogger(APNsBinaryApiProvider.class.getCanonicalName());
	protected final EventBus eventBus = EventBusFactory.getInstance();
	@Inject
	private AffiliationChangedModule affiliationChangedModule;
	@ConfigField(desc = "Password for certificate file", alias = "cert-password", type = ConfigFieldType.Password)
	private String certificatePassword;
	@ConfigField(desc = "Path to certificate file (.p12)", alias = "cert-file")
	private String certificatePath;
	// FIXME: Should we remove annotation as reloadAPNSSecretsFromRepository() overwrites value from the config with value from user repository
	@ConfigField(desc = "Certificate in PEM format", alias = "cert-base64")
	private String base64certificate;
	@ConfigField(desc = "Path to PushKit certificate file (.p12)", alias = "pushkit-cert-file")
	private String pushkitCertificatePath;
	// FIXME: Should we remove annotation as reloadAPNSSecretsFromRepository() overwrites value from the config with value from user repository
	@ConfigField(desc = "Path to PushKit certificate file (.p12)", alias = "pushkit-cert-base64")
	private String base64pushkitCertificate;
	@ConfigField(desc = "Path to encryption file (.p12)", alias = "key-file")
	private String encryptionKeyPath;
	@ConfigField(desc = "ID of encryption key", alias = "key-id")
	private String encryptionKeyId;
	private String encryptionKey;
	@ConfigField(desc = "Team ID", alias = "team-id")
	private String teamId;
	@ConfigField(desc = "Provider description")
	private String description = "Push provider for APNs - Binary API";
	@ConfigField(desc = "Fallback to sandbox")
	private boolean fallbackToSandbox = false;
	@ConfigField(desc = "APNS-Topic", alias = "apns-topic")
	private String apnsTopic;
	@Inject
	private IPushRepository repository;
	@ConfigField(desc = "Sandbox")
	private boolean sandbox = false;
	private ApnsService service;
	private ApnsService sandboxService;
	private ApnsService pushkitService;
	private ApnsService pushkitSandboxService;

	private final Counter numberOfPushes = new Counter("Number of push notifications sent", Level.FINE);
	private final Counter numberOfErrors = new Counter("Number of failed push notifications", Level.FINE);

	public APNsBinaryApiProvider() {
		super("apns-binary-api");
	}

	@Override
	public String getDescription() {
		return description;
	}

	@Override
	public void initialize() {
		if (eventBus != null) {
			eventBus.registerAll(this);
		}
	}

	@Override
	public Optional<Integer> maxPayloadSize() {
		return Optional.of(3 * 1024);
	}

	@Override
	public Set<Feature> supportedFeatures() {
		return Set.of(Feature.plain, Feature.encrypted);
	}

	@Override
	public CompletableFuture<String> pushNotification(IPushSettings.IDevice device, INotification notification) {
		// TODO: better support for error handling may be needed
		boolean isVoIP = notification.getType() == INotification.Type.voip && notification.getPriority() == INotification.Priority.high;
		String deviceId = (!isVoIP) ? device.getDeviceId() : device.getDeviceSecondId();
		if (deviceId == null) {
			log.log(Level.FINEST, "there is no device (second) id for push notification for account " + notification.getAccount());
			if (isVoIP) {
				return CompletableFuture.completedFuture(null);
			} else {
				return CompletableFuture.failedFuture(new ComponentException(Authorization.ITEM_NOT_FOUND, "Unknown device ID"));
			}
		}

		ApnsPayload.Builder builder = preparePayload(APNS.newPayload(), notification);
		ApnsNotification.Builder apnsNotificationBuilder = APNS.newNotification();
		if (isVoIP && pushkitService != null) {
			apnsNotificationBuilder.pushType(ApnsNotification.PushType.voip);
		} else {
			if (apnsTopic != null) {
				String topic = apnsTopic;
				if (isVoIP) {
					topic += ".voip";
				}
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "setting topic for notifications " + topic);
				}
				apnsNotificationBuilder.topic(topic);
			}
			switch (notification.getPriority()) {
				case low:
					apnsNotificationBuilder.pushType(ApnsNotification.PushType.background);
					break;

				case high:
					builder.body("New message!").sound("default").mutableContent(1).category("MESSAGE").threadId(notification.getAccount().toString());
					if (isVoIP) {
						apnsNotificationBuilder.pushType(ApnsNotification.PushType.voip);
					} else {
						apnsNotificationBuilder.pushType(ApnsNotification.PushType.alert);
					}
					break;
			}
		}
		
		ApnsNotification apnsNotification = apnsNotificationBuilder.deviceId(deviceId).payload(builder.build()).build();
		
		numberOfPushes.inc();
		if (isVoIP && pushkitService != null) {
			return pushNotification(device, apnsNotification, pushkitService, pushkitSandboxService);
		} else {
			return pushNotification(device, apnsNotification, service, sandboxService);
		}
	}

	@Override
	public Stream<Result> getSSLCertificatesValidPeriod() {
		Stream<Result> result = Stream.empty();
		try {
			if (base64certificate != null) {
				result = Stream.concat(result, APNSUtil.getCertificateValidPeriodFromBase64(base64certificate,
																							certificatePassword));
			} else if (certificatePath != null) {
				result = Stream.concat(result, APNSUtil.getCertificateValidPeriodFromFile(certificatePath,
																						  certificatePassword));
			}
		} catch (Throwable ex) {
			log.log(Level.WARNING, ex, () -> "Could not retrieve valid period for Push SSL certificate");
		}

		try {
			if (base64pushkitCertificate != null) {
				result = Stream.concat(result, APNSUtil.getCertificateValidPeriodFromBase64(base64pushkitCertificate,
																							certificatePassword));
			} else if (pushkitCertificatePath != null) {
				result = Stream.concat(result, APNSUtil.getCertificateValidPeriodFromFile(pushkitCertificatePath,
																						  certificatePassword));
			}
		} catch (Throwable ex) {
			log.log(Level.WARNING, ex, () -> "Could not retrieve valid period for PushKit SSL certificate");
		}
		return result;
	}

	protected CompletableFuture<String> pushNotification(IPushSettings.IDevice device, ApnsNotification apnsNotification, ApnsService service, ApnsService fallbackService) {
		return sendWithRetryAndFallback(service, fallbackService, apnsNotification).whenComplete((r,ex) -> {
			if (ex != null && ex instanceof ApnsServiceException) {
				ApnsServiceException e = (ApnsServiceException) ex;
				switch (e.getErrorType()) {
					case badDeviceToken:
					case deviceTokenNotForTopic:
					case unregistred:
						unregisterDevice(device.getDeviceId());
						break;
					default:
						break;
				}
			}
		});
	}

	private CompletableFuture<String> sendWithRetry(ApnsService service, ApnsNotification apnsNotification) {
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "sending push notification " + apnsNotification.getId() + " at service " + service);
		}
		return sendWithRetry(service, apnsNotification, 1).whenComplete((r,ex) -> {
			if (ex == null) {
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST,
							"sent push notification = " + apnsNotification.getId() + " at service = " +
									service);
				}
			} else if (ex instanceof CompletionException) {
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "failed to send push notification = " + apnsNotification.getId() + " at service = " + service + ", error: " +  ex.getCause());
				}
			} else {
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "failed to send push notification = " + apnsNotification.getId() + " at service = " + service, ex);
				}
			}
		});
	}
	
	private CompletableFuture<String> sendWithRetry(ApnsService service, ApnsNotification apnsNotification, int tryNo) {
		CompletableFuture<String> future = new CompletableFuture<String>();
		try {
			service.push(apnsNotification).thenAccept(result -> future.complete(result)).exceptionally(ex -> {
				if (tryNo <= 3 && shouldRetry(ex)) {
					sendWithRetry(service, apnsNotification, tryNo + 1).thenAccept(result -> future.complete(result)).exceptionally(ex1 -> {
						future.completeExceptionally(ex1);
						return null;
					});
				} else {
					future.completeExceptionally(ex);
				}
				return null;
			});
		} catch (Throwable ex) {
			future.completeExceptionally(ex);
		}
		return future;
	}

	private CompletableFuture<String> sendWithRetryAndFallback(ApnsService service, ApnsService fallbackService, ApnsNotification apnsNotification) {
		CompletableFuture<String> future = new CompletableFuture<>();
		sendWithRetry(service, apnsNotification).thenAccept(result -> future.complete(result)).exceptionally(ex -> {
			if (fallbackService != null && isErrorType(ex, ApnsService.ErrorType.badDeviceToken)) {
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "trying to use fallback service " + fallbackService + " for notification " + apnsNotification.getId());
				}
				sendWithRetry(fallbackService, apnsNotification).thenAccept(result -> future.complete(result)).exceptionally(ex1 -> {
					future.completeExceptionally(ex1);
					return null;
				});
			} else {
				future.completeExceptionally(ex);
			}
			return null;
		});
		
		return future;
	}

	private static boolean shouldRetry(Throwable ex) {
		if (ex instanceof CompletionException) {
			return shouldRetry(ex.getCause());
		}
		if (ex instanceof IOException) {
			// in case of IOException we should always retry..
			return true;
		}
		if (ex instanceof ApnsServiceException) {
			return ((ApnsServiceException) ex).getErrorType().shouldRetry();
		}
		return false;
	}

	private static boolean isErrorType(Throwable ex, ApnsService.ErrorType errorType) {
		if (ex instanceof CompletionException) {
			return isErrorType(ex.getCause(), errorType);
		}
		if (ex instanceof ApnsServiceException) {
			return ((ApnsServiceException) ex).isErrorType(errorType);
		}
		return false;
	}

	@Override
	public void beforeUnregister() {
		if (eventBus != null) {
			eventBus.unregisterAll(this);
		}
		setService(null);
		setSandboxService(null);
		setPushKitService(null);
		setPushKitSandboxService(null);
	}

	@HandleEvent(filter = HandleEvent.Type.remote)
	public void certificateChange(APNSCertificateChangedEvent event) {
		try {
			reloadAPNSSecretsFromRepository();
		} catch (TigaseDBException e) {
			log.log(Level.WARNING, "Reloading APNS certificate failed", e);
		}
	}

	public void reloadAPNSCertificateFromRepository() throws TigaseDBException {
		reloadAPNSSecretsFromRepository();
	}

	public void reloadAPNSSecretsFromRepository() throws TigaseDBException {
		final ArrayList<String> changedFields = new ArrayList<>();
		this.base64certificate = getData("base64certificate");
		if (this.base64certificate != null) {
			changedFields.add("base64certificate");
		}
		this.base64pushkitCertificate = getData("base64pushkitCertificate");
		if (this.base64pushkitCertificate != null) {
			changedFields.add("base64pushkitCertificate");
		}
		if (base64certificate != null || base64pushkitCertificate != null) {
			this.certificatePassword = getData("certificatePassword");
			if (this.certificatePassword != null) {
				changedFields.add("certificatePassword");
			}
		}
		this.encryptionKey = getData("encryptionKey");
		if (this.encryptionKey != null) {
			changedFields.add("encryptionKey");
			this.encryptionKeyId = getData("encryptionKeyId");
			if (encryptionKeyId != null) {
				changedFields.add("encryptionKeyId");
			}
			this.teamId = getData("teamId");
			if (teamId != null) {
				changedFields.add("teamId");
			}
		}
		beanConfigurationChanged(changedFields);
	}

	public void setAPNSCertificate(String base64certificate, String base64pushkitCertificate, String certificatePassword) throws TigaseDBException {
		Objects.requireNonNull(base64certificate);
		Objects.requireNonNull(base64pushkitCertificate);
		Objects.requireNonNull(certificatePassword);
		this.base64certificate = base64certificate;
		this.base64pushkitCertificate = base64pushkitCertificate;
		this.certificatePassword = certificatePassword;
		beanConfigurationChanged(List.of("base64certificate", "base64pushkitCertificate", "certificatePassword"));
		storeAPNSCertificateInRepository(base64certificate, base64pushkitCertificate, certificatePassword);
		eventBus.fire(new APNSCertificateChangedEvent(COMPONENT_ID, getName()));
	}

	public void setAPNSEncryptionKey(String encryptionKeyId, String encryptionKey, String teamId) throws TigaseDBException {
		Objects.requireNonNull(encryptionKeyId);
		Objects.requireNonNull(encryptionKey);
		Objects.requireNonNull(teamId);
		this.encryptionKeyId = encryptionKeyId;
		this.encryptionKey = encryptionKey;
		this.teamId = teamId;
		beanConfigurationChanged(List.of("encryptionKeyId", "encryptionKey", "teamId"));
		storeAPNSEncryptionKeyInRepository(encryptionKeyId, encryptionKey, teamId);
		// reusing APNSCertificateChangedEvent, but this even should be renamed in the future releases
		eventBus.fire(new APNSCertificateChangedEvent(COMPONENT_ID, getName()));
	}

	@Override
	public void beanConfigurationChanged(Collection<String> changedFields) {
		if (changedFields.contains("certificatePath") || changedFields.contains("pushkitCertificatePath") ||
				changedFields.contains("base64certificate") || changedFields.contains("base64pushkitCertificate") ||
				changedFields.contains("certificatePassword") || changedFields.contains("poolSize") ||
				changedFields.contains("fallbackToSandbox") || changedFields.contains("userRepository") ||
				changedFields.contains("encryptionKey") || changedFields.contains("encryptionKeyId") ||
				changedFields.contains("encryptionKeyPath") || changedFields.contains("teamId")) {

			try {
				if (!hasMainConfig()) {
					setSandboxService(null);
					setService(null);
				} else {
					setSandboxService(fallbackToSandbox ? newMainServiceBuilder(false) : null);
					setService(newMainServiceBuilder(!sandbox));
				}

				if (!hasPushKitConfig()) {
					setPushKitSandboxService(null);
					setPushKitService(null);
				} else {
					setPushKitSandboxService(fallbackToSandbox ? newPushkitServiceBuilder(false) : null);
					setPushKitService(newPushkitServiceBuilder(!sandbox));
				}
			} catch (InvalidKeySpecException|NoSuchAlgorithmException|IOException ex) {
				throw new RuntimeException("Could not configure APNs provider!", ex);
			}
		}
	}

	private boolean hasMainConfig() {
		return certificatePath != null || base64certificate != null || encryptionKey != null || encryptionKeyPath != null;
	}

	private boolean hasPushKitConfig() {
		return pushkitCertificatePath != null || base64pushkitCertificate != null;
	}

	private ApnsService newMainServiceBuilder(boolean appleDestination)
			throws InvalidKeySpecException, NoSuchAlgorithmException, IOException {
		return APNS.newService()
				.withEncryptionKey(encryptionKeyId, encryptionKey, encryptionKeyPath, teamId)
				.withCert(certificatePath, base64certificate, certificatePassword)
				.withAppleDestination(appleDestination)
				.build();
	}

	private ApnsService newPushkitServiceBuilder(boolean appleDestination)
			throws InvalidKeySpecException, NoSuchAlgorithmException, IOException {
		// this config does not work for encryption keys (P8) as VoIP with token based authentication is covered by main
		// APNS service this is covered by
		return APNS.newService()
				.withCert(pushkitCertificatePath, base64pushkitCertificate, certificatePassword)
				.withAppleDestination(appleDestination)
				.build();
	}


	@Override
	public void everyHour() {
		numberOfPushes.everyHour();
		numberOfErrors.everyHour();
	}

	@Override
	public void everyMinute() {
		numberOfPushes.everyMinute();
		numberOfErrors.everyMinute();
	}

	@Override
	public void everySecond() {
		numberOfPushes.everySecond();
		numberOfErrors.everySecond();
	}

	@Override
	public void getStatistics(String compName, StatisticsList list) {
		numberOfPushes.getStatistics(compName + "/" + getName(), list);
		numberOfErrors.getStatistics(compName + "/" + getName(), list);
	}

	void storeAPNSEncryptionKeyInRepository(String encryptionKeyId, String encryptionKey, String teamId) throws TigaseDBException {
		setData("encryptionKeyId", encryptionKeyId);
		setData("encryptionKey", encryptionKey);
		setData("teamId", teamId);
		deleteAPNSCertificateFromRepository();
	}

	private void deletePNSEncryptionKeyFromRepository() throws TigaseDBException {
		removeData("encryptionKeyId", "encryptionKey", "teamId");
	}

	void storeAPNSCertificateInRepository(String base64certificate, String base64pushkitCertificate, String certificatePassword) throws TigaseDBException {
		setData("base64certificate", base64certificate);
		setData("base64pushkitCertificate", base64pushkitCertificate);
		setData("certificatePassword", certificatePassword);
		deletePNSEncryptionKeyFromRepository();
	}

	private void deleteAPNSCertificateFromRepository() throws TigaseDBException {
		removeData("base64certificate", "base64pushkitCertificate", "certificatePassword");
	}

	protected ApnsPayload.Builder preparePayload(ApnsPayload.Builder builder, INotification notification) {
		if (notification instanceof IPlainNotification) {
			return preparePlainPayload(builder, (IPlainNotification) notification);
		} else if (notification instanceof IEncryptedNotification) {
			return prepareEncryptedPayload(builder, (IEncryptedNotification) notification);
		} else {
			throw new IllegalArgumentException(
					"Not supported notification class:" + notification.getClass().getCanonicalName());
		}
	}

	protected ApnsPayload.Builder preparePlainPayload(ApnsPayload.Builder builder, IPlainNotification notification) {
		if (notification.getPriority() != INotification.Priority.high) {
			builder.instantDeliveryOrSilentNotification();
		}
		builder.customField("account", notification.getAccount().toString());

		notification.ifMessageCount(count -> builder.customField("unread-messages", count));
		notification.ifLastMessageSender(sender -> builder.customField("sender", sender.getBareJID().toString()));
		notification.ifGroupchatSenderNickname(nickname -> builder.customField("nickname", nickname));
		notification.ifLastMessageBody(body -> {
			// reducing body size to make sure that payload will be below 2KB - for now cutting after 512 chars
			if (body.length() > 512) {
				body = body.substring(0, 500) + "...";
			}
			//builder.body(body);
			builder.customField("body", body);
		});

		return builder;
	}

	protected ApnsPayload.Builder prepareEncryptedPayload(ApnsPayload.Builder builder, IEncryptedNotification notification) {
		if (notification.getPriority() != INotification.Priority.high) {
			builder.instantDeliveryOrSilentNotification();
		}
		builder.customField("account", notification.getAccount().toString())
				.customField("encrypted", notification.getEncrypted()).customField("iv", notification.getIV());

		return builder;
	}

	protected void unregisterDevice(String deviceId) {
		try {
			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST,"unregistering device " + deviceId + " as it was reported as inactive by APNs");
			}
			Stream<IPushSettings> settingsStream = repository.getNodeSettings(getName(), deviceId);
			settingsStream.forEach(settings -> {
				// pushSettings is not equal to settings!
				try {
					IPushSettings pushSettings = repository.unregisterDevice(settings.getServiceJid(),
																			 settings.getOwnerJid(), getName(),
																			 deviceId);
					if (pushSettings != null && (pushSettings.getDevices().isEmpty() || pushSettings.getVersion() > 0)) {
						affiliationChangedModule.notifyAffiliationChanged(pushSettings.getServiceJid(),
																		  pushSettings.getOwnerJid(),
																		  pushSettings.getNode(), Affiliation.none);
					}
				} catch (ComponentException ex) {
					log.log(Level.FINER, getName() + ", failed to unregister device = " + deviceId, ex);
				} catch (RepositoryException ex) {
					log.log(Level.WARNING, getName() + ", failed to unregister device = " + deviceId, ex);
				}
			});
		} catch (RepositoryException ex) {
			log.log(Level.WARNING, getName() + ", failed to unregister device = " + deviceId, ex);
		}
	}

	private void setService(ApnsService service) {
		ApnsService oldService = this.service;
		this.service = service;
		if (oldService != null) {
			oldService.stop();
		}
	}

	private void setSandboxService(ApnsService sandboxService) {
		ApnsService oldService = this.sandboxService;
		this.sandboxService = sandboxService;
		if (oldService != null) {
			oldService.stop();
		}
	}

	private void setPushKitService(ApnsService service) {
		ApnsService oldService = pushkitService;
		this.pushkitService = service;
		if (oldService != null) {
			oldService.stop();
		}
	}

	public void setUserRepository(UserRepository userRepository) {
		super.setUserRepository(userRepository);
		try {
			reloadAPNSCertificateFromRepository();
		} catch (TigaseDBException e) {
			log.log(Level.WARNING, "Reloading APNS certificate failed", e);
		}
	}

	private void setPushKitSandboxService(ApnsService sandboxService) {
		ApnsService oldService = this.pushkitSandboxService;
		this.pushkitSandboxService = sandboxService;
		if (oldService != null) {
			oldService.stop();
		}
	}

	// TODO: Rename in future releases as APNSSecretsChangedEvent
	public static class APNSCertificateChangedEvent
			implements Serializable {

		private String componentId;
		private String name;

		public APNSCertificateChangedEvent() {}

		public APNSCertificateChangedEvent(String componentId, String name) {
			this.componentId = componentId;
			this.name = name;
		}
	}
}
