/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.apns;

import tigase.push.monitor.SSLCertificateExpirationAware;
import tigase.util.Base64;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.*;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class APNSUtil {
	
	public static KeyStore loadCertificate(InputStream is, String certPass) throws IOException {
		try {
			KeyStore keyStore = KeyStore.getInstance("PKCS12");
			keyStore.load(is, certPass.toCharArray());
			return keyStore;
		} catch (Exception ex) {
			throw new IOException("Could not load key store", ex);
		}
	}

	public static PrivateKey loadPrivateKey(String data) throws InvalidKeySpecException, NoSuchAlgorithmException {
		String base64Str = Arrays.stream(data.replace("\r\n", "\n").split("\n"))
				.filter(line -> !line.startsWith("-----"))
				.collect(Collectors.joining());
		return loadPrivateKey(Base64.decode(base64Str));
	}

	public static PrivateKey loadPrivateKey(byte[] data) throws NoSuchAlgorithmException, InvalidKeySpecException {
		PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(data);
		KeyFactory keyFactory = KeyFactory.getInstance("EC");
		return keyFactory.generatePrivate(spec);
	}

	public static InputStream inputStreamFromBase64(String base64) {
		final byte[] certificate = Base64.decode(base64);
		return new ByteArrayInputStream(certificate);
	}

	public static Stream<SSLCertificateExpirationAware.Result> getCertificateValidPeriodFromBase64(String base64Certificate, String password) throws IOException {
		KeyStore keyStore = loadCertificate(inputStreamFromBase64(base64Certificate), password);
		return getCertificateValidPeriod(keyStore);
	}
	
	public static Stream<SSLCertificateExpirationAware.Result> getCertificateValidPeriodFromFile(String path, String password) throws IOException {
		try (FileInputStream fis = new FileInputStream(path)) {
			KeyStore keyStore = loadCertificate(fis, password);
			return getCertificateValidPeriod(keyStore);
		}
	}
	
	public static Stream<SSLCertificateExpirationAware.Result> getCertificateValidPeriod(KeyStore keyStore) throws IOException {
		try {
			List<SSLCertificateExpirationAware.Result> results = new ArrayList<>();
			Enumeration<String> enumeration = keyStore.aliases();
			while (enumeration.hasMoreElements()) {
				String alias = enumeration.nextElement();
				Certificate certificate = keyStore.getCertificate(alias);
				if (certificate instanceof X509Certificate) {
					X509Certificate x509Certificate = (X509Certificate) certificate;
					results.add(new SSLCertificateExpirationAware.Result(x509Certificate.getSubjectDN().toString(), x509Certificate.getNotBefore().toInstant().atZone(
							ZoneId.systemDefault()).toLocalDateTime(), x509Certificate.getNotAfter().toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime()));
				}
			}
			return results.stream();
		} catch (KeyStoreException ex) {
			throw new IOException("Could not validate key store", ex);
		}
	}
	
}
