/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.adhoc;

import tigase.component.adhoc.AdHocCommandException;
import tigase.component.adhoc.AdHocResponse;
import tigase.component.adhoc.AdhHocRequest;
import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.form.Field;
import tigase.form.Form;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.pubsub.Affiliation;
import tigase.push.PushNotificationsComponent;
import tigase.push.api.IPushProvider;
import tigase.push.api.IPushRepository;
import tigase.push.api.IPushSettings;
import tigase.push.modules.AffiliationChangedModule;
import tigase.xmpp.Authorization;
import tigase.xmpp.jid.JID;

import java.util.Collections;
import java.util.List;

/**
 * Created by andrzej on 02.01.2017.
 */
@Bean(name = "unregister-device", parent = PushNotificationsComponent.class, active = true)
public class UnregisterDevice
		extends AbstractAdHocCommand {

	@Inject
	private AffiliationChangedModule affiliationChangedModule;

	@Inject(nullAllowed = true)
	private List<IPushProvider> pushProviders;

	@Inject
	private IPushRepository repository;

	@Override
	public String getName() {
		return "Unregister device";
	}

	@Override
	public String getNode() {
		return "unregister-device";
	}

	@Override
	public boolean isAllowedFor(JID jid) {
		return true;
	}

	public void setPushProviders(List<IPushProvider> pushProviders) {
		if (pushProviders == null) {
			pushProviders = Collections.emptyList();
		}
		this.pushProviders = pushProviders;
	}

	@Override
	protected Form prepareForm(AdhHocRequest request, AdHocResponse response) throws AdHocCommandException {
		Form form = new Form("result", "Unregister device",
							 "Use this form to unregister device from push notifications");

		String[] labels = new String[pushProviders.size()];
		String[] options = new String[pushProviders.size()];
		for (int i = 0; i < pushProviders.size(); i++) {
			IPushProvider pushProvider = pushProviders.get(i);
			options[i] = pushProvider.getName();
			labels[i] = pushProvider.getDescription();
		}

		form.addField(Field.fieldListSingle("provider", "", "Provider", labels, options));
		form.addField(Field.fieldTextSingle("device-token", "", "Device token"));

		return form;
	}

	@Override
	protected Form submitForm(AdhHocRequest request, AdHocResponse response, Form form) throws AdHocCommandException {
		String providerName = form.getAsString("provider");
		String deviceToken = form.getAsString("device-token");
		if (providerName == null || deviceToken == null) {
			throw new AdHocCommandException(Authorization.BAD_REQUEST, "Provider and device token must not be empty");
		}

		if (!pushProviders.stream().filter(provider -> provider.getName().equals(providerName)).findAny().isPresent()) {
			throw new AdHocCommandException(Authorization.BAD_REQUEST, "Unsupported push provider");
		}

		try {
			IPushSettings pushSettings = repository.unregisterDevice(request.getRecipient().getBareJID(),
																	 request.getSender().getBareJID(), providerName,
																	 deviceToken);

			if (pushSettings.getDevices().isEmpty() || pushSettings.getVersion() > 0) {
				affiliationChangedModule.notifyAffiliationChanged(pushSettings.getServiceJid(),
																  pushSettings.getOwnerJid(), pushSettings.getNode(),
																  Affiliation.none);
			}

			Form responseForm = new Form("result", "Device unregistered",
										 "Device " + deviceToken + " was successfully unregistered");

			response.completeSession();
			return responseForm;
		} catch (ComponentException ex) {
			throw new AdHocCommandException(ex.getErrorCondition(), ex.getMessage());
		} catch (RepositoryException ex) {
			throw new RuntimeException(ex);
		}

	}
}
