/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.adhoc;

import tigase.component.adhoc.AdHocCommandException;
import tigase.component.adhoc.AdHocResponse;
import tigase.component.adhoc.AdhHocRequest;
import tigase.component.exceptions.RepositoryException;
import tigase.form.Field;
import tigase.form.Form;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.push.PushNotificationsComponent;
import tigase.push.api.IPushProvider;
import tigase.push.apns.APNSUtil;
import tigase.push.apns.APNsBinaryApiProvider;
import tigase.xmpp.Authorization;
import tigase.xmpp.jid.JID;

import java.util.List;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Created by andrzej on 02.01.2017.
 */
@Bean(name = "set-apns-encryption-key", parent = PushNotificationsComponent.class, active = true)
public class SetAPNSEncryptionKey
		extends AbstractAdHocCommand {

	private static final Logger log = Logger.getLogger(SetAPNSEncryptionKey.class.getCanonicalName());

	@Inject
	PushNotificationsComponent component;

	@Inject(nullAllowed = true)
	private List<APNsBinaryApiProvider> pushProviders;

	@Override
	public String getName() {
		return "Set APNS encryption key";
	}

	@Override
	public String getNode() {
		return "set-apns-encryption-key";
	}

	@Override
	public boolean isAllowedFor(JID jid) {
		return component.isAdmin(jid);
	}
	
	@Override
	protected Form prepareForm(AdhHocRequest request, AdHocResponse response) throws AdHocCommandException {
		Form form = new Form("form", "Select APNS provider to set APNS encryption key",
							 "Use this form to set APNS encryption key for push provider");

		String[] labels = new String[pushProviders.size()];
		String[] optionsValues = new String[pushProviders.size()];
		for (int i = 0; i < pushProviders.size(); i++) {
			IPushProvider pushProvider = pushProviders.get(i);
			labels[i] = pushProvider.getDescription() + ": " + pushProvider.getName();
			optionsValues[i] = pushProvider.getName();
		}

		form.addField(Field.fieldListSingle("provider", "", "Provider", labels, optionsValues));
		form.addField(Field.fieldTextSingle("apns-key-id", "", "APNS Key ID"));
		form.addField(Field.fieldTextMulti("apns-key", "", "APNS Key"));
		form.addField(Field.fieldTextSingle("apns-team-id", "", "APNS Team ID"));

		return form;
	}

	@Override
	protected Form submitForm(AdhHocRequest request, AdHocResponse response, Form form) throws AdHocCommandException {
		String providerName = form.getAsString("provider");
		String keyId = Optional.ofNullable(form.getAsString("apns-key-id")).map(String::trim).orElse("");
		String key = Optional.ofNullable(form.getAsStrings("apns-key"))
				.map(lines -> String.join("\n", lines))
				.orElse("");
		String teamId = Optional.ofNullable(form.getAsString("apns-team-id")).map(String::trim).orElse("");
		if (providerName == null || keyId.isBlank() || key.isBlank() || teamId.isBlank()) {
			throw new AdHocCommandException(Authorization.BAD_REQUEST,
											"Provider, Key ID, Key, and Team ID must not be empty and key must be a valid p8 encoded private key.");
		}
		try {
			APNSUtil.loadPrivateKey(key);
		} catch (Exception e) {
			log.log(Level.FINEST, "Invalid ad-hoc payload", e);
			throw new AdHocCommandException(Authorization.BAD_REQUEST, "Key is not a valid p8 encoded private key!");
		}

		APNsBinaryApiProvider provider = pushProviders.stream()
				.filter(providerx -> providerx.getName().equals(providerName))
				.findAny()
				.orElseThrow(() -> new AdHocCommandException(Authorization.BAD_REQUEST, "Unsupported push provider"));

		try {
			provider.setAPNSEncryptionKey(keyId, key, teamId);

			Form responseForm = new Form("result", "APNS encryption key updated", "APNS encryption key updated");
			responseForm.addField(Field.fieldFixed("APNS encryption key updated"));
			response.completeSession();
			return responseForm;
		} catch (RepositoryException ex) {
			throw new RuntimeException(ex);
		}
	}
}
