/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.adhoc;

import tigase.component.adhoc.AdHocCommandException;
import tigase.component.adhoc.AdHocResponse;
import tigase.component.adhoc.AdhHocRequest;
import tigase.component.exceptions.RepositoryException;
import tigase.form.Field;
import tigase.form.Form;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.push.PushNotificationsComponent;
import tigase.push.api.IPushProvider;
import tigase.push.apns.APNsBinaryApiProvider;
import tigase.util.Base64;
import tigase.xmpp.Authorization;
import tigase.xmpp.jid.JID;

import java.util.List;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Created by andrzej on 02.01.2017.
 */
@Bean(name = "set-apns-certificate", parent = PushNotificationsComponent.class, active = true)
public class SetAPNSCertificate
		extends AbstractAdHocCommand {

	private static final Logger log = Logger.getLogger(SetAPNSCertificate.class.getCanonicalName());

	@Inject
	PushNotificationsComponent component;

	@Inject(nullAllowed = true)
	private List<APNsBinaryApiProvider> pushProviders;

	@Override
	public String getName() {
		return "Set APNS certificate";
	}

	@Override
	public String getNode() {
		return "set-apns-certificate";
	}

	@Override
	public boolean isAllowedFor(JID jid) {
		return component.isAdmin(jid);
	}

//	public void setPushProviders(List<IPushProvider> pushProviders) {
//		if (pushProviders == null) {
//			pushProviders = Collections.emptyList();
//		}
//		this.pushProviders = pushProviders.stream()
//				.filter(APNsBinaryApiProvider.class::isInstance)
//				.collect(Collectors.toList());
//	}

	@Override
	protected Form prepareForm(AdhHocRequest request, AdHocResponse response) throws AdHocCommandException {
		Form form = new Form("form", "Select APNS provider to set APNS certificate",
							 "Use this form to set APNS certificate for push provider");

		String[] labels = new String[pushProviders.size()];
		String[] optionsValues = new String[pushProviders.size()];
		for (int i = 0; i < pushProviders.size(); i++) {
			IPushProvider pushProvider = pushProviders.get(i);
			labels[i] = pushProvider.getDescription() + ": " + pushProvider.getName();
			optionsValues[i] = pushProvider.getName();
		}

		form.addField(Field.fieldListSingle("provider", "", "Provider", labels, optionsValues));
		form.addField(Field.fieldTextMulti("apns-certificate", "", "APNS certificate"));
		form.addField(Field.fieldTextMulti("apns-pushkit-certificate", "", "APNS pushkit certificate"));
		form.addField(Field.fieldTextPrivate("apns-certificate-password", "", "APNS certificate password"));

		return form;
	}

	@Override
	protected Form submitForm(AdhHocRequest request, AdHocResponse response, Form form) throws AdHocCommandException {
		String providerName = form.getAsString("provider");
		String apnsCertificate = form.getAsStrings("apns-certificate") == null
								 ? null
								 : String.join("", form.getAsStrings("apns-certificate"));
		String pushkitCertificate = form.getAsStrings("apns-pushkit-certificate") == null
									? null
									: String.join("", form.getAsStrings("apns-pushkit-certificate"));
		String apnsCertificatePassword = form.getAsString("apns-certificate-password");
		if (providerName == null || apnsCertificate == null || apnsCertificate.isEmpty() ||
				pushkitCertificate == null || pushkitCertificate.isEmpty() || apnsCertificatePassword == null ||
				apnsCertificatePassword.isEmpty()) {
			throw new AdHocCommandException(Authorization.BAD_REQUEST,
											"Provider and certificate must not be empty and certificates must be encoded as valid Base64");
		}
		try {
			Base64.decode(apnsCertificate);
			Base64.decode(pushkitCertificate);
		} catch (Exception e) {
			log.log(Level.FINEST, "Invalid ad-hoc payload", e);
			throw new AdHocCommandException(Authorization.BAD_REQUEST, "Certificates must be encoded as valid Base64");
		}

		Optional<APNsBinaryApiProvider> provider = pushProviders.stream()
				.filter(providerx -> providerx.getName().equals(providerName))
				.findAny();

		if (provider.isEmpty()) {
			throw new AdHocCommandException(Authorization.BAD_REQUEST, "Unsupported push provider");
		}

		try {
			provider.get().setAPNSCertificate(apnsCertificate, pushkitCertificate, apnsCertificatePassword);

			Form responseForm = new Form("result", "APNS certificate updated", "APNS certificate updated");
			responseForm.addField(Field.fieldFixed("APNS certificate updated"));
			response.completeSession();
			return responseForm;
		} catch (RepositoryException ex) {
			throw new RuntimeException(ex);
		}
	}
}
