/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push.adhoc;

import tigase.component.adhoc.AdHocCommandException;
import tigase.component.adhoc.AdHocResponse;
import tigase.component.adhoc.AdhHocRequest;
import tigase.component.exceptions.RepositoryException;
import tigase.form.Field;
import tigase.form.Form;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.pubsub.Affiliation;
import tigase.push.PushNotificationsComponent;
import tigase.push.api.IPushProvider;
import tigase.push.api.IPushRepository;
import tigase.push.api.IPushSettings;
import tigase.push.modules.AffiliationChangedModule;
import tigase.xmpp.Authorization;
import tigase.xmpp.jid.JID;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

/**
 * Created by andrzej on 02.01.2017.
 */
@Bean(name = "register-device", parent = PushNotificationsComponent.class, active = true)
public class RegisterDevice
		extends AbstractAdHocCommand {

	private static final Logger log = Logger.getLogger(RegisterDevice.class.getCanonicalName());

	@Inject
	private AffiliationChangedModule affiliationChangedModule;

	@Inject(nullAllowed = true)
	private List<IPushProvider> pushProviders;

	@Inject
	private IPushRepository repository;

	@Override
	public String getName() {
		return "Register device";
	}

	@Override
	public String getNode() {
		return "register-device";
	}

	@Override
	public boolean isAllowedFor(JID jid) {
		return true;
	}

	public void setPushProviders(List<IPushProvider> pushProviders) {
		if (pushProviders == null) {
			pushProviders = Collections.emptyList();
		}
		this.pushProviders = pushProviders;
	}

	@Override
	protected Form prepareForm(AdhHocRequest request, AdHocResponse response) throws AdHocCommandException {
		Form form = new Form("result", "Register device", "Use this form to register device for push notifications");

		String[] labels = new String[pushProviders.size()];
		String[] options = new String[pushProviders.size()];
		for (int i = 0; i < pushProviders.size(); i++) {
			IPushProvider pushProvider = pushProviders.get(i);
			options[i] = pushProvider.getName();
			labels[i] = pushProvider.getDescription();
		}

		form.addField(Field.fieldListSingle("provider", "", "Provider", labels, options));
		form.addField(Field.fieldTextSingle("device-token", "", "Device token"));

		return form;
	}

	@Override
	protected Form submitForm(AdhHocRequest request, AdHocResponse response, Form form) throws AdHocCommandException {
		String providerName = form.getAsString("provider");
		String deviceToken = form.getAsString("device-token");
		if (providerName == null || deviceToken == null) {
			throw new AdHocCommandException(Authorization.BAD_REQUEST, "Provider and device token must not be empty");
		}

		Optional<IPushProvider> provider = pushProviders.stream().filter(providerx -> providerx.getName().equals(providerName)).findAny();
		if (!provider.isPresent()) {
			throw new AdHocCommandException(Authorization.BAD_REQUEST, "Unsupported push provider");
		}

		String deviceSecondToken = form.getAsString("device-second-token");

		try {
			Stream<IPushSettings> pushSettingsStream = repository.getNodeSettings(request.getSender().getBareJID(), providerName,
																			deviceToken);

			Optional<IPushSettings> pushSettingsOptional = pushSettingsStream.filter(pushSettings -> {
				if (pushSettings.getVersion() == 0) {
					try {
						IPushSettings old = repository.unregisterDevice(pushSettings.getServiceJid(), pushSettings.getOwnerJid(),
													providerName, deviceToken);
						if (old != null && old.getDevices().isEmpty()) {
							affiliationChangedModule.notifyAffiliationChanged(pushSettings.getServiceJid(),
																			  pushSettings.getOwnerJid(),
																			  pushSettings.getNode(), Affiliation.none);
						}
					} catch (Throwable ex) {
						log.log(Level.FINE, "Could not unregister device");
					}
					return false;
				}
				if (pushSettings.getDevices().stream().noneMatch(device -> Objects.equals(device.getDeviceId(), deviceToken) && Objects.equals(device.getDeviceSecondId(), deviceSecondToken))) {
					return false;
				}
				return true;
			}).findFirst();

			IPushSettings pushSettings = pushSettingsOptional.orElseGet(() -> {
				try {
					return repository.registerDevice(request.getRecipient().getBareJID(), request.getSender().getBareJID(), providerName, deviceToken, deviceSecondToken);
				} catch (RepositoryException ex) {
					throw new RuntimeException(ex);
				}
			});

			Form responseForm = new Form("result", "Device registered",
										 "Use data from form to set notifications to push service");
			responseForm.addField(
					Field.fieldTextSingle("node", pushSettings.getNode(), "Node to publish notifications to"));

			provider.get().maxPayloadSize().ifPresent(maxPayloadSize -> {
				responseForm.addField(Field.fieldTextSingle("max-payload-size", String.valueOf(maxPayloadSize), "Maximal payload size"));
			});

			responseForm.addField(Field.fieldTextMulti("features", provider.get()
					.supportedFeatures()
					.stream()
					.map(IPushProvider.Feature::getXMLNS)
					.toArray(String[]::new), "Supported features"));

			response.completeSession();
			return responseForm;
		} catch (RepositoryException ex) {
			throw new RuntimeException(ex);
		}
	}
}
