/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push;

import tigase.component.AbstractKernelBasedComponent;
import tigase.component.ComponenScriptCommandProcessor;
import tigase.component.adhoc.AdHocCommandManager;
import tigase.component.exceptions.RepositoryException;
import tigase.component.modules.impl.AdHocCommandModule;
import tigase.component.modules.impl.DiscoveryModule;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.kernel.beans.selector.ConfigType;
import tigase.kernel.beans.selector.ConfigTypeEnum;
import tigase.kernel.core.Kernel;
import tigase.licence.LicenceChecker;
import tigase.licence.LicenceCheckerUpdateCallback;
import tigase.licence.LicenceCheckerUpdateCallbackImpl;
import tigase.push.api.IPushProvider;
import tigase.push.api.IPushRepository;
import tigase.push.monitor.SSLCertificateExpirationAware;
import tigase.server.Packet;
import tigase.stats.ComponentStatisticsProvider;
import tigase.stats.StatisticsList;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;
import java.util.stream.Stream;

/**
 * Created by andrzej on 30.12.2016.
 */
@Bean(name = "push", parent = Kernel.class, active = false, exportable = true)
@ConfigType(ConfigTypeEnum.DefaultMode)
public class PushNotificationsComponent
		extends AbstractKernelBasedComponent implements SSLCertificateExpirationAware {

	public static final String COMPONENT_ID = "push";

	private LicenceChecker licenceChecker;

	@Inject
	private IPushRepository repository;

	@Inject(nullAllowed = true)
	private List<IPushProvider> providers;

	@Override
	public synchronized void everyHour() {
		super.everyHour();
		for (IPushProvider provider : providers) {
			if (provider instanceof ComponentStatisticsProvider) {
				((ComponentStatisticsProvider) provider).everyHour();
			}
		}
	}

	@Override
	public synchronized void everyMinute() {
		super.everyMinute();
		for (IPushProvider provider : providers) {
			if (provider instanceof ComponentStatisticsProvider) {
				((ComponentStatisticsProvider) provider).everyMinute();
			}
		}
	}

	@Override
	public synchronized void everySecond() {
		super.everySecond();
		for (IPushProvider provider : providers) {
			if (provider instanceof ComponentStatisticsProvider) {
				((ComponentStatisticsProvider) provider).everySecond();
			}
		}
	}

	@Override
	public String getComponentVersion() {
		String version = this.getClass().getPackage().getImplementationVersion();
		return version == null ? "0.0.0" : version;
	}

	public String getDiscoCategory() {
		return "pubsub";
	}

	public String getDiscoCategoryType() {
		return "push";
	}

	public String getDiscoDescription() {
		return "Push Notifications component";
	}

	@Override
	public int hashCodeForPacket(Packet packet) {
		if (packet.getStanzaFrom() != null) {
			return packet.getStanzaFrom().getBareJID().hashCode();
		}
		return super.hashCodeForPacket(packet);
	}

	@Override
	public void initialize() {
		super.initialize();

		LicenceCheckerUpdateCallback upCall = new LicenceCheckerUpdateCallbackImpl(COMPONENT_ID) {
			@Override
			public String getMissingLicenseWarning() {
				return "\n" +
						"This installation contains Tigase Push Notifications component package, not an open source software.\n" +
						"The Tigase Push Notifications component is only available under a commercial license.\n" +
						"The full text of the commercial license agreement is available upon request.\n" +
						"\n" +
						"The Tigase Push Notifications component is provided free of charge for testing and\n" +
						"development purposes only. Any use of the component on production systems,\n" +
						"either commercial or not-for-profit, requires the purchase of a license.\n" +
						"\n" +
						"If the Tigase Push Notifications component is activated without a valid license, it will\n" +
						"continue to work, including its full set of features, but it will send\n" +
						"certain statistical information to Tigase's servers on a regular basis.\n" +
						"If the Tigase Push Notifications component cannot access our servers to send information,\n" +
						"it will stop working. Once a valid license is installed, the Tigase Push Notifications \n" +
						"component will stop sending statistical information to Tigase's servers.\n" +
						"\n" +
						"By activating the Tigase Push Notifications component without a valid license you agree\n" +
						"and accept that the component will send certain statistical information\n" +
						"(such as DNS domain names, vhost names, number of online users, number of\n" +
						"cluster nodes, etc.) which may be considered confidential and proprietary\n" +
						"by the user. You accept and confirm that such information, which may be\n" +
						"considered confidential or proprietary, will be transferred to Tigase's\n" +
						"servers and that you will not pursue any remedy at law as a result of the\n" +
						"information transfer.\n" +
						"If the Tigase Push Notifications component is installed but not activated, no statistical\n" +
						"information will be sent to Tigase's servers.";
			}
		};
		licenceChecker = LicenceChecker.getLicenceChecker(COMPONENT_ID, upCall);
	}

	@Override
	public boolean isDiscoNonAdmin() {
		return true;
	}

	@Override
	public boolean isSubdomain() {
		return true;
	}

	@Override
	public int processingInThreads() {
		return Runtime.getRuntime().availableProcessors() * 4;
	}

	@Override
	protected void registerModules(Kernel kernel) {
		kernel.registerBean(DiscoveryModule.class).exec();
		kernel.registerBean(ComponenScriptCommandProcessor.class).exec();
		kernel.registerBean(AdHocCommandManager.class).exec();
		kernel.registerBean(AdHocCommandModule.class).exec();
	}

	@Override
	public void getStatistics(StatisticsList list) {
		super.getStatistics(list);
		try {
			Map<String, IPushRepository.Statistics> statisticsMap = repository.getStatistics();
			for (Map.Entry<String, IPushRepository.Statistics> e : statisticsMap.entrySet()) {
				for (String name : e.getValue().getCounterNames()) {
					list.add(getName() + "/repository/" + e.getKey(), name, e.getValue().getCounterValue(name),
							 Level.FINEST);
				}
			}
		} catch (RepositoryException ex) {
			if (log.isLoggable(Level.FINER)) {
				log.log(Level.FINER, "Could not retrieve statistics", ex);
			}
		}
		for (IPushProvider provider : providers) {
			if (provider instanceof ComponentStatisticsProvider) {
				((ComponentStatisticsProvider) provider).getStatistics(getName() + "/provider", list);
			}
		}
	}

	public void setProviders(List<IPushProvider> providers) {
		this.providers = Optional.ofNullable(providers).orElseGet(Collections::emptyList);
	}

	@Override
	public Stream<Result> getSSLCertificatesValidPeriod() {
		return providers.stream().filter(SSLCertificateExpirationAware.class::isInstance).flatMap(provider -> {
			return ((SSLCertificateExpirationAware) provider).getSSLCertificatesValidPeriod();
		});
	}

}
