/*
 * Tigase Push - Push notifications component for Tigase
 * Copyright (C) 2017 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.push;

import tigase.db.TigaseDBException;
import tigase.db.UserRepository;
import tigase.kernel.beans.Inject;
import tigase.kernel.beans.config.ConfigField;
import tigase.push.api.IPushProvider;
import tigase.xmpp.jid.BareJID;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;

import static tigase.push.PushNotificationsComponent.COMPONENT_ID;

public abstract class AbstractProvider implements IPushProvider {

	private static final Logger logger = Logger.getLogger(AbstractProvider.class.getName());
	@Inject
	private UserRepository userRepository;
	private boolean userRepositoryInitialized = false;

	@ConfigField(desc = "Provider name")
	private String name;
	private final BareJID repoUser = BareJID.bareJIDInstanceNS(COMPONENT_ID);
	
	protected AbstractProvider(String name) {
		this.name = name;
	}

	@Override
	public String getName() {
		return name;
	}

	public void setUserRepository(UserRepository userRepository) {
		this.userRepository = userRepository;
		userRepositoryInitialized = false;
	}

	private void ensureRepoInitialized() throws TigaseDBException {
		if (!userRepositoryInitialized) {
			Objects.requireNonNull(userRepository);
			if (!userRepository.userExists(repoUser)) {
				try {
					userRepository.addUser(repoUser);
					userRepositoryInitialized = true;
				} catch (TigaseDBException ex) {
					if (!userRepository.userExists(repoUser)) {
						logger.log(Level.WARNING, "failed to initialize data repository", ex);
					} else {
						userRepositoryInitialized = true;
					}
				}
			}
		}
	}

	protected String getData(String key) throws TigaseDBException {
		ensureRepoInitialized();
		return userRepository.getData(repoUser, name, key);
	}

	protected Map<String,String> getDataMap(String... keys) throws TigaseDBException {
		ensureRepoInitialized();
		Map<String,String> result = new HashMap<>();
		for (String key : keys) {
			String value = userRepository.getData(repoUser, name, key);
			if (value != null) {
				result.put(key, value);
			}
		}
		return result;
	}

	protected void setData(String key, String value) throws TigaseDBException {
		ensureRepoInitialized();
		userRepository.setData(repoUser, name, key, value);
	}

	protected void removeData(String... keys) throws TigaseDBException {
		ensureRepoInitialized();
		for (String key : keys) {
			userRepository.removeData(repoUser, name, key);
		}
	}
}
