/*
 * JDBCMessageArchiveRepository.java
 *
 * Tigase Jabber/XMPP Server
 * Copyright (C) 2004-2014 "Tigase, Inc." <office@tigase.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 *
 */
package tigase.archive.db;

//~--- non-JDK imports --------------------------------------------------------

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import tigase.archive.AbstractCriteria;
import tigase.archive.RSM;
import tigase.archive.db.JDBCMessageArchiveRepository.Criteria;
import tigase.db.DBInitException;
import tigase.db.DataRepository;
import static tigase.db.DataRepository.dbTypes.derby;
import tigase.db.Repository;
import tigase.db.RepositoryFactory;
import tigase.db.TigaseDBException;
import tigase.xml.DomBuilderHandler;
import tigase.xml.Element;
import tigase.xml.SimpleParser;
import tigase.xml.SingletonFactory;
import tigase.xmpp.BareJID;

/**
 * Class description
 *
 *
 * @version        Enter version here..., 13/02/16
 * @author         Enter your name here...
 */
@Repository.Meta( supportedUris = { "jdbc:[^:]+:.*" } )
public class JDBCMessageArchiveRepository extends AbstractMessageArchiveRepository<Criteria> {
	private static final String JIDS_ID  = "jid_id";
	private static final String JIDS_JID = "jid";

	// jids table
	private static final String JIDS_TABLE = "tig_ma_jids";
	private static final Logger log        =
		Logger.getLogger(JDBCMessageArchiveRepository.class.getCanonicalName());
	private static final long LONG_NULL              = 0;
	private static final long MILIS_PER_DAY          = 24 * 60 * 60 * 1000;
	
	private static final String[] MSG_BODY_PATH = { "message", "body" };
	
	private static final String MSGS_ID	        = "msg_id";
	private static final String MSGS_BUDDY_ID  = "buddy_id";
	private static final String MSGS_BODY      = "body";
	private static final String MSGS_DIRECTION = "direction";
	private static final String MSGS_MSG       = "msg";
	private static final String MSGS_OWNER_ID  = "owner_id";

//+ "create unique index " + JIDS_TABLE + "_" + JIDS_JID + " on "
//+ JIDS_TABLE + " ( " + JIDS_JID + "(765));";
	// messages table
	private static final String MSGS_TABLE        = "tig_ma_msgs";
	private static final String MSGS_TIMESTAMP    = "ts";
	private static final String MSGS_TYPE         = "type";
	
	private static final String TAGS_TABLE        = "tig_ma_tags";
	private static final String TAGS_ID           = "tag_id";
	private static final String TAGS_TAG          = "tag";
	private static final String TAGS_OWNER_ID      = "owner_id";
	
	private static final String MSGS_TAGS_TABLE   = "tig_ma_msgs_tags";
	
	private static final SimpleParser parser      = SingletonFactory.getParserInstance();
	private static final String GET_JID_IDS_QUERY = "select " + JIDS_JID + ", " + JIDS_ID +
																									" from " + JIDS_TABLE + " where " +
																									JIDS_JID + " = ?" + " or " + JIDS_JID +
																									" = ?";
	private static final String GET_JID_ID_QUERY = "select " + JIDS_JID + ", " + JIDS_ID +
																								 " from " + JIDS_TABLE + " where " +
																								 JIDS_JID + " = ?";
	private static final String ADD_JID_QUERY = "insert into " + JIDS_TABLE + " (" +
																							JIDS_JID + ") values (?)";
	private static final String DERBY_CREATE_JIDS = "create table " + JIDS_TABLE + " ( "
																									+ JIDS_ID + " bigint generated by default as identity not null, " + JIDS_JID
																									+ " varchar(2049), primary key ( " + JIDS_ID + " ));"
																									+ "create unique index " + JIDS_TABLE + "_" + JIDS_JID + " on " + JIDS_TABLE
																									+ " (" + JIDS_JID + ");";
	private static final String PGSQL_CREATE_JIDS = "create table " + JIDS_TABLE + " ( "
																									+ JIDS_ID + " bigserial, " + JIDS_JID
																									+ " varchar(2049), primary key ("
																									+ JIDS_ID + ")); "
																									+ "create unique index " + JIDS_TABLE
																									+ "_" + JIDS_JID + " on " + JIDS_TABLE
																									+ " ( " + JIDS_JID + ");";
	private static final String SQLSERVER_CREATE_JIDS = "create table " + JIDS_TABLE + " ( "
																									+ JIDS_ID + " bigint identity(1,1), " + JIDS_JID
																									+ " nvarchar(2049),"
																									+ JIDS_JID + "_fragment as left (" + JIDS_JID + ", 765),"
																									+ "primary key (" + JIDS_ID + ")); "
																									+ "create unique index " + JIDS_TABLE
																									+ "_" + JIDS_JID + " on " + JIDS_TABLE
																									+ " ( " + JIDS_JID + "_fragment " + ");";
	private static final String MYSQL_CREATE_JIDS =
															"create table " + JIDS_TABLE + " ( " + JIDS_ID
															+ " bigint unsigned NOT NULL auto_increment, " + JIDS_JID
															+ " varchar(2049), primary key (" + JIDS_ID + ")); ";
	private static final String GENERIC_GET_MESSAGES_START = "select m." + MSGS_MSG + ", m." + MSGS_TIMESTAMP + ", m." + MSGS_DIRECTION 
			+ " from " + MSGS_TABLE + " m where m." + MSGS_OWNER_ID + " = ? ";
	private static final String MSSQL2008_GET_MESSAGES_START = "select x." + MSGS_MSG + ", x." + MSGS_TIMESTAMP + ", x." + MSGS_DIRECTION + " FROM ( "
			+ "select m." + MSGS_MSG + ", m." + MSGS_TIMESTAMP + ", m." + MSGS_DIRECTION + ", ROW_NUMBER() over (order by m." + MSGS_TIMESTAMP + ") as rn"
			+ " from " + MSGS_TABLE + " m where m." + MSGS_OWNER_ID + " = ? ";
	private static final String GENERIC_GET_MESSAGES_START_WITH = "select m." + MSGS_MSG + ", m." + MSGS_TIMESTAMP + ", m." + MSGS_DIRECTION + ", b." + JIDS_JID
			+ " from " + MSGS_TABLE + " m inner join " + JIDS_TABLE + " b ON b." + JIDS_ID + " = m." + MSGS_BUDDY_ID + " where m." + MSGS_OWNER_ID + " = ? ";
	private static final String MSSQL2008_GET_MESSAGES_START_WITH = "select x." + MSGS_MSG + ", x." + MSGS_TIMESTAMP + ", x." + MSGS_DIRECTION + ", b." + JIDS_JID + " FROM ( "
			+ "select m." + MSGS_MSG + ", m." + MSGS_TIMESTAMP + ", m." + MSGS_DIRECTION + ", ROW_NUMBER() over (order by m." + MSGS_TIMESTAMP + ") as rn,"
			+ " m." + MSGS_BUDDY_ID
			+ " from " + MSGS_TABLE + " m where m." + MSGS_OWNER_ID + " = ? ";
	private static final String MSSQL2008_GET_MESSAGES_END = ") x";
	private static final String MSSQL2008_GET_MESSAGES_END_WITH = ") x inner join " + JIDS_TABLE + " b ON b." + JIDS_ID + " = x." + MSGS_BUDDY_ID;
	private static final String GENERIC_GET_MESSAGES_END = "select " + MSGS_MSG + ", " + MSGS_TIMESTAMP + "," + MSGS_DIRECTION 
			+ " from " + MSGS_TABLE + " where " + MSGS_OWNER_ID + " = ? and " 
			+ MSGS_BUDDY_ID + " = ? and " + MSGS_TIMESTAMP + " >= ? and " + MSGS_TIMESTAMP + " <= ?";
	private static final String GENERIC_GET_MESSAGES_COUNT = "select count(m." + MSGS_TIMESTAMP + ") from " + MSGS_TABLE + " m where m." + MSGS_OWNER_ID 
			+ " = ?";
	private static final String GENERIC_GET_MESSAGES_ORDER_BY = " order by " + MSGS_TIMESTAMP;
	private static final String GENERIC_GET_COLLECTIONS_SELECT = "select min(m." + MSGS_TIMESTAMP + ") as ts, j." + JIDS_JID + " from " + MSGS_TABLE + " m "
			+ "inner join " + JIDS_TABLE + " j on m." + MSGS_BUDDY_ID + " = j." + JIDS_ID + " where m." + MSGS_OWNER_ID + " = ? ";
	private static final String MSSQL2008_GET_COLLECTIONS_SELECT = "select x." + MSGS_TIMESTAMP + ", x." + JIDS_JID + " from ( "
			+ "select min(m." + MSGS_TIMESTAMP + ") as " + MSGS_TIMESTAMP + ", j." + JIDS_JID + ", ROW_NUMBER() over (order by min(m." + MSGS_TIMESTAMP + "), j." + JIDS_JID + ") as rn from " + MSGS_TABLE + " m "
			+ "inner join " + JIDS_TABLE + " j on m." + MSGS_BUDDY_ID + " = j." + JIDS_ID + " where m." + MSGS_OWNER_ID + " = ? ";
	private static final String GENERIC_GET_COLLECTIONS_SELECT_GROUP = " group by date(m." + MSGS_TIMESTAMP + "), m." + MSGS_BUDDY_ID + ", j." + JIDS_JID;
	private static final String MSSQL2008_GET_COLLECTIONS_SELECT_GROUP = " group by cast(m." + MSGS_TIMESTAMP + " as date), m." + MSGS_BUDDY_ID + ", j." + JIDS_JID + ") x ";
	private static final String GENERIC_GET_COLLECTIONS_SELECT_ORDER = " order by min(m." + MSGS_TIMESTAMP + "), j." + JIDS_JID;
	private static final String MSSQL2008_GET_COLLECTIONS_SELECT_ORDER = " order by x." + MSGS_TIMESTAMP + ", x." + JIDS_JID;
	private static final String GENERIC_GET_COLLECTIONS_COUNT = "select count(1) from (select min(m." + MSGS_TIMESTAMP + ") as " + MSGS_TIMESTAMP + ", m." + MSGS_BUDDY_ID + " from " 
			+ MSGS_TABLE + " m where m." + MSGS_OWNER_ID + " = ? ";
	private static final String GENERIC_GET_COLLECTIONS_COUNT_GROUP = "group by date(m." + MSGS_TIMESTAMP + "), m." + MSGS_BUDDY_ID + ") x";
	private static final String MSSQL2008_GET_COLLECTIONS_COUNT_GROUP = "group by cast(m." + MSGS_TIMESTAMP + " as date), m." + MSGS_BUDDY_ID + ") x";
	private static final String GENERIC_LIMIT = " limit ? offset ?";					// limit, offset
	private static final String DERBY_LIMIT = " offset ? rows fetch next ? rows only";	// offset, limit
	private static final String MSSQL2008_LIMIT = " where x.rn > ? and x.rn <= ?";				// offset, limit + offset

	private static final String DERBY_CREATE_TAGS = "create table " + TAGS_TABLE + " ( "
																									+ TAGS_ID + " bigint generated by default as identity not null, " + TAGS_TAG
																									+ " varchar(255), " + TAGS_OWNER_ID + " bigint references  " + JIDS_TABLE + "(" + JIDS_ID + ") on delete cascade, " 
																									+ "primary key ( " + TAGS_ID + " ));"
																									+ "create index " + TAGS_TABLE + "_" + TAGS_OWNER_ID + " on " + TAGS_TABLE
																									+ " (" + TAGS_OWNER_ID + ");"
																									+ "create unique index " + TAGS_TABLE
																									+ "_" + TAGS_TAG + "_" + TAGS_OWNER_ID + " on " + TAGS_TABLE
																									+ " ( " + TAGS_OWNER_ID + "," + TAGS_TAG + ");";
	private static final String PGSQL_CREATE_TAGS = "create table " + TAGS_TABLE + " ( "
																									+ TAGS_ID + " bigserial, " + TAGS_TAG
																									+ " varchar(255), " + TAGS_OWNER_ID + " bigint not null,"
																									+ "	primary key (" + TAGS_ID + "),"
			+ " foreign key (" + TAGS_OWNER_ID + ") references " + JIDS_TABLE + " (" + JIDS_ID + ") on delete cascade ); "
																									+ "create index " + TAGS_TABLE + "_" + TAGS_OWNER_ID + " on " + TAGS_TABLE
																									+ " (" + TAGS_OWNER_ID + ");"
																									+ "create unique index " + TAGS_TABLE
																									+ "_" + TAGS_TAG + "_" + TAGS_OWNER_ID + " on " + TAGS_TABLE
																									+ " ( " + TAGS_OWNER_ID + "," + TAGS_TAG + ");";
	private static final String SQLSERVER_CREATE_TAGS = "create table " + TAGS_TABLE + " ( "
																									+ TAGS_ID + " bigint identity(1,1), " + TAGS_TAG
																									+ " nvarchar(255),"
			+ TAGS_OWNER_ID + " bigint not null,"
																									+ "primary key (" + TAGS_ID + ")," 
			+ " foreign key (" + TAGS_OWNER_ID + ") references " + JIDS_TABLE + "(" + JIDS_ID + ") on delete cascade ); "
																									+ "create index " + TAGS_TABLE + "_" + TAGS_OWNER_ID + " on " + TAGS_TABLE
																									+ " (" + TAGS_OWNER_ID + ");"
																									+ "create unique index " + TAGS_TABLE
																									+ "_" + TAGS_TAG + "_" + TAGS_OWNER_ID + " on " + TAGS_TABLE
																									+ " ( " + TAGS_OWNER_ID + "," + TAGS_TAG + ");";
	private static final String MYSQL_CREATE_TAGS =
															"create table " + TAGS_TABLE + " ( " + TAGS_ID
															+ " bigint unsigned NOT NULL auto_increment, " + TAGS_TAG
															+ " varchar(255), " + TAGS_OWNER_ID + " bigint unsigned not null, " 
															+ " primary key (" + TAGS_ID + "), foreign key (" + TAGS_OWNER_ID + ") references " + JIDS_TABLE + "(" + JIDS_ID + ") on delete cascade,"
															+ " key " + TAGS_TABLE + "_" + TAGS_OWNER_ID + " ( " + TAGS_OWNER_ID + " ), "
															+ " unique key " + TAGS_TABLE + "_" + TAGS_TAG + "_" + TAGS_OWNER_ID + " ( " + TAGS_OWNER_ID + "," + TAGS_TAG + " )"
															+ " ); ";	
	
	private static final String DERBY_CREATE_MSGS_TAGS = "create table " + MSGS_TAGS_TABLE + " ("
			+ MSGS_ID + " bigint not null references " + MSGS_TABLE + "(" + MSGS_ID + ") on delete cascade, " 
			+ TAGS_ID + " bigint not null references " + TAGS_TABLE + "(" + TAGS_ID + ") on delete cascade);"
			+ "create index " + MSGS_TAGS_TABLE + "_" + MSGS_ID + " on " + MSGS_TAGS_TABLE + " (" + MSGS_ID + ");"
			+ "create index " + MSGS_TAGS_TABLE + "_" + TAGS_ID + " on " + MSGS_TAGS_TABLE + " (" + TAGS_ID + ");";
	private static final String PGSQL_CREATE_MSGS_TAGS = "create table " + MSGS_TAGS_TABLE + " ("
			+ MSGS_ID + " bigint not null, " 
			+ TAGS_ID + " bigint not null, "
			+ "foreign key (" + MSGS_ID + ") references " + MSGS_TABLE + "(" + MSGS_ID + ") on delete cascade, "
			+ "foreign key (" + TAGS_ID + ") references " + TAGS_TABLE + "(" + TAGS_ID + ") on delete cascade);"
			+ "create index " + MSGS_TAGS_TABLE + "_" + MSGS_ID + " on " + MSGS_TAGS_TABLE + " (" + MSGS_ID + ");"
			+ "create index " + MSGS_TAGS_TABLE + "_" + TAGS_ID + " on " + MSGS_TAGS_TABLE + " (" + TAGS_ID + ");";	
	private static final String SQLSERVER_CREATE_MSGS_TAGS = "create table " + MSGS_TAGS_TABLE + " ("
			+ MSGS_ID + " bigint not null, " 
			+ TAGS_ID + " bigint not null, "
			+ "foreign key (" + MSGS_ID + ") references " + MSGS_TABLE + "(" + MSGS_ID + ") on delete cascade, "
			+ "foreign key (" + TAGS_ID + ") references " + TAGS_TABLE + "(" + TAGS_ID + ") on delete cascade);"
			+ "create index " + MSGS_TAGS_TABLE + "_" + MSGS_ID + " on " + MSGS_TAGS_TABLE + " (" + MSGS_ID + ");"
			+ "create index " + MSGS_TAGS_TABLE + "_" + TAGS_ID + " on " + MSGS_TAGS_TABLE + " (" + TAGS_ID + ");";		
	private static final String MYSQL_CREATE_MSGS_TAGS = "create table " + MSGS_TAGS_TABLE + " ("
			+ MSGS_ID + " bigint unsigned not null, " 
			+ TAGS_ID + " bigint unsigned not null, "
			+ "foreign key (" + MSGS_ID + ") references " + MSGS_TABLE + "(" + MSGS_ID + ") on delete cascade, "
			+ "foreign key (" + TAGS_ID + ") references " + TAGS_TABLE + "(" + TAGS_ID + ") on delete cascade, "
			+ " key " + MSGS_TAGS_TABLE + "_" + MSGS_ID + " (" + MSGS_ID + "), "
			+ " key " + MSGS_TAGS_TABLE + "_" + TAGS_ID + " (" + TAGS_ID + ") );";		
	
	private static final String[][] GET_COLLECTIONS_WHERES = { 
			{ "FROM", " and m." + MSGS_TIMESTAMP + " >= ? " },
			{ "TO", " and m." + MSGS_TIMESTAMP + " <= ? " },
			{ "WITH", " and m." + MSGS_BUDDY_ID + " = ? " }
	};
	private static final String[] GET_COLLECTIONS_COMBINATIONS = {
		"", "FROM", "FROM_TO", "FROM_TO_WITH", "FROM_WITH",
		"TO", "TO_WITH", "WITH"
	};
	private static final String ADD_MESSAGE = "insert into " + MSGS_TABLE + " (" +
																						MSGS_OWNER_ID + ", " + MSGS_BUDDY_ID + ", " +
																						MSGS_TIMESTAMP + ", " + MSGS_DIRECTION +
																						", " + MSGS_TYPE + ", " + MSGS_BODY + ", " + MSGS_MSG + ")" +
																						" values (?, ?, ?, ?, ?, ?, ?)";
	private static final String REMOVE_MSGS = "delete from " + MSGS_TABLE + " where " +
																						MSGS_OWNER_ID + " = ? and " + MSGS_BUDDY_ID +
																						" = ?" + " and " + MSGS_TIMESTAMP +
																						" <= ? and " + MSGS_TIMESTAMP + " >= ?";
	private static final String DERBY_CREATE_MSGS = "create table " + MSGS_TABLE + " ("
																									+ MSGS_ID + " bigint generated by default as identity not null PRIMARY KEY,"
																									+ MSGS_OWNER_ID + " bigint references " + JIDS_TABLE + "(" + JIDS_ID + "),"
																									+ MSGS_BUDDY_ID + " bigint references " + JIDS_TABLE + "(" + JIDS_ID + "),"
																									+ MSGS_TIMESTAMP + " timestamp, "
																									+ MSGS_DIRECTION + " smallint, "
																									+ MSGS_TYPE + " varchar(10), "
																									+ MSGS_BODY + " varchar(32672), "
																									+ MSGS_MSG + " varchar(32672));"
																									+ "create index " + MSGS_TABLE + "_" + MSGS_OWNER_ID + "_index on " + MSGS_TABLE
																									+ " (" + MSGS_OWNER_ID + ");"
																									+ "create index " + MSGS_TABLE + "_" + MSGS_OWNER_ID + "_" + MSGS_BUDDY_ID
																									+ "_index on " + MSGS_TABLE + " (" + MSGS_OWNER_ID + ", " + MSGS_BUDDY_ID + ");"
																									+ "create index " + MSGS_TABLE + "_" + MSGS_OWNER_ID + "_" + MSGS_TIMESTAMP + "_"
																									+ MSGS_BUDDY_ID + "_index on " + MSGS_TABLE + " (" + MSGS_OWNER_ID + ", "
																									+ MSGS_TIMESTAMP + ", " + MSGS_BUDDY_ID + ");";
	private static final String PGSQL_CREATE_MSGS = "create table " + MSGS_TABLE + " (" +
																									MSGS_ID + " bigserial, " +
																									MSGS_OWNER_ID + " bigint, " +
																									MSGS_BUDDY_ID + " bigint, " +
																									MSGS_TIMESTAMP + " timestamp, " +
																									MSGS_DIRECTION + " smallint, " +
																									MSGS_TYPE + " varchar(10), " +
																									MSGS_BODY + " text, " +
																									MSGS_MSG + " text," +
																									" primary key (" + MSGS_ID + ")," +
																									" foreign key (" + MSGS_BUDDY_ID +
																									") references " + JIDS_TABLE + " (" +
																									JIDS_ID + ")," + " foreign key (" +
																									MSGS_OWNER_ID + ") references " +
																									JIDS_TABLE + " (" + JIDS_ID + ") ); " +
																									"create index " + MSGS_TABLE + "_" +
																									MSGS_OWNER_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									"); " + "create index " + MSGS_TABLE +
																									"_" + MSGS_OWNER_ID + "_" +
																									MSGS_BUDDY_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									", " + MSGS_BUDDY_ID + "); " +
																									"create index " + MSGS_TABLE + "_" +
																									MSGS_OWNER_ID + "_" + MSGS_TIMESTAMP +
																									"_" + MSGS_BUDDY_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									", " + MSGS_TIMESTAMP + ", " + 
																									MSGS_BUDDY_ID + "); ";
	private static final String SQLSERVER_CREATE_MSGS = "create table " + MSGS_TABLE + " (" +
																									MSGS_ID + " bigint IDENTITY(1,1) NOT NULL, " +
																									MSGS_OWNER_ID + " bigint, " +
																									MSGS_BUDDY_ID + " bigint, " +
																									MSGS_TIMESTAMP + " datetime, " +
																									MSGS_DIRECTION + " smallint, " +
																									MSGS_TYPE + " nvarchar(10)," +
																									MSGS_BODY + " ntext, " +
																									MSGS_MSG + " ntext," +
																									" primary key (" + MSGS_ID + ")," +
																									" foreign key (" + MSGS_BUDDY_ID +
																									") references " + JIDS_TABLE + " (" +
																									JIDS_ID + ")," + " foreign key (" +
																									MSGS_OWNER_ID + ") references " +
																									JIDS_TABLE + " (" + JIDS_ID + ") ); " +
																									"create index " + MSGS_TABLE + "_" +
																									MSGS_OWNER_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									"); " + "create index " + MSGS_TABLE +
																									"_" + MSGS_OWNER_ID + "_" +
																									MSGS_BUDDY_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									", " + MSGS_BUDDY_ID + "); " +
																									"create index " + MSGS_TABLE + "_" +
																									MSGS_OWNER_ID + "_" + MSGS_TIMESTAMP +
																									"_" + MSGS_BUDDY_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									", " + MSGS_TIMESTAMP + ", " +
																									MSGS_BUDDY_ID + "); ";
	private static final String MYSQL_CREATE_MSGS = "create table " + MSGS_TABLE + " (" +
																									MSGS_ID + " bigint unsigned NOT NULL auto_increment, " + 
																									MSGS_OWNER_ID + " bigint unsigned, " +
																									MSGS_BUDDY_ID + " bigint unsigned, " +
																									MSGS_TIMESTAMP + " timestamp, " +
																									MSGS_DIRECTION + " smallint, " +
																									MSGS_TYPE + " varchar(10)," +
																									MSGS_BODY + " text, " + 
																									MSGS_MSG + " text," +
																									" primary key (" + MSGS_ID + "), " +
																									" foreign key (" + MSGS_BUDDY_ID +
																									") references " + JIDS_TABLE + " (" +
																									JIDS_ID + ")," + " foreign key (" +
																									MSGS_OWNER_ID + ") references " +
																									JIDS_TABLE + " (" + JIDS_ID + "), " +
																									"key (" + MSGS_OWNER_ID + "), " + 
																									"key (" + MSGS_OWNER_ID + ", " + 
																									MSGS_BUDDY_ID + "), key (" + 
																									MSGS_OWNER_ID + ", " + MSGS_TIMESTAMP + 
																									", " + MSGS_BUDDY_ID + "));";

	private static final String ADD_TAG = "insert into " + TAGS_TABLE + " (" + TAGS_OWNER_ID + ", " + TAGS_TAG + ") values (?,?)";
	private static final String ADD_MESSAGE_TAG = "insert into " + MSGS_TAGS_TABLE + " (" + MSGS_ID + ", " + TAGS_ID + ") values (?,?)";
	private static final String GET_TAG_IDS = "select " + TAGS_ID + ", " + TAGS_TAG + " from " + TAGS_TABLE + " WHERE " + TAGS_OWNER_ID + " = ? AND ( ";
	private static final String GET_TAG_IDS_WHERE_PART = TAGS_TAG + " = ? ";
	private static final String GET_TAG_IDS_END = " )";
	
	private static final String GET_TAGS_FOR_USER = "select t." + TAGS_TAG + " from " + TAGS_TABLE + " t inner join " + JIDS_TABLE 
			+ " j on t." + TAGS_OWNER_ID + " = j." + JIDS_ID + " where j." + JIDS_JID + " = ? and t." + TAGS_TAG + " like ? ";
	private static final String GET_TAGS_FOR_USER_COUNT = "select count(t." + TAGS_ID + ") from " + TAGS_TABLE + " t inner join " + JIDS_TABLE 
			+ " j on t." + TAGS_OWNER_ID + " = j." + JIDS_ID + " where j." + JIDS_JID + " = ? and t." + TAGS_TAG + " like ? ";
	private static final String GET_TAGS_FOR_USER_ORDER = " order by " + TAGS_TAG;
	private static final String SQLSERVER_GET_TAGS_FOR_USER = "select x." + TAGS_TAG + " from ("
			+ " select t." + TAGS_TAG + ", ROW_NUMBER() over (order by t." + TAGS_TAG + ") as rn from " + TAGS_TABLE + " t inner join " + JIDS_TABLE 
			+ " j on t." + TAGS_OWNER_ID + " = j." + JIDS_ID + " where j." + JIDS_JID + " = ? and t." + TAGS_TAG + " like ? ) x ";
	
	private static final String STORE_PLAINTEXT_BODY_KEY = "store-plaintext-body";
	
	//~--- fields ---------------------------------------------------------------

	private DataRepository data_repo = null;
	private boolean storePlaintextBody = true;

	//~--- methods --------------------------------------------------------------

	/**
	 * Method description
	 *
	 *
	 * @param conn_str
	 * @param params
	 *
	 * @throws SQLException
	 */
	@Override
	public void initRepository(String conn_str, Map<String, String> params)
					throws DBInitException {
		try {
			data_repo = RepositoryFactory.getDataRepository( null, conn_str, params );
			if (params.containsKey(STORE_PLAINTEXT_BODY_KEY)) {
				storePlaintextBody = Boolean.parseBoolean(params.get(STORE_PLAINTEXT_BODY_KEY));
			} else {
				storePlaintextBody = true;
			}

			// create tables if not exist
			switch ( data_repo.getDatabaseType() ) {
				case mysql:
					data_repo.checkTable( JIDS_TABLE, MYSQL_CREATE_JIDS );
					data_repo.checkTable( MSGS_TABLE, MYSQL_CREATE_MSGS );
					break;
				case derby:
					data_repo.checkTable( JIDS_TABLE, DERBY_CREATE_JIDS );
					data_repo.checkTable( MSGS_TABLE, DERBY_CREATE_MSGS );
					break;
				case postgresql:
					data_repo.checkTable( JIDS_TABLE, PGSQL_CREATE_JIDS );
					data_repo.checkTable( MSGS_TABLE, PGSQL_CREATE_MSGS );
					break;
				case jtds:
				case sqlserver:
					data_repo.checkTable( JIDS_TABLE, SQLSERVER_CREATE_JIDS );
					data_repo.checkTable( MSGS_TABLE, SQLSERVER_CREATE_MSGS );
					break;
			}
			
			checkDB();
			
			switch ( data_repo.getDatabaseType() ) {
				case mysql:
					data_repo.checkTable( TAGS_TABLE, MYSQL_CREATE_TAGS );
					data_repo.checkTable( MSGS_TAGS_TABLE, MYSQL_CREATE_MSGS_TAGS );
					break;
				case derby:
					data_repo.checkTable( TAGS_TABLE, DERBY_CREATE_TAGS );
					data_repo.checkTable( MSGS_TAGS_TABLE, DERBY_CREATE_MSGS_TAGS );
					break;
				case postgresql:
					data_repo.checkTable( TAGS_TABLE, PGSQL_CREATE_TAGS );
					data_repo.checkTable( MSGS_TAGS_TABLE, PGSQL_CREATE_MSGS_TAGS );
					break;
				case jtds:
				case sqlserver:
					data_repo.checkTable( TAGS_TABLE, SQLSERVER_CREATE_TAGS );
					data_repo.checkTable( MSGS_TAGS_TABLE, SQLSERVER_CREATE_MSGS_TAGS );
					break;
			}
			
			data_repo.initPreparedStatement(ADD_JID_QUERY, ADD_JID_QUERY);
			data_repo.initPreparedStatement(GET_JID_ID_QUERY, GET_JID_ID_QUERY);
			data_repo.initPreparedStatement(GET_JID_IDS_QUERY, GET_JID_IDS_QUERY);

			data_repo.initPreparedStatement(ADD_MESSAGE, ADD_MESSAGE, Statement.RETURN_GENERATED_KEYS);
			//data_repo.initPreparedStatement(GET_COLLECTIONS, GET_COLLECTIONS);
			
			Map<String,String> combinations = new HashMap<String,String>();
			for (String combination : GET_COLLECTIONS_COMBINATIONS) {
				StringBuilder sbMain = new StringBuilder();

				if (!combination.isEmpty()) {
					String[] whereParts = combination.split("_");

					for (String part : whereParts) {
						for (String[] where : GET_COLLECTIONS_WHERES) {
							if (!part.equals(where[0])) {
								continue;
							}

							sbMain.append(where[1]);
						}
					}
				}
				
				for (int j = 0; j < 6; j++) {
					StringBuilder combinationSb1 = new StringBuilder().append(combination);
					StringBuilder querySb1 = new StringBuilder().append(sbMain);

					if (j > 0) {
						if (combinationSb1.length() > 0) {
							combinationSb1.append("_");
						}
						combinationSb1.append("TAGS[").append(j).append("]");
						querySb1.append(" and EXISTS( select 1 from ").append(TAGS_TABLE)
								.append(" t inner join ").append(MSGS_TAGS_TABLE).append(" tm on t.")
								.append(TAGS_ID).append(" = tm.").append(TAGS_ID).append(" where m.")
								.append(MSGS_ID).append(" = tm.").append(MSGS_ID).append(" and (");
						for (int x = 0; x < j; x++) {
							if (x > 0) {
								querySb1.append(" or ");
							}
							querySb1.append("t.").append(TAGS_TAG).append(" = ?");
						}
						querySb1.append(" )) ");
					}
					
					for (int i = 0; i < 6; i++) {
						StringBuilder combinationSb = new StringBuilder().append(combinationSb1);
						StringBuilder querySb = new StringBuilder().append(querySb1);

						if (i > 0) {
							if (combinationSb.length() > 0) {
								combinationSb.append("_");
							}
							combinationSb.append("CONTAINS[").append(i).append("]");
							for (int x = 0; x < i; x++) {
								querySb.append(" and m.").append(MSGS_BODY).append(" like ? ");
							}
						}

						combinations.put(combinationSb.toString(), querySb.toString());
					}				
				}
			}	
			
			for (Map.Entry<String,String> entry : combinations.entrySet()) {
				StringBuilder select = new StringBuilder();
				StringBuilder count = new StringBuilder().append(GENERIC_GET_COLLECTIONS_COUNT);
				
				switch ( data_repo.getDatabaseType() ) {
					case jtds:
					case sqlserver:
						select.append(MSSQL2008_GET_COLLECTIONS_SELECT);
						break;
					default:
						select.append(GENERIC_GET_COLLECTIONS_SELECT);
						break;
				}
				
				select.append(entry.getValue());
				count.append(entry.getValue());
				
				switch ( data_repo.getDatabaseType() ) {
					case jtds:
					case sqlserver:
						select.append(MSSQL2008_GET_COLLECTIONS_SELECT_GROUP);
						count.append(MSSQL2008_GET_COLLECTIONS_COUNT_GROUP);
						break;
					default:
						select.append(GENERIC_GET_COLLECTIONS_SELECT_GROUP + GENERIC_GET_COLLECTIONS_SELECT_ORDER);
						count.append(GENERIC_GET_COLLECTIONS_COUNT_GROUP);
						break;
				}
				
				switch ( data_repo.getDatabaseType() ) {
					case derby:
						select.append(DERBY_LIMIT);
						break;
					case jtds:
					case sqlserver:
						select.append(MSSQL2008_LIMIT).append(MSSQL2008_GET_COLLECTIONS_SELECT_ORDER);
						break;
					default:
						select.append(GENERIC_LIMIT);
						break;
				}
				
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "prepared collection select query for " + entry.getKey() + " as '" + select.toString() + "'");
					log.log(Level.FINEST, "prepared collection count query for " + entry.getKey() + " as '" + count.toString() + "'");
				}
					
				data_repo.initPreparedStatement("GET_COLLECTIONS_" + entry.getKey() + "_SELECT", select.toString());
				data_repo.initPreparedStatement("GET_COLLECTIONS_" + entry.getKey() + "_COUNT", count.toString());						
			}
			
			for (Map.Entry<String,String> entry : combinations.entrySet()) {
				StringBuilder select = new StringBuilder();
				StringBuilder count = new StringBuilder().append(GENERIC_GET_MESSAGES_COUNT);
				
				boolean containsWith = entry.getKey().contains("WITH");
				
				switch ( data_repo.getDatabaseType() ) {
					case jtds:
					case sqlserver:
						if (containsWith) {
							select.append(MSSQL2008_GET_MESSAGES_START);
						} else {
							select.append(MSSQL2008_GET_MESSAGES_START_WITH);
						}
						break;
					default:
						if (containsWith) {
							select.append(GENERIC_GET_MESSAGES_START);						
						} else {
							select.append(GENERIC_GET_MESSAGES_START_WITH);							
						}
						break;
				}
				
				select.append(entry.getValue());
				count.append(entry.getValue());

				switch (data_repo.getDatabaseType()) {
					case derby:
						select.append(GENERIC_GET_MESSAGES_ORDER_BY).append(DERBY_LIMIT);
						break;
					case jtds:
					case sqlserver:
						if (containsWith) {
							select.append(MSSQL2008_GET_MESSAGES_END);
						} else {
							select.append(MSSQL2008_GET_MESSAGES_END_WITH);
						}
						select.append(MSSQL2008_LIMIT).append(GENERIC_GET_MESSAGES_ORDER_BY);
						break;
					default:
						select.append(GENERIC_GET_MESSAGES_ORDER_BY).append(GENERIC_LIMIT);
						break;
				}			
				
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "prepared messages select query for " + entry.getKey() + " as '" + select.toString() + "'");
					log.log(Level.FINEST, "prepared messages count query for " + entry.getKey() + " as '" + count.toString() + "'");
				}

				data_repo.initPreparedStatement("GET_MESSAGES_" + entry.getKey() + "_SELECT", select.toString());
				data_repo.initPreparedStatement("GET_MESSAGES_" + entry.getKey() + "_COUNT", count.toString());
			}			
			
			data_repo.initPreparedStatement(REMOVE_MSGS, REMOVE_MSGS);
			
			for (int i=0; i<=5; i++) {
				StringBuilder select = new StringBuilder().append(GET_TAG_IDS);
				for (int j=1; j<=i; j++) {
					if (j > 1)
						select.append(" or ");
					select.append(GET_TAG_IDS_WHERE_PART);
				}
				if (i==0) {
					select.append("1=1");
				}
				select.append(")");
				data_repo.initPreparedStatement(GET_TAG_IDS + "_" + i, select.toString());
			}
			data_repo.initPreparedStatement(ADD_TAG, ADD_TAG, Statement.RETURN_GENERATED_KEYS);
			data_repo.initPreparedStatement(ADD_MESSAGE_TAG, ADD_MESSAGE_TAG);
			
			data_repo.initPreparedStatement(GET_TAGS_FOR_USER_COUNT, GET_TAGS_FOR_USER_COUNT);
			switch (data_repo.getDatabaseType()) {
				case derby:
					data_repo.initPreparedStatement(GET_TAGS_FOR_USER, GET_TAGS_FOR_USER + GET_TAGS_FOR_USER_ORDER + DERBY_LIMIT);
					break;
				case jtds:
				case sqlserver:
					data_repo.initPreparedStatement(GET_TAGS_FOR_USER, SQLSERVER_GET_TAGS_FOR_USER + MSSQL2008_LIMIT + GET_TAGS_FOR_USER_ORDER);
					break;
				default:
					data_repo.initPreparedStatement(GET_TAGS_FOR_USER, GET_TAGS_FOR_USER + GET_TAGS_FOR_USER_ORDER + GENERIC_LIMIT);					
					break;
			}
		} catch (Exception ex) {
			log.log(Level.WARNING, "MessageArchiveDB initialization exception", ex);
		}
	}

	@Override
	public void destroy() {
		// here we use cached instance of repository pool cached by RepositoryFactory
		// so we should not close it
	}
	
	private void checkDB() {
		Statement stmt = null;
		try {
			try {
				stmt = data_repo.createStatement(null);
				stmt.executeQuery("select " + MSGS_BODY + " from " + MSGS_TABLE + " where " + MSGS_OWNER_ID + " = 0");
			} catch (SQLException ex) {
				// if this happens then we have issue with old database schema and missing body columns in MSGS_TABLE
				String alterTable = null;
				switch (data_repo.getDatabaseType()) {
					case derby:
						alterTable = "alter table " + MSGS_TABLE + " add " + MSGS_BODY + " varchar(32672)";
						break;
					case mysql:
						alterTable = "alter table " + MSGS_TABLE + " add " + MSGS_BODY + " text";
						break;
					case postgresql:
						alterTable = "alter table " + MSGS_TABLE + " add " + MSGS_BODY + " text";
						break;
					case jtds:
					case sqlserver:
						alterTable = "alter table " + MSGS_TABLE + " add " + MSGS_BODY + " ntext";
						break;
				}
				try {
					stmt.execute(alterTable);
				} catch (SQLException ex1) {
					log.log(Level.SEVERE, "could not alter table " + MSGS_TABLE + " to add missing column by SQL:\n" + alterTable, ex1);
				}
			}
			try {
				stmt = data_repo.createStatement(null);
				stmt.executeQuery("select " + MSGS_ID + " from " + MSGS_TABLE + " where " + MSGS_OWNER_ID + " = 0");
			} catch (SQLException ex) {
				// if this happens then we have issue with old database schema and missing id column in MSGS_TABLE
				String alterTable = null;
				try {
					switch (data_repo.getDatabaseType()) {
						case derby:
							alterTable = "alter table " + MSGS_TABLE + " add " + MSGS_ID + " bigint generated by default as identity not null";
							stmt.execute(alterTable);
							alterTable = "alter table " + MSGS_TABLE + " add primary key (" + MSGS_ID + ")";
							stmt.execute(alterTable);
							break;
						case mysql:
							alterTable = "alter table " + MSGS_TABLE + " add " + MSGS_ID + " bigint unsigned NOT NULL auto_increment, add primary key (" + MSGS_ID + ")";
							stmt.execute(alterTable);
							break;
						case postgresql:
							alterTable = "alter table " + MSGS_TABLE + " add " + MSGS_ID + " serial";
							stmt.execute(alterTable);
							alterTable = "alter table " + MSGS_TABLE + " add primary key (" + MSGS_ID + ")";
							stmt.execute(alterTable);
							break;
						case jtds:
						case sqlserver:
							alterTable = "alter table " + MSGS_TABLE + " add " + MSGS_ID + " bigint identity(1,1)";
							stmt.execute(alterTable);
							alterTable = "alter table " + MSGS_TABLE + " add primary key (" + MSGS_ID + ")";
							stmt.execute(alterTable);
							break;
					}
				} catch (SQLException ex1) {
					log.log(Level.SEVERE, "could not alter table " + MSGS_TABLE + " to add missing column by SQL:\n" + alterTable, ex1);
				}
			}			
		} finally {
			data_repo.release(stmt, null);
		}
	}
	
	/**
	 * Method description
	 *
	 *
	 * @param owner
	 * @param buddy
	 * @param direction
	 * @param timestamp
	 * @param msg
	 * @param tags
	 */
	@Override
	public void archiveMessage(BareJID owner, BareJID buddy, Direction direction, Date timestamp, Element msg, Set<String> tags) {
		ResultSet rs = null;
		try {
			String owner_str         = owner.toString();
			String buddy_str         = buddy.toString();
			long[] jids_ids          = getJidsIds(owner_str, buddy_str);
			long owner_id            = (jids_ids[0] != LONG_NULL)
																 ? jids_ids[0]
																 : addJidId(owner_str);
			long buddy_id            = (jids_ids[1] != LONG_NULL)
																 ? jids_ids[1]
																 : addJidId(buddy_str);
			java.sql.Timestamp mtime = new java.sql.Timestamp(timestamp.getTime());
			msg.addAttribute("time", String.valueOf(mtime.getTime()));

			String type                      = msg.getAttributeStaticStr("type");
			String msgStr                    = msg.toString();
			String body                      = storePlaintextBody ? msg.getChildCData(MSG_BODY_PATH) : null;
			PreparedStatement add_message_st = data_repo.getPreparedStatement(owner,
																					 ADD_MESSAGE);

			Long msgId = null;
			
			synchronized (add_message_st) {
				add_message_st.setLong(1, owner_id);
				add_message_st.setLong(2, buddy_id);
				add_message_st.setTimestamp(3, mtime);
				add_message_st.setShort(4, direction.getValue());
				add_message_st.setString(5, type);
				add_message_st.setString(6, body);
				add_message_st.setString(7, msgStr);
				add_message_st.executeUpdate();
				
				if (tags != null) {
					rs = add_message_st.getGeneratedKeys();
					if (rs.next()) {
						switch (data_repo.getDatabaseType()) {
							case postgresql:
								msgId = rs.getLong(MSGS_ID);
								break;
							default:
								msgId = rs.getLong(1);
								break;
						}
					}
				}
			}
			
			if (tags != null && !tags.isEmpty()) {
				Map<String,Long> tagsMap = ensureTags(owner, owner_id, tags);
				PreparedStatement add_message_tag_st = data_repo.getPreparedStatement(owner, ADD_MESSAGE_TAG);
				synchronized (add_message_tag_st) {
					for (Long tagId : tagsMap.values()) {
						add_message_tag_st.setLong(1, msgId);
						add_message_tag_st.setLong(2, tagId);
						add_message_tag_st.addBatch();
					}
					add_message_tag_st.executeBatch();
				}
			}
		} catch (SQLException ex) {
			log.log(Level.WARNING, "Problem adding new entry to DB: " + msg, ex);
		} finally {
		   data_repo.release(null, rs);
		}
	}

	private Map<String,Long> ensureTags(BareJID owner, long owner_id, Set<String> tags) throws SQLException {
		Map<String,Long> tagsMap = new HashMap<String,Long>();
		ResultSet rs = null;
		try {
			Iterator<String> it = tags.iterator();
			int iters = (tags.size()/5)+1;
			for (int i = 0; i < iters; i++) {
				int params = (i == (iters - 1)) ? tags.size() % 5 : 5;
				PreparedStatement get_tag_ids_st = data_repo.getPreparedStatement(owner, GET_TAG_IDS + "_" + params);
				synchronized (get_tag_ids_st) {
					get_tag_ids_st.setLong(1, owner_id);
					for (int j=0; j<params; j++) {
						String tag = it.next();
						get_tag_ids_st.setString(j+2, tag);
					}
					rs = get_tag_ids_st.executeQuery();
					while (rs.next()) {
						long id = rs.getLong(1);
						String tag = rs.getString(2);
						tagsMap.put(tag, id);
					}
					data_repo.release(null, rs);
					rs = null;
				}
			}
			
			if (tagsMap.size() < tags.size()) {
				PreparedStatement add_tag_st = data_repo.getPreparedStatement(owner, ADD_TAG);
				for (String tag : tags) {
					if (tagsMap.containsKey(tag))
						continue;
					
					synchronized (add_tag_st) {
						add_tag_st.setLong(1, owner_id);
						add_tag_st.setString(2, tag);
						add_tag_st.executeUpdate();
						rs = add_tag_st.getGeneratedKeys();
						if (rs.next()) {
							tagsMap.put(tag, rs.getLong(1));
						}
						data_repo.release(null, rs);
						rs = null;						
					}
				}
			}
		} finally {
			data_repo.release(null, rs);
		}
		
		return tagsMap;
	}
	
	//~--- get methods ----------------------------------------------------------

	/**
	 * Method description
	 *
	 *
	 * @param owner
	 * @param crit
	 *
	 * @return
	 * @throws tigase.db.TigaseDBException
	 */
	@Override
	public List<Element> getCollections(BareJID owner, Criteria crit)
					 throws TigaseDBException {
		try {
			long[] jids_ids = crit.getWith() == null ? getJidsIds(owner.toString()) : getJidsIds(owner.toString(), crit.getWith());

			crit.setOwnerId(jids_ids[0]);
			if (jids_ids.length > 1)
				crit.setBuddyId(jids_ids[1]);

			Integer count = getCollectionsCount(owner, crit);
			if (count == null)
				count = 0;
			crit.setSize(count);

			List<Element> results = getCollectionsItems(owner, crit);

			RSM rsm = crit.getRSM();
			rsm.setResults(count, crit.getOffset());
			if (!results.isEmpty()) {
				rsm.setFirst(String.valueOf(crit.getOffset()));
				rsm.setLast(String.valueOf(crit.getOffset() + (results.size() - 1)));
			}

			return results;
		} catch (SQLException ex) {
			throw new TigaseDBException("Cound not retrieve collections", ex);
		}		
	}

	/**
	 * Method description
	 *
	 *
	 * @param owner
	 * @param crit
	 *
	 * @return
	 * @throws tigase.db.TigaseDBException
	 */
	@Override
	public List<Element> getItems(BareJID owner, Criteria crit)
					 throws TigaseDBException {
		try {
			long[] jids_ids = crit.getWith() != null ? getJidsIds(owner.toString(), crit.getWith()) : getJidsIds(owner.toString());

			crit.setOwnerId(jids_ids[0]);
			if (jids_ids.length > 1)
				crit.setBuddyId(jids_ids[1]);


			Integer count = getItemsCount(owner, crit);
			if (count == null) {
				count = 0;
			}
			crit.setSize(count);

			List<Element> items = getItemsItems(owner, crit);

			RSM rsm = crit.getRSM();
			rsm.setResults(count, crit.getOffset());
			if (items!= null && !items.isEmpty()) {
				rsm.setFirst(String.valueOf(crit.getOffset()));
				rsm.setLast(String.valueOf(crit.getOffset() + (items.size() - 1)));
			}

			return items;
		} catch (SQLException ex) {
			throw new TigaseDBException("Cound not retrieve items", ex);
		}		
	}

	//~--- methods --------------------------------------------------------------

	/**
	 * Method description
	 *
	 *
	 * @param owner
	 * @param withJid
	 * @param start
	 * @param end
	 *
	 * @throws TigaseDBException
	 */
	@Override
	public void removeItems(BareJID owner, String withJid, Date start, Date end)
					throws TigaseDBException {
		try {
			long[] jids_ids = getJidsIds(owner.toString(), withJid);

			if (start == null) {
				start = new Date(0);
			}
			if (end == null) {
				end = new Date(0);
			}

			java.sql.Timestamp start_ = new java.sql.Timestamp(start.getTime());
			java.sql.Timestamp end_ = new java.sql.Timestamp(end.getTime());
			PreparedStatement remove_msgs_st = data_repo.getPreparedStatement(owner, REMOVE_MSGS);

			synchronized (remove_msgs_st) {
				synchronized (remove_msgs_st) {
					remove_msgs_st.setLong(1, jids_ids[0]);
					remove_msgs_st.setLong(2, jids_ids[1]);
					remove_msgs_st.setTimestamp(3, end_);
					remove_msgs_st.setTimestamp(4, start_);
					remove_msgs_st.executeUpdate();
				}
			}
		} catch (SQLException ex) {
			throw new TigaseDBException("Cound not remove items", ex);
		}
	}

	/**
	 * Method description
	 * 
	 * @param owner
	 * @param startsWith
	 * @param crit
	 * @return
	 * @throws TigaseDBException 
	 */
	@Override
	public List<String> getTags(BareJID owner, String startsWith, Criteria crit) throws TigaseDBException {
		List<String> results = new ArrayList<String>();
		ResultSet rs = null;
		try {
			int count = 0;
			startsWith = startsWith + "%";
			
			PreparedStatement get_tags_count_st = data_repo.getPreparedStatement(owner, GET_TAGS_FOR_USER_COUNT);
			synchronized (get_tags_count_st) {
				get_tags_count_st.setString(1, owner.toString());
				get_tags_count_st.setString(2, startsWith);
				
				rs = get_tags_count_st.executeQuery();
				if (rs.next()) {
					count = rs.getInt(1);
				}
				data_repo.release(null, rs);
			}
			crit.setSize(count);

			PreparedStatement get_tags_st = data_repo.getPreparedStatement(owner, GET_TAGS_FOR_USER);
			synchronized (get_tags_st) {
				int i=1;
				get_tags_st.setString(i++, owner.toString());
				get_tags_st.setString(i++, startsWith);

				switch (data_repo.getDatabaseType()) {
					case derby:
						get_tags_st.setInt(i++, crit.getOffset());
						get_tags_st.setInt(i++, crit.getLimit());
						break;
					case jtds:
					case sqlserver:
						get_tags_st.setInt(i++, crit.getOffset());
						get_tags_st.setInt(i++, crit.getOffset() + crit.getLimit());
						break;
					default:
						get_tags_st.setInt(i++, crit.getLimit());
						get_tags_st.setInt(i++, crit.getOffset());
						break;
				}				
				
				rs = get_tags_st.executeQuery();
				while (rs.next()) {
					results.add(rs.getString(1));
				}
			}
			
			RSM rsm = crit.getRSM();
			rsm.setResults(count, crit.getOffset());
			if (results!= null && !results.isEmpty()) {
				rsm.setFirst(String.valueOf(crit.getOffset()));
				rsm.setLast(String.valueOf(crit.getOffset() + (results.size() - 1)));
			}			
		} catch (SQLException ex) {
			throw new TigaseDBException("Could not retrieve known tags from database", ex);
		} finally {
			data_repo.release(null, rs);
		}
		
		return results;
	}

	private List<Element> getCollectionsItems(BareJID owner, Criteria crit)
					throws SQLException {
		List<Element> results = new LinkedList<Element>();
		ResultSet selectRs   = null;
		try {
			PreparedStatement get_collections_st = data_repo.getPreparedStatement(owner, "GET_COLLECTIONS_" 
					+ crit.getQueryName() + "_SELECT");

			int i=2;
			synchronized (get_collections_st) {
				crit.setItemsQueryParams(get_collections_st, data_repo.getDatabaseType());

				selectRs = get_collections_st.executeQuery();
				while (selectRs.next()) {
					Timestamp startTs = selectRs.getTimestamp(1);
					String with = selectRs.getString(2);
					addCollectionToResults(results, with, startTs);
				}
			}
		} finally {
			data_repo.release(null, selectRs);
		}
		return results;
	}	
	
	private Integer getCollectionsCount(BareJID owner, Criteria crit) throws SQLException {
		ResultSet countRs = null;		
		Integer count = null;
		try {
			PreparedStatement get_collections_count = data_repo.getPreparedStatement(owner, "GET_COLLECTIONS_" 
					+ crit.getQueryName() + "_COUNT");
			int i=2;
			synchronized (get_collections_count) {
				crit.setCountQueryParams(get_collections_count, data_repo.getDatabaseType());
				countRs = get_collections_count.executeQuery();
				if (countRs.next()) {
					count = countRs.getInt(1);
				}
			}
		} finally {
			data_repo.release(null, countRs);
		}
		return count;
	}
	
	private List<Element> getItemsItems(BareJID owner, Criteria crit) throws SQLException {
		ResultSet rs      = null;		
		Queue<Item> results = new ArrayDeque<Item>();
		int i=1;
		try {
			boolean containsWith = crit.getQueryName().contains("WITH");
			PreparedStatement get_messages_st = data_repo.getPreparedStatement(owner, "GET_MESSAGES_" + crit.getQueryName() + "_SELECT");
			synchronized (get_messages_st) {
				crit.setItemsQueryParams(get_messages_st, data_repo.getDatabaseType());

				rs = get_messages_st.executeQuery();
				while (rs.next()) {
					Item item = new Item();
					item.message = rs.getString(1);
					item.timestamp = rs.getTimestamp(2);
					item.direction = Direction.getDirection(rs.getShort(3));
					if (!containsWith) {
						item.with = rs.getString(4);
					}
					results.offer(item);
				}
			}
		} finally {
			data_repo.release(null, rs);
		}

		List<Element> msgs = new LinkedList<Element>();

		if (!results.isEmpty()) {
			DomBuilderHandler domHandler = new DomBuilderHandler();

			Date startTimestamp = crit.getStart();
			Item item = null;
			while ((item = results.poll()) != null) {
				// workaround for case in which start was not specified
				if (startTimestamp == null)
					startTimestamp = item.timestamp;
				
				parser.parse(domHandler, item.message.toCharArray(), 0, item.message.length());

				Queue<Element> queue = domHandler.getParsedElements();
				Element msg = null;

				while ((msg = queue.poll()) != null) {
					addMessageToResults(msgs, startTimestamp, msg, item.timestamp, item.direction, item.with);
				}			
			}

			crit.setStart(startTimestamp);
			
			// no point in sorting messages by secs attribute as messages are already
			// sorted in SQL query and also this sorting is incorrect
//			Collections.sort(msgs, new Comparator<Element>() {
//				@Override
//				public int compare(Element m1, Element m2) {
//					return m1.getAttributeStaticStr("secs").compareTo(
//							m2.getAttributeStaticStr("secs"));
//				}
//			});
		}

		return msgs;		
	}

	private Integer getItemsCount(BareJID owner, Criteria crit) throws SQLException {
		ResultSet rs      = null;		
		Integer count = null;
		try {
			PreparedStatement get_messages_st = data_repo.getPreparedStatement(owner, "GET_MESSAGES_" + crit.getQueryName() + "_COUNT");
			synchronized (get_messages_st) {
				crit.setCountQueryParams(get_messages_st, data_repo.getDatabaseType());

				rs = get_messages_st.executeQuery();
				if (rs.next()) {
					count = rs.getInt(1);
				}
			}
		} finally {
			data_repo.release(null, rs);
		}
		return count;
	}
	
	/**
	 * Method description
	 *
	 *
	 * @param jids
	 *
	 * @return
	 *
	 * @throws SQLException
	 */
	protected long[] getJidsIds(String... jids) throws SQLException {
		ResultSet rs = null;

		try {
			long[] results = new long[jids.length];

			Arrays.fill(results, LONG_NULL);
			if (jids.length == 1) {
				PreparedStatement get_jid_id_st = data_repo.getPreparedStatement(null,
																						GET_JID_ID_QUERY);

				synchronized (get_jid_id_st) {
					get_jid_id_st.setString(1, jids[0]);
					rs = get_jid_id_st.executeQuery();
					if (rs.next()) {
						results[0] = rs.getLong("jid_id");

						return results;
					}
				}

				return results;
			} else {
				PreparedStatement get_jids_id_st = data_repo.getPreparedStatement(null,
																						 GET_JID_IDS_QUERY);

				synchronized (get_jids_id_st) {
					for (int i = 0; i < jids.length; i++) {
						get_jids_id_st.setString(i + 1, jids[i]);
					}
					rs = get_jids_id_st.executeQuery();

					int cnt = 0;

					while (rs.next()) {
						String db_jid = rs.getString("jid");

						for (int i = 0; i < jids.length; i++) {
							if (db_jid.equals(jids[i])) {
								results[i] = rs.getLong("jid_id");
								++cnt;
							}
						}
					}

					return results;
				}
			}
		} finally {
			data_repo.release(null, rs);
		}
	}

	//~--- methods --------------------------------------------------------------

	private long addJidId(String jid) throws SQLException {
		PreparedStatement add_jid_st = data_repo.getPreparedStatement(null, ADD_JID_QUERY);

		try {
			synchronized (add_jid_st) {
				add_jid_st.setString(1, jid);
				add_jid_st.executeUpdate();
			}
		} catch (SQLException ex) {
			log.log(Level.FINEST, "Exception adding jid to tig_ma_jids table, it may occur "
					+ "if other thread added this jid in the meantime", ex);
		}

		// This is not the most effective solution but this method shouldn't be
		// called very often so the perfrmance impact should be insignificant.
		long[] jid_ids = getJidsIds(jid);

		if (jid_ids != null) {
			return jid_ids[0];
		} else {

			// That should never happen here, but just in case....
			log.log(Level.WARNING, "I have just added new jid but it was not found.... {0}",
							jid);

			return LONG_NULL;
		}
	}

	@Override
	public AbstractCriteria newCriteriaInstance() {
		return new Criteria();
	}
	
	private class Item {
		
		String message;
		Date timestamp;
		Direction direction;
		String with;
		
	}
	
	public static class Criteria extends AbstractCriteria<Timestamp> {

		private long ownerId;
		private long buddyId;
		private String queryName;
		
		@Override
		protected Timestamp convertTimestamp(Date date) {
			if (date == null)
				return null;
			return new Timestamp(date.getTime());
		}
		
		public String getQueryName() {
			if (queryName == null)
				return updateQueryName();
			return queryName;
		}
		
		public String updateQueryName() {
			StringBuilder query = new StringBuilder(20);
			if (getStart() != null) {
				query.append("FROM");
			}
			if (getEnd() != null) {
				if (query.length() > 0) {
					query.append("_");
				}
				query.append("TO");
			}
			if (getWith() != null) {
				if (query.length() > 0) {
					query.append("_");
				}
				query.append("WITH");
			} else {
				// not supported
			}
			if (!getTags().isEmpty()) {
				if (query.length() > 0) {
					query.append("_");
				}
				query.append("TAGS[").append(getTags().size()).append("]");
			}
			if (!getContains().isEmpty()) {
				if (query.length() > 0) {
					query.append("_");
				}
				query.append("CONTAINS[").append(getContains().size()).append("]");
			}
			queryName = query.toString();
			return queryName;
		}
		
		public void setOwnerId(Long id) {
			if (id == null)
				ownerId = 0;
			else
				ownerId = id;
		}
		
		public void setBuddyId(Long id) {
			if (id == null)
				buddyId = 0;
			else
				buddyId = id;
		}
		
		public int setCountQueryParams(PreparedStatement stmt, DataRepository.dbTypes dbType) throws SQLException {
			int i=1;
			stmt.setLong(i++, ownerId);
			if (getStart() != null) {
				stmt.setTimestamp(i++, getStart());
			}
			if (getEnd() != null) {
				stmt.setTimestamp(i++, getEnd());
			}
			if (getWith() != null) {
				stmt.setLong(i++, buddyId);
			}
			for (String tag : getTags()) {
				stmt.setString(i++, tag);
			}
			for (String contains : getContains()) {
				stmt.setString(i++, "%" + contains + "%");
			}
			return i;
		}
		
		public void setItemsQueryParams(PreparedStatement stmt, DataRepository.dbTypes dbType) throws SQLException {
			int i = setCountQueryParams(stmt, dbType);
			switch (dbType) {
				case derby:
					stmt.setInt(i++, getOffset());
					stmt.setInt(i++, getLimit());
					break;
				case jtds:
				case sqlserver:
					stmt.setInt(i++, getOffset());
					stmt.setInt(i++, getOffset() + getLimit());
					break;
				default:
					stmt.setInt(i++, getLimit());
					stmt.setInt(i++, getOffset());
					break;
			}
		}
	}
}


//~ Formatted in Tigase Code Convention on 13/02/20
