/*
 * MessageArchiveDB.java
 *
 * Tigase Jabber/XMPP Server
 * Copyright (C) 2004-2012 "Artur Hefczyc" <artur.hefczyc@tigase.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. Look for COPYING file in the top folder.
 * If not, see http://www.gnu.org/licenses/.
 *
 */



package tigase.archive;

//~--- non-JDK imports --------------------------------------------------------

import tigase.db.DataRepository;
import tigase.db.RepositoryFactory;

import tigase.server.Message;
import tigase.server.Packet;

import tigase.xml.DomBuilderHandler;
import tigase.xml.Element;
import tigase.xml.SimpleParser;
import tigase.xml.SingletonFactory;

import tigase.xmpp.BareJID;

//~--- JDK imports ------------------------------------------------------------

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;

import java.text.ParseException;
import java.text.SimpleDateFormat;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.Map;
import java.util.Queue;
import java.util.TimeZone;
import static tigase.db.DataRepository.dbTypes.derby;

/**
 * Class description
 *
 *
 * @version        Enter version here..., 13/02/16
 * @author         Enter your name here...
 */
public class MessageArchiveDB {
	private static final String JIDS_ID  = "jid_id";
	private static final String JIDS_JID = "jid";

	// jids table
	private static final String JIDS_TABLE = "tig_ma_jids";
	private static final Logger log        =
		Logger.getLogger(MessageArchiveDB.class.getCanonicalName());
	private static final long LONG_NULL              = 0;
	private static final long MILIS_PER_DAY          = 24 * 60 * 60 * 1000;
	private final static SimpleDateFormat formatter =
		new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
	private final static SimpleDateFormat formatter2 =
		new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");
	private final static SimpleDateFormat formatter3 =
		new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
	private final static SimpleDateFormat formatter4 =
		new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
	
	static {
		formatter.setTimeZone(TimeZone.getTimeZone("UTC"));
		formatter2.setTimeZone(TimeZone.getTimeZone("UTC"));
		formatter3.setTimeZone(TimeZone.getTimeZone("UTC"));
		formatter4.setTimeZone(TimeZone.getTimeZone("UTC"));
	}
	private static final String MSGS_BUDDY_ID  = "buddy_id";
	private static final String MSGS_DIRECTION = "direction";
	private static final String MSGS_MSG       = "msg";
	private static final String MSGS_OWNER_ID  = "owner_id";

//+ "create unique index " + JIDS_TABLE + "_" + JIDS_JID + " on "
//+ JIDS_TABLE + " ( " + JIDS_JID + "(765));";
	// messages table
	private static final String MSGS_TABLE        = "tig_ma_msgs";
	private static final String MSGS_TIMESTAMP    = "ts";
	private static final String MSGS_TYPE         = "type";
	private static final SimpleParser parser      = SingletonFactory.getParserInstance();
	private static final String GET_JID_IDS_QUERY = "select " + JIDS_JID + ", " + JIDS_ID +
																									" from " + JIDS_TABLE + " where " +
																									JIDS_JID + " = ?" + " or " + JIDS_JID +
																									" = ?";
	private static final String GET_JID_ID_QUERY = "select " + JIDS_JID + ", " + JIDS_ID +
																								 " from " + JIDS_TABLE + " where " +
																								 JIDS_JID + " = ?";
	private static final String ADD_JID_QUERY = "insert into " + JIDS_TABLE + " (" +
																							JIDS_JID + ") values (?)";
	private static final String DERBY_CREATE_JIDS = "create table " + JIDS_TABLE + " ( "
																									+ JIDS_ID + " bigint generated by default as identity not null, " + JIDS_JID
																									+ " varchar(2049), primary key ( " + JIDS_ID + " ));"
																									+ "create unique index " + JIDS_TABLE + "_" + JIDS_JID + " on " + JIDS_TABLE
																									+ " (" + JIDS_JID + ");";
	private static final String PGSQL_CREATE_JIDS = "create table " + JIDS_TABLE + " ( "
																									+ JIDS_ID + " bigserial, " + JIDS_JID
																									+ " varchar(2049), primary key ("
																									+ JIDS_ID + ")); "
																									+ "create unique index " + JIDS_TABLE
																									+ "_" + JIDS_JID + " on " + JIDS_TABLE
																									+ " ( " + JIDS_JID + ");";
	private static final String SQLSERVER_CREATE_JIDS = "create table " + JIDS_TABLE + " ( "
																									+ JIDS_ID + " bigint identity(1,1), " + JIDS_JID
																									+ " nvarchar(2049),"
																									+ JIDS_JID + "_fragment as left (" + JIDS_JID + ", 765),"
																									+ "primary key (" + JIDS_ID + ")); "
																									+ "create unique index " + JIDS_TABLE
																									+ "_" + JIDS_JID + " on " + JIDS_TABLE
																									+ " ( " + JIDS_JID + "_fragment " + ");";
	private static final String MYSQL_CREATE_JIDS =
															"create table " + JIDS_TABLE + " ( " + JIDS_ID
															+ " bigint unsigned NOT NULL auto_increment, " + JIDS_JID
															+ " varchar(2049), primary key (" + JIDS_ID + ")); ";
//	private static final String GET_MESSAGES = "select " + MSGS_MSG + " from " +
//																						 MSGS_TABLE + " where " + MSGS_OWNER_ID +
//																						 " = ? and " + MSGS_BUDDY_ID + " = ?" +
//																						 " and date(" + MSGS_TIMESTAMP +
//																						 ") = date(?)" + " order by " +
//																						 MSGS_TIMESTAMP + " limit ? offset ?";
//	private static final String DERBY_GET_MESSAGES = "select " + MSGS_MSG + " from " +
//																						 MSGS_TABLE + " where " + MSGS_OWNER_ID +
//																						 " = ? and " + MSGS_BUDDY_ID + " = ?" +
//																						 " and date(" + MSGS_TIMESTAMP +
//																						 ") = ?" + " order by " +
//																						 MSGS_TIMESTAMP + " offset ? rows fetch next ? rows only";
	private static final String GET_MESSAGES = "select " + MSGS_MSG + " from " + MSGS_TABLE + " where " + MSGS_OWNER_ID + " = ? and " 
			+ MSGS_BUDDY_ID + " = ? and " + MSGS_TIMESTAMP + " >= ? order by " + MSGS_TIMESTAMP;
	private static final String GET_MESSAGES_END = "select " + MSGS_MSG + " from " + MSGS_TABLE + " where " + MSGS_OWNER_ID + " = ? and " 
			+ MSGS_BUDDY_ID + " = ? and " + MSGS_TIMESTAMP + " >= ? and " + MSGS_TIMESTAMP + " <= ? order by " + MSGS_TIMESTAMP;
	private static final String GET_MESSAGES_COUNT = "select count(" + MSGS_TIMESTAMP + ") from " + MSGS_TABLE + " where " + MSGS_OWNER_ID 
			+ " = ? and " + MSGS_BUDDY_ID + " = ? and " + MSGS_TIMESTAMP + " >= ?";
	private static final String GET_MESSAGES_END_COUNT = "select count(" + MSGS_TIMESTAMP + ") from " + MSGS_TABLE + " where " 
			+ MSGS_OWNER_ID + " = ? and " + MSGS_BUDDY_ID + " = ? and " + MSGS_TIMESTAMP + " >= ? and " + MSGS_TIMESTAMP + " <= ?";
//	private static final String GET_COLLECTIONS = "select distinct date(" +
//																								MSGS_TIMESTAMP + ")" + " from " +
//																								MSGS_TABLE + " where " + MSGS_OWNER_ID +
//																								" = ?" + " and " + MSGS_BUDDY_ID +
//																								" = ? and " + MSGS_TIMESTAMP + " <= ?" +
//																								" and " + MSGS_TIMESTAMP + " >= ?" +
//																								" order by date(" + MSGS_TIMESTAMP + ")";
	private static final String GET_COLLECTIONS_SELECT = "select min(m." + MSGS_TIMESTAMP + "), j." + JIDS_JID + " from " + MSGS_TABLE + " m "
			+ "inner join " + JIDS_TABLE + " j on m." + MSGS_BUDDY_ID + " = j." + JIDS_ID + " where m." + MSGS_OWNER_ID + " = ? ";
	private static final String GET_COLLECTIONS_SELECT_GROUP = "group by date(m." + MSGS_TIMESTAMP + "), m." + MSGS_BUDDY_ID + ", j." + JIDS_JID 
			+ " order by min(m." + MSGS_TIMESTAMP + "), j." + JIDS_JID;
	private static final String GET_COLLECTIONS_COUNT = "select count(1) from (select min(m." + MSGS_TIMESTAMP + "), m." + MSGS_BUDDY_ID + " from " 
			+ MSGS_TABLE + " m where m." + MSGS_OWNER_ID + " = ? ";
	private static final String GET_COLLECTIONS_COUNT_GROUP = "group by date(m." + MSGS_TIMESTAMP + "), m." + MSGS_BUDDY_ID + ") x";
	private static final String GENERIC_LIMIT = " limit ? offset ?";
	private static final String DERBY_LIMIT = " offset ? rows fetch next ? rows only";
	private static final String[][] GET_COLLECTIONS_WHERES = { 
			{ "FROM", "and m." + MSGS_TIMESTAMP + " >= ? " },
			{ "TO", "and m." + MSGS_TIMESTAMP + " <= ? " },
			{ "WITH", "and m." + MSGS_BUDDY_ID + " = ? " }
	};
	private static final String[] GET_COLLECTIONS_COMBINATIONS = {
		"FROM", "FROM_TO", "FROM_TO_WITH", "FROM_WITH",
		"TO", "TO_WITH", "WITH"
	};
//	private static final String GET_COLLECTIONS_PGSQL = "select min(ts),j.jid from tig_ma_msgs m inner join tig_ma_jids j on m.buddy_id = j.jid_id "
//			+ "where owner_id = 4 and ts >= '2013-11-13' "
//			+ "group by floor(EXTRACT(EPOCH FROM (ts - cast('2013-11-12T22:00:00-0000' as timestamp with time zone)))/(60*60*24)), buddy_id, j.jid;"
//	private static final String GET_COLLECTIONS_MYSQL = "select min(ts), j.jid from tig_ma_msgs m "
//			+ "inner join tig_ma_jids j on m.buddy_id = j.jid_id where owner_id = 4 group by datediff(ts,'2013-11-01T00:00:00'), buddy_id"
	private static final String ADD_MESSAGE = "insert into " + MSGS_TABLE + " (" +
																						MSGS_OWNER_ID + ", " + MSGS_BUDDY_ID + ", " +
																						MSGS_TIMESTAMP + ", " + MSGS_DIRECTION +
																						", " + MSGS_TYPE + ", " + MSGS_MSG + ")" +
																						" values (?, ?, ?, ?, ?, ?)";
	private static final String REMOVE_MSGS = "delete from " + MSGS_TABLE + " where " +
																						MSGS_OWNER_ID + " = ? and " + MSGS_BUDDY_ID +
																						" = ?" + " and " + MSGS_TIMESTAMP +
																						" <= ? and " + MSGS_TIMESTAMP + " >= ?";
	private static final String DERBY_CREATE_MSGS = "create table " + MSGS_TABLE + " ("
																									+ MSGS_OWNER_ID + " bigint references " + JIDS_TABLE + "(" + JIDS_ID + "),"
																									+ MSGS_BUDDY_ID + " bigint references " + JIDS_TABLE + "(" + JIDS_ID + "),"
																									+ MSGS_TIMESTAMP + " timestamp, "
																									+ MSGS_DIRECTION + " smallint, "
																									+ MSGS_TYPE + " varchar(10), "
																									+ MSGS_MSG + " varchar(32672));"
																									+ "create index " + MSGS_TABLE + "_" + MSGS_OWNER_ID + "_index on " + MSGS_TABLE
																									+ " (" + MSGS_OWNER_ID + ");"
																									+ "create index " + MSGS_TABLE + "_" + MSGS_OWNER_ID + "_" + MSGS_BUDDY_ID
																									+ "_index on " + MSGS_TABLE + " (" + MSGS_OWNER_ID + ", " + MSGS_BUDDY_ID + ");"
																									+ "create index " + MSGS_TABLE + "_" + MSGS_OWNER_ID + "_" + MSGS_TIMESTAMP + "_"
																									+ MSGS_BUDDY_ID + "_index on " + MSGS_TABLE + " (" + MSGS_OWNER_ID + ", "
																									+ MSGS_TIMESTAMP + ", " + MSGS_BUDDY_ID + ");";
	private static final String PGSQL_CREATE_MSGS = "create table " + MSGS_TABLE + " (" +
																									MSGS_OWNER_ID + " bigint, " +
																									MSGS_BUDDY_ID + " bigint, " +
																									MSGS_TIMESTAMP + " timestamp, " +
																									MSGS_DIRECTION + " smallint, " +
																									MSGS_TYPE + " varchar(10)," +
																									MSGS_MSG + " text," +
																									" foreign key (" + MSGS_BUDDY_ID +
																									") references " + JIDS_TABLE + " (" +
																									JIDS_ID + ")," + " foreign key (" +
																									MSGS_OWNER_ID + ") references " +
																									JIDS_TABLE + " (" + JIDS_ID + ") ); " +
																									"create index " + MSGS_TABLE + "_" +
																									MSGS_OWNER_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									"); " + "create index " + MSGS_TABLE +
																									"_" + MSGS_OWNER_ID + "_" +
																									MSGS_BUDDY_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									", " + MSGS_BUDDY_ID + "); " +
																									"create index " + MSGS_TABLE + "_" +
																									MSGS_OWNER_ID + "_" + MSGS_TIMESTAMP +
																									"_" + MSGS_BUDDY_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									", " + MSGS_TIMESTAMP + ", " + 
																									MSGS_BUDDY_ID + "); ";
	private static final String SQLSERVER_CREATE_MSGS = "create table " + MSGS_TABLE + " (" +
																									MSGS_OWNER_ID + " bigint, " +
																									MSGS_BUDDY_ID + " bigint, " +
																									MSGS_TIMESTAMP + " datetime, " +
																									MSGS_DIRECTION + " smallint, " +
																									MSGS_TYPE + " nvarchar(10)," +
																									MSGS_MSG + " ntext," +
																									" foreign key (" + MSGS_BUDDY_ID +
																									") references " + JIDS_TABLE + " (" +
																									JIDS_ID + ")," + " foreign key (" +
																									MSGS_OWNER_ID + ") references " +
																									JIDS_TABLE + " (" + JIDS_ID + ") ); " +
																									"create index " + MSGS_TABLE + "_" +
																									MSGS_OWNER_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									"); " + "create index " + MSGS_TABLE +
																									"_" + MSGS_OWNER_ID + "_" +
																									MSGS_BUDDY_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									", " + MSGS_BUDDY_ID + "); " +
																									"create index " + MSGS_TABLE + "_" +
																									MSGS_OWNER_ID + "_" + MSGS_TIMESTAMP +
																									"_" + MSGS_BUDDY_ID + "_index on " +
																									MSGS_TABLE + " ( " + MSGS_OWNER_ID +
																									", " + MSGS_TIMESTAMP + ", " +
																									MSGS_BUDDY_ID + "); ";
	private static final String MYSQL_CREATE_MSGS = "create table " + MSGS_TABLE + " (" +
																									MSGS_OWNER_ID + " bigint unsigned, " +
																									MSGS_BUDDY_ID + " bigint unsigned, " +
																									MSGS_TIMESTAMP + " timestamp, " +
																									MSGS_DIRECTION + " smallint, " +
																									MSGS_TYPE + " varchar(10)," +
																									MSGS_MSG + " text," +
																									" foreign key (" + MSGS_BUDDY_ID +
																									") references " + JIDS_TABLE + " (" +
																									JIDS_ID + ")," + " foreign key (" +
																									MSGS_OWNER_ID + ") references " +
																									JIDS_TABLE + " (" + JIDS_ID + "), " +
																									"key (" + MSGS_OWNER_ID + "), " + 
																									"key (" + MSGS_OWNER_ID + ", " + 
																									MSGS_BUDDY_ID + "), key (" + 
																									MSGS_OWNER_ID + ", " + MSGS_TIMESTAMP + 
																									", " + MSGS_BUDDY_ID + "));";

	//~--- fields ---------------------------------------------------------------

	private DataRepository data_repo = null;

	//~--- methods --------------------------------------------------------------

	/**
	 * Method description
	 *
	 *
	 * @param conn_str
	 * @param params
	 *
	 * @throws SQLException
	 */
	public void initRepository(String conn_str, Map<String, String> params)
					throws SQLException {
		synchronized(formatter) {
			formatter.setTimeZone(TimeZone.getTimeZone("UTC"));
		}
		try {
			data_repo = RepositoryFactory.getDataRepository( null, conn_str, params );

			// create tables if not exist
			switch ( data_repo.getDatabaseType() ) {
				case mysql:
					data_repo.checkTable( JIDS_TABLE, MYSQL_CREATE_JIDS );
					data_repo.checkTable( MSGS_TABLE, MYSQL_CREATE_MSGS );
					break;
				case derby:
					data_repo.checkTable( JIDS_TABLE, DERBY_CREATE_JIDS );
					data_repo.checkTable( MSGS_TABLE, DERBY_CREATE_MSGS );
					break;
				case postgresql:
					data_repo.checkTable( JIDS_TABLE, PGSQL_CREATE_JIDS );
					data_repo.checkTable( MSGS_TABLE, PGSQL_CREATE_MSGS );
					break;
				case jtds:
				case sqlserver:
					data_repo.checkTable( JIDS_TABLE, SQLSERVER_CREATE_JIDS );
					data_repo.checkTable( MSGS_TABLE, SQLSERVER_CREATE_MSGS );
					break;
			}
			data_repo.initPreparedStatement(ADD_JID_QUERY, ADD_JID_QUERY);
			data_repo.initPreparedStatement(GET_JID_ID_QUERY, GET_JID_ID_QUERY);
			data_repo.initPreparedStatement(GET_JID_IDS_QUERY, GET_JID_IDS_QUERY);

			data_repo.initPreparedStatement(ADD_MESSAGE, ADD_MESSAGE);
			//data_repo.initPreparedStatement(GET_COLLECTIONS, GET_COLLECTIONS);
			
			for (String combination : GET_COLLECTIONS_COMBINATIONS) {
				String[] whereParts = combination.split("_");
				String select = GET_COLLECTIONS_SELECT;
				String count = GET_COLLECTIONS_COUNT;
				for (String part : whereParts) {
					for (String[] where : GET_COLLECTIONS_WHERES) {
						if (!part.equals(where[0]))
							continue;
						
						select += where[1];
						count += where[1];
					}
				}
				select += GET_COLLECTIONS_SELECT_GROUP;
				count += GET_COLLECTIONS_COUNT_GROUP;
				switch ( data_repo.getDatabaseType() ) {
					case derby:
						select += DERBY_LIMIT;
						break;
					default:
						select += GENERIC_LIMIT;
						break;
				}
				data_repo.initPreparedStatement("GET_COLLECTIONS_" + combination + "_SELECT", select);
				data_repo.initPreparedStatement("GET_COLLECTIONS_" + combination + "_COUNT", count);
			}
			
			switch ( data_repo.getDatabaseType() ) {
				case derby:
					data_repo.initPreparedStatement( GET_MESSAGES, GET_MESSAGES + DERBY_LIMIT );
					data_repo.initPreparedStatement( GET_MESSAGES_END, GET_MESSAGES_END + DERBY_LIMIT);
					break;
				default:
					data_repo.initPreparedStatement( GET_MESSAGES, GET_MESSAGES + GENERIC_LIMIT );
					data_repo.initPreparedStatement( GET_MESSAGES_END, GET_MESSAGES_END + GENERIC_LIMIT );
					break;
			}
			data_repo.initPreparedStatement(GET_MESSAGES_COUNT, GET_MESSAGES_COUNT);
			data_repo.initPreparedStatement(GET_MESSAGES_END_COUNT, GET_MESSAGES_END_COUNT);
			data_repo.initPreparedStatement(REMOVE_MSGS, REMOVE_MSGS);
		} catch (Exception ex) {
			log.log(Level.WARNING, "MessageArchiveDB initialization exception", ex);
		}
	}

	/**
	 * Method description
	 *
	 *
	 * @param owner
	 * @param buddy
	 * @param direction
	 * @param msg
	 */
	public void archiveMessage(BareJID owner, BareJID buddy, short direction, Element msg) {
		try {
			String owner_str         = owner.toString();
			String buddy_str         = buddy.toString();
			long[] jids_ids          = getJidsIds(owner_str, buddy_str);
			long owner_id            = (jids_ids[0] != LONG_NULL)
																 ? jids_ids[0]
																 : addJidId(owner_str);
			long buddy_id            = (jids_ids[1] != LONG_NULL)
																 ? jids_ids[1]
																 : addJidId(buddy_str);
			java.sql.Timestamp mtime = null;
			Element delay            = msg.findChildStaticStr(Message.MESSAGE_DELAY_PATH);

			if (delay != null) {
				try {
					String stamp = delay.getAttributeStaticStr("stamp");
					mtime = parseTimestamp(stamp);
				} catch (ParseException e1) {}
			} else {
				mtime = new java.sql.Timestamp(System.currentTimeMillis());
			}
			msg.addAttribute("time", String.valueOf(mtime.getTime()));

			String type                      = msg.getAttributeStaticStr("type");
			String msgStr                    = msg.toString();
			PreparedStatement add_message_st = data_repo.getPreparedStatement(owner,
																					 ADD_MESSAGE);

			synchronized (add_message_st) {
				add_message_st.setLong(1, owner_id);
				add_message_st.setLong(2, buddy_id);
				add_message_st.setTimestamp(3, mtime);
				add_message_st.setShort(4, direction);
				add_message_st.setString(5, type);
				add_message_st.setString(6, msgStr);
				add_message_st.executeUpdate();
			}
		} catch (SQLException ex) {
			log.log(Level.WARNING, "Problem adding new entry to DB: {0}", msg);
		} finally {

//    data_repo.release(null, rs);
		}
	}

	//~--- get methods ----------------------------------------------------------

	/**
	 * Method description
	 *
	 *
	 * @param owner
	 * @param withJid
	 * @param start
	 * @param end
	 * @param rsm
	 *
	 * @return
	 *
	 * @throws SQLException
	 */
	public List<Element> getCollections(BareJID owner, String withJid, Date start,
					Date end, RSM rsm)
					throws SQLException {
		
		long[] jids_ids      = withJid == null ? getJidsIds(owner.toString()) : getJidsIds(owner.toString(), withJid);

		Long buddyId = jids_ids.length > 1 ? jids_ids[1] : null;
		java.sql.Timestamp start_ = start != null ? new java.sql.Timestamp(start.getTime()) : null;
		java.sql.Timestamp end_ = end != null ? new java.sql.Timestamp(end.getTime()) : null;
		int index = rsm.getIndex() == null ? 0 : rsm.getIndex();

		StringBuilder query = new StringBuilder(20);
		if (start_ != null) {
			query.append("FROM");
		}
		if (end_ != null) {
			if (query.length() > 0) {
				query.append("_");
			}
			query.append("TO");
		}
		if (buddyId != null) {
			if (query.length() > 0) {
				query.append("_");
			}
			query.append("WITH");
		} else {
			// not supported
		}
		String queryStr = query.toString();		
		
		List<Element> results = getCollections(owner, jids_ids[0], buddyId, start_, end_, index, rsm.getMax(), queryStr);
		Integer count = getCollectionsCount(owner, jids_ids[0], buddyId, start_, end_, queryStr);
				
		rsm.setResults(count, index);
		
		return results;
	}

	/**
	 * Method description
	 *
	 *
	 * @param owner
	 * @param withJid
	 * @param start
	 * @param limit
	 * @param offset
	 *
	 * @return
	 *
	 * @throws SQLException
	 */
	public List<Element> getItems(BareJID owner, String withJid, Date start, Date end, RSM rsm)
					throws SQLException {
		long[] jids_ids   = getJidsIds(owner.toString(), withJid);

		Timestamp startTimestamp = new Timestamp(start.getTime());
		Timestamp endTimestamp = end != null ? new Timestamp(end.getTime()) : null;

		int offset = rsm.getIndex() != null ? rsm.getIndex() : 0;
		int limit = rsm.getMax();
		
		List<Element> items = getItems(owner, jids_ids[0], jids_ids[1], startTimestamp, 
				endTimestamp, offset, limit);
		
		int count = getItemsCount(owner, jids_ids[0], jids_ids[1], startTimestamp, endTimestamp);
		rsm.setResults(count, offset);
		
		return items;
	}

	//~--- methods --------------------------------------------------------------

	/**
	 * Method description
	 *
	 *
	 * @param owner
	 * @param withJid
	 * @param start
	 * @param end
	 *
	 * @throws SQLException
	 */
	public void removeItems(BareJID owner, String withJid, Date start, Date end)
					throws SQLException {
		long[] jids_ids = getJidsIds(owner.toString(), withJid);

		if (start == null) {
			start = new Date(0);
		}
		if (end == null) {
			end = new Date(0);
		}

		java.sql.Timestamp start_        = new java.sql.Timestamp(start.getTime());
		java.sql.Timestamp end_          = new java.sql.Timestamp(end.getTime());
		PreparedStatement remove_msgs_st = data_repo.getPreparedStatement(owner, REMOVE_MSGS);

		synchronized (remove_msgs_st) {
			synchronized (remove_msgs_st) {
				remove_msgs_st.setLong(1, jids_ids[0]);
				remove_msgs_st.setLong(2, jids_ids[1]);
				remove_msgs_st.setTimestamp(3, end_);
				remove_msgs_st.setTimestamp(4, start_);
				remove_msgs_st.executeUpdate();
			}
		}
	}

	private List<Element> getCollections(BareJID owner,long ownerId, Long buddyId, Timestamp start,
					Timestamp end, int index, int limit, String queryStr)
					throws SQLException {
		List<Element> results = new LinkedList<Element>();
		ResultSet selectRs   = null;
		try {
			PreparedStatement get_collections_st = data_repo.getPreparedStatement(owner, "GET_COLLECTIONS_" 
					+ queryStr + "_SELECT");

			int i=2;
			synchronized (get_collections_st) {
				get_collections_st.setLong(1, ownerId);
				if (start != null) {
					get_collections_st.setTimestamp(i++, start);
				}
				if (end != null) {
					get_collections_st.setTimestamp(i++, end);
				}
				if (buddyId != null) {
					get_collections_st.setLong(i++, buddyId);
				}
				
				switch (data_repo.getDatabaseType()) {
					case derby:
						get_collections_st.setInt(i++, index);						
						get_collections_st.setInt(i++, limit);
						break;
					default:
						get_collections_st.setInt(i++, limit);
						get_collections_st.setInt(i++, index);
						break;
				}
				selectRs = get_collections_st.executeQuery();
				while (selectRs.next()) {
					Timestamp startTs = selectRs.getTimestamp(1);
					String with = selectRs.getString(2);
					String formattedStart = null;
					synchronized (formatter2) {
						formattedStart = formatter2.format(startTs);
					}
					results.add(new Element("chat", new String[] { "with", "start" },
							new String[] { with, formattedStart }));
				}
			}
		} finally {
			data_repo.release(null, selectRs);
		}
		return results;
	}	
	
	private Integer getCollectionsCount(BareJID owner, long ownerId, Long buddyId, Timestamp start_, 
			Timestamp end_, String queryStr) throws SQLException {
		ResultSet countRs = null;		
		Integer count = null;
		try {
			PreparedStatement get_collections_count = data_repo.getPreparedStatement(owner, "GET_COLLECTIONS_" 
					+ queryStr + "_COUNT");
			int i=2;
			synchronized (get_collections_count) {
				get_collections_count.setLong(1, ownerId);
				if (start_ != null) {
					get_collections_count.setTimestamp(i++, start_);
				}
				if (end_ != null) {
					get_collections_count.setTimestamp(i++, end_);
				}
				if (buddyId != null) {
					get_collections_count.setLong(i++, buddyId);
				}				
				countRs = get_collections_count.executeQuery();
				if (countRs.next()) {
					count = countRs.getInt(1);
				}
			}
		} finally {
			data_repo.release(null, countRs);
		}
		return count;
	}
	
	private List<Element> getItems(BareJID owner, long ownerId, long withId, Timestamp startTimestamp, 
			Timestamp endTimestamp, int offset, int limit) throws SQLException {
		ResultSet rs      = null;		
		StringBuilder buf = new StringBuilder(16 * 1024);
		int i=1;
		try {
			PreparedStatement get_messages_st = data_repo.getPreparedStatement(owner, endTimestamp != null 
					? GET_MESSAGES_END : GET_MESSAGES);
			synchronized (get_messages_st) {
				get_messages_st.setLong(i++, ownerId);
				get_messages_st.setLong(i++, withId);
				get_messages_st.setTimestamp(i++, startTimestamp);

				if (endTimestamp != null) {
					get_messages_st.setTimestamp(i++, endTimestamp);
				}
				switch (data_repo.getDatabaseType()) {
					case derby:
						get_messages_st.setInt(i++, offset);
						get_messages_st.setInt(i++, limit);
						break;
					default:
						get_messages_st.setInt(i++, limit);
						get_messages_st.setInt(i++, offset);
						break;
				}

				rs = get_messages_st.executeQuery();
				while (rs.next()) {
					buf.append(rs.getString(1));
				}
			}
		} finally {
			data_repo.release(null, rs);
		}

		List<Element> msgs = null;

		if (buf != null) {
			String results = buf.toString();

			msgs = new LinkedList<Element>();

			DomBuilderHandler domHandler = new DomBuilderHandler();

			parser.parse(domHandler, results.toCharArray(), 0, results.length());

			Queue<Element> queue = domHandler.getParsedElements();
			String ownerStr      = owner.toString();
			Element msg          = null;

			while ((msg = queue.poll()) != null) {
				Element item =
					new Element(msg.getAttributeStaticStr(Packet.FROM_ATT).startsWith(ownerStr)
											? Packet.TO_ATT
											: Packet.FROM_ATT);

				item.addChild(msg.getChild("body"));
				item.setAttribute(
						"secs",
						String.valueOf(
							(Long.valueOf(msg.getAttributeStaticStr("time")) - startTimestamp.getTime()) /
							1000));
				msgs.add(item);
			}
			Collections.sort(msgs, new Comparator<Element>() {
				@Override
				public int compare(Element m1, Element m2) {
					return m1.getAttributeStaticStr("secs").compareTo(
							m2.getAttributeStaticStr("secs"));
				}
			});
		}

		return msgs;		
	}

	private Integer getItemsCount(BareJID owner, long ownerId, long withId, Timestamp startTimestamp, 
			Timestamp endTimestamp) throws SQLException {
		ResultSet rs      = null;		
		Integer count = null;
		int i=1;
		try {
			PreparedStatement get_messages_st = data_repo.getPreparedStatement(owner, endTimestamp != null 
					? GET_MESSAGES_END_COUNT : GET_MESSAGES_COUNT);
			synchronized (get_messages_st) {
				get_messages_st.setLong(i++, ownerId);
				get_messages_st.setLong(i++, withId);
				get_messages_st.setTimestamp(i++, startTimestamp);

				if (endTimestamp != null) {
					get_messages_st.setTimestamp(i++, endTimestamp);
				}

				rs = get_messages_st.executeQuery();
				if (rs.next()) {
					count = rs.getInt(1);
				}
			}
		} finally {
			data_repo.release(null, rs);
		}
		return count;
	}
	
	/**
	 * Method description
	 *
	 *
	 * @param jids
	 *
	 * @return
	 *
	 * @throws SQLException
	 */
	protected long[] getJidsIds(String... jids) throws SQLException {
		ResultSet rs = null;

		try {
			long[] results = new long[jids.length];

			Arrays.fill(results, LONG_NULL);
			if (jids.length == 1) {
				PreparedStatement get_jid_id_st = data_repo.getPreparedStatement(null,
																						GET_JID_ID_QUERY);

				synchronized (get_jid_id_st) {
					get_jid_id_st.setString(1, jids[0]);
					rs = get_jid_id_st.executeQuery();
					if (rs.next()) {
						results[0] = rs.getLong("jid_id");

						return results;
					}
				}

				return null;
			} else {
				PreparedStatement get_jids_id_st = data_repo.getPreparedStatement(null,
																						 GET_JID_IDS_QUERY);

				synchronized (get_jids_id_st) {
					for (int i = 0; i < jids.length; i++) {
						get_jids_id_st.setString(i + 1, jids[i]);
					}
					rs = get_jids_id_st.executeQuery();

					int cnt = 0;

					while (rs.next()) {
						String db_jid = rs.getString("jid");

						for (int i = 0; i < jids.length; i++) {
							if (db_jid.equals(jids[i])) {
								results[i] = rs.getLong("jid_id");
								++cnt;
							}
						}
					}

					return results;
				}
			}
		} finally {
			data_repo.release(null, rs);
		}
	}

	//~--- methods --------------------------------------------------------------

	private long addJidId(String jid) throws SQLException {
		PreparedStatement add_jid_st = data_repo.getPreparedStatement(null, ADD_JID_QUERY);

		synchronized (add_jid_st) {
			add_jid_st.setString(1, jid);
			add_jid_st.executeUpdate();
		}

		// This is not the most effective solution but this method shouldn't be
		// called very often so the perfrmance impact should be insignificant.
		long[] jid_ids = getJidsIds(jid);

		if (jid_ids != null) {
			return jid_ids[0];
		} else {

			// That should never happen here, but just in case....
			log.log(Level.WARNING, "I have just added new jid but it was not found.... {0}",
							jid);

			return LONG_NULL;
		}
	}
	
	private Timestamp parseTimestamp(String tmp) throws ParseException {
		Date date = null;
		
		if (tmp.endsWith("Z")) {
			if (tmp.contains(".")) {
				synchronized(formatter4) {
					date = formatter4.parse(tmp);
				}
			}
			else {
				synchronized(formatter) {
					date = formatter.parse(tmp);
				}
			}
		}
		else if (tmp.contains(".")) {
			synchronized(formatter3) {
				date = formatter3.parse(tmp);
			}			
		}
		else {
			synchronized(formatter2) {
				date = formatter2.parse(tmp);
			}			
		}
		
		return new Timestamp(date.getTime());
	}
	
}


//~ Formatted in Tigase Code Convention on 13/02/20
