/*
 * Tigase ACS - Tigase Advanced Clustering Strategy
 * Copyright (C) 2004 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.server.cluster.strategy;

import tigase.cluster.api.ClusterControllerIfc;
import tigase.cluster.api.SessionManagerClusteredIfc;
import tigase.cluster.strategy.DefaultClusteringStrategyAbstract;
import tigase.licence.LicenceChecker;
import tigase.licence.LicenceCheckerUpdateCallback;
import tigase.licence.LicencedComponent;
import tigase.licence.callbacks.LicenceCheckerUpdateCallbackImplACS;
import tigase.server.Packet;
import tigase.server.Presence;
import tigase.server.cluster.strategy.cmd.*;
import tigase.stats.MaxDailyCounterQueue;
import tigase.stats.StatisticsList;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.Element;
import tigase.xmpp.NoConnectionIdException;
import tigase.xmpp.NotAuthorizedException;
import tigase.xmpp.StanzaType;
import tigase.xmpp.XMPPResourceConnection;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import static tigase.licence.LicenceChecker.INSTALLATION_ID_KEY;
import static tigase.server.cluster.strategy.cmd.CachingCmdAbstract.*;

/**
 * Created: Jun 27, 2009 9:42:40 PM
 *
 * @author <a href="mailto:artur.hefczyc@tigase.org">Artur Hefczyc</a>
 * @version $Rev$
 */
public class OnlineUsersCachingStrategy
		extends DefaultClusteringStrategyAbstract<ConnectionRecordExt>
		implements LicencedComponent {

	public static final String ONLINE_PRESENCE_CACHE_PROP_KEY = "cluster-strategy-presence-cache";

	public static final String REQUEST_SYNCONLINE_CMD = "req-sync-online-sm-cmd";

	public static final String RESPOND_SYNCONLINE_CMD = "resp-sync-online-sm-cmd";

	public static final String STRATEGY_STATS_CMD = "strategy-statistics";

	public static final String USER_CONNECTED_CMD = "user-connected-sm-cmd";

	public static final String USER_CONN_ID_CHANGED_CMD = "user-conn-id-changed-sm-cmd";

	public static final String USER_DISCONNECTED_CMD = "user-disconnected-sm-cmd";

	public static final String USER_PRESENCE_CMD = "user-presence-sm-cmd";

	private static final String AUTH_TIME = "auth-time";
	private static final String INITIAL_PRESENCE_KEY = "cluster-initial-presence";
	private static final String PRESENCE_TYPE_UPDATE = "update";
	private static final String COMPONENT_ID = "acs";

	//~--- fields ---------------------------------------------------------------
	private static final Logger log = Logger.getLogger(OnlineUsersCachingStrategy.class.getName());
	protected static LicenceChecker licenceChecker;
	private CacheContenerV2 cache = new CacheContenerV2();
	private boolean cachePresences = false;
	private MaxDailyCounterQueue<Integer> maxClusterUsersConnections = new MaxDailyCounterQueue<>(31);
	private int maxClusterUsersConnectionsWithinLastWeek = 0;
	private MaxDailyCounterQueue<Integer> maxClusterUsersSessions = new MaxDailyCounterQueue<>(31);
	private int maxClusterUsersSessionsWithinLastWeek = 0;

	//~--- constructors ---------------------------------------------------------

	public OnlineUsersCachingStrategy() {
		super();
		addCommandListener(new RequestSyncOnlineCmd(REQUEST_SYNCONLINE_CMD, this));
		addCommandListener(new RespondSyncOnlineCmd(RESPOND_SYNCONLINE_CMD, this));
		addCommandListener(new UserConnectedCmd(USER_CONNECTED_CMD, this));
		addCommandListener(new UserDisconnectedCmd(USER_DISCONNECTED_CMD, this));
		addCommandListener(new UserPresenceCmd(USER_PRESENCE_CMD, this));
		addCommandListener(new UserConnIdChangedCmd(USER_CONN_ID_CHANGED_CMD, this));
		addCommandListener(new TrafficStatisticsCmd(STRATEGY_STATS_CMD, this));

		LicenceCheckerUpdateCallback upCall = new LicenceCheckerUpdateCallbackImplACS(COMPONENT_ID, this);
		licenceChecker = LicenceChecker.getLicenceChecker(COMPONENT_ID, upCall);
	}

	//~--- methods --------------------------------------------------------------
	@Override
	public boolean containsJid(BareJID jid) {
		return cache.containsJid(jid);
	}

	@Override
	public void handleLocalPacket(Packet packet, XMPPResourceConnection conn) {
		if (packet.getElemName() == Presence.ELEM_NAME) {
			try {
				if (packet.getType() != null) {
					switch (packet.getType()) {
						case subscribe:
						case subscribed:
						case unsubscribe:
						case unsubscribed:
							return;
						default:
							break;
					}
				}

				boolean initPresence = conn.getSessionData(INITIAL_PRESENCE_KEY) == null;
				Map<String, String> params = prepareConnectionParams(conn);

				if (initPresence) {
					conn.putSessionData(INITIAL_PRESENCE_KEY, INITIAL_PRESENCE_KEY);
					params.put(PRESENCE_TYPE_KEY, PRESENCE_TYPE_INITIAL);
				} else {
					params.put(PRESENCE_TYPE_KEY, PRESENCE_TYPE_UPDATE);
				}

				// this below is not atomic, so other thread could 
				// change value of conn.getPresence() to be null!!
				// Element	  presence = conn.getPresence() != null 
				//					? conn.getPresence() : packet.getElement();
				//
				// code below is not only not atomic but uses conn.getPresence()
				// when "this" presence packet was not processed as it is done
				// in SM in other thread
//				Element presence = conn.getPresence();
//				if ( presence == null ){
//					presence = packet.getElement();
//				}

				// using following code instead, which should be always correct
				Element presence = packet.getElement();

				List<JID> cl_nodes = getNodesForPacketForward(sm.getComponentId(), null,
															  Packet.packetInstance(presence));

				if ((cl_nodes != null) && (cl_nodes.size() > 0)) {

					// ++clusterSyncOutTraffic;
					cluster.sendToNodes(USER_PRESENCE_CMD, params, presence, sm.getComponentId(), null,
										cl_nodes.toArray(new JID[cl_nodes.size()]));
				}
			} catch (NotAuthorizedException e) {
				log.log(Level.FINE, "Problem with broadcast user presence for: " + conn, e);
			} catch (NoConnectionIdException | TigaseStringprepException e) {
				log.log(Level.WARNING, "Problem with broadcast user presence for: " + conn, e);
			}
		}
	}

	@Override
	public void handleLocalUserLogout(BareJID userId, XMPPResourceConnection conn) {
		try {
			Element presence = conn.getPresence();
			if (presence != null) {
				if (StanzaType.unavailable.name().equals(presence.getAttributeStaticStr("type"))) {
					presence = null;
				} else {
					presence = presence.clone();
					presence.setAttribute("from", conn.getJID().toString());
					presence.setAttribute("type", StanzaType.unavailable.name());
				}
			}

			Map<String, String> params = prepareConnectionParams(conn);
			List<JID> cl_nodes = getNodesForUserDisconnect(conn.getJID());

			// ++clusterSyncOutTraffic;
			if (presence == null) {
				cluster.sendToNodes(USER_DISCONNECTED_CMD, params, sm.getComponentId(),
									cl_nodes.toArray(new JID[cl_nodes.size()]));
			} else {
				cluster.sendToNodes(USER_DISCONNECTED_CMD, params, presence, sm.getComponentId(), null,
									cl_nodes.toArray(new JID[cl_nodes.size()]));
			}
		} catch (NotAuthorizedException ex) {
			log.log(Level.INFO,
					"NotAuthorizedException: This should really not happen as it is called for authenticated users, maybe the session was removed in the meantime. BareJID: {0}, conn: {1}",
					new Object[]{userId, conn});
		} catch (NoConnectionIdException ex) {
			log.log(Level.INFO, "NoConnectionIdException: This really should not happen. BareJID: {0}, conn: {1}",
					new Object[]{userId, conn});
		}
	}

	@Override
	public void handleLocalResourceBind(XMPPResourceConnection conn) {
		try {
			Map<String, String> params = prepareConnectionParams(conn);
			List<JID> cl_nodes = getNodesForUserConnect(conn.getJID());

			// ++clusterSyncOutTraffic;
			cluster.sendToNodes(USER_CONNECTED_CMD, params, sm.getComponentId(),
								cl_nodes.toArray(new JID[cl_nodes.size()]));
		} catch (NotAuthorizedException | NoConnectionIdException e) {
			log.log(Level.WARNING, "Problem with broadcast user presence for: " + conn, e);
		}
	}

	@Override
	public void handleLocalUserChangedConnId(BareJID userId, XMPPResourceConnection conn, JID oldConnId,
											 JID newConnId) {
		try {
			List<JID> cl_nodes = getNodesForUserDisconnect(conn.getJID());

			Map<String, String> params = new LinkedHashMap<String, String>();

			params.put(USER_ID, conn.getBareJID().toString());
			params.put(RESOURCE, conn.getResource());
			params.put(CONNECTION_ID + "_new", newConnId.toString());
			params.put(CONNECTION_ID + "_old", oldConnId.toString());
			params.put(XMPP_SESSION_ID, conn.getSessionId());

			cluster.sendToNodes(USER_CONN_ID_CHANGED_CMD, params, sm.getComponentId(),
								cl_nodes.toArray(new JID[cl_nodes.size()]));
		} catch (NotAuthorizedException ex) {
			log.log(Level.WARNING, "Problem with broadcasting info about changed connection id for:" + conn, ex);
		}
	}

	@Override
	public void nodeConnected(JID node) {
		super.nodeConnected(node);
		requestSync(node);
	}

	@Override
	public void nodeDisconnected(JID node) {
		super.nodeDisconnected(node);
		cache.removeAllForNode(node);
	}

	/**
	 * If presence caching is enabled method updates given connection in the cache with the information about presence
	 * for the given connection.
	 *
	 * @param presence {@link Element} containing presence information
	 * @param rec {@link ConnectionRecordExt} for which presence should be updated
	 */
	public void presenceUpdate(Element presence, ConnectionRecordExt rec) {
		if (cachePresences) {
			cache.updatePresence(presence, rec);
		}
	}

	@Override
	public String toString() {
		return getInfo();
	}

	/**
	 * Method responsible for user disconnection events. Calls {@code userDisconnected()} on particular cache
	 * implementation.
	 *
	 * @param rec {@link ConnectionRecordExt} for which presence should be updated
	 */
	public void userDisconnected(ConnectionRecordExt rec) {
		cache.userDisconnected(rec);
	}

	/**
	 * Method responsible for user connection events. Calls {@code usersConnected()} on particular cache
	 * implementation.
	 *
	 * @param recs {@link ConnectionRecordExt} for which presence should be updated
	 */
	public void usersConnected(ConnectionRecordExt... recs) {
		cache.usersConnected(recs);
	}

	/**
	 * Method responsible for changing connection id for session for which connection id was changed, ie. due to stream
	 * resumption.
	 *
	 * @param userId - bare jid of user
	 * @param oldConnectionId - connection id which was changed
	 * @param newConnectionId - connection id which replaced old connection id
	 */
	public void userChangedConnId(BareJID userId, JID oldConnectionId, JID newConnectionId) {
		cache.userChangedConnId(userId, oldConnectionId, newConnectionId);
	}

	//~--- get methods ----------------------------------------------------------

	/**
	 * Method returns implementation {@link ClusterControllerIfc} currently used.
	 *
	 * @return a value of {@link ClusterControllerIfc}
	 */
	public ClusterControllerIfc getCluster() {
		return cluster;
	}

	@Override
	public JID[] getConnectionIdsForJid(BareJID jid) {
		return cache.getConnectionIdsForJid(jid);
	}

	@Override
	public ConnectionRecordExt getConnectionRecord(JID jid) {
		return cache.getConnectionRecord(jid);
	}

	@Override
	public ConnectionRecordExt getConnectionRecordInstance() {
		return new ConnectionRecordExt();
	}

	@Override
	public Set<ConnectionRecordExt> getConnectionRecords(BareJID bareJID) {
		return cache.getConnectionRecords(bareJID);
	}

	@Override
	public List<ConnectionRecordExt> getConnectionRecordsByCreationTime(BareJID bareJID) {
		return cache.getConnectionRecords(bareJID)
				.stream()
				.filter(Objects::nonNull)
				.sorted(Comparator.comparing(ConnectionRecordExt::getCreationTime))
				.collect(Collectors.toList());
	}

	@Override
	public Map<String, Object> getDefaults(Map<String, Object> params) {
		return super.getDefaults(params);
	}

	@Override
	public String getInfo() {
		return "acs-online-cache strategy";
	}
	
	/**
	 * Retrieves {@link CacheContener} instance used by this ACS instance.
	 *
	 * @return {@link CacheContener} instance used by this ACS instance.
	 */
	public CacheContenerIfc getCacheContener() {
		return cache;
	}

	@Override
	public List<JID> getNodesForPacketForward(JID fromNode, Set<JID> visitedNodes, Packet packet) {

		// If visited nodes is not null then we return null as this strategy never
		// sends packets in ring, the first node decides where to send a packet
		if (visitedNodes != null) {
			return null;
		}

		List<JID> nodes = null;
		JID jidLookup = packet.getStanzaTo();
		boolean presenceUpdate = false;

		// Presence status change set by the user have a special treatment:
		if ((packet.getElemName() == "presence") && (packet.getType() != StanzaType.error) &&
				(packet.getStanzaFrom() != null) && (packet.getStanzaTo() == null)) {
			presenceUpdate = true;
			jidLookup = packet.getStanzaFrom().copyWithoutResource();
		}
		if (presenceUpdate || isSuitableForForward(packet)) {
			if (presenceUpdate && cachePresences) {
				nodes = getNodesConnected();
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST,
							"Presence update and cachePresences on, selecting all nodes: {0}, " + "for packet: {1}",
							new Object[]{nodes, packet});
				}
			} else {
				if (isIqResponseToNode(packet)) {
					nodes = getNodesForIqResponse(packet);
				} else {
					nodes = cache.getNodesForJid(jidLookup);
				}
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "Selected nodes: {0}, for packet: {1}", new Object[]{nodes, packet});
				}
			}
		} else {
			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST, "Packet not suitable for forwarding: {0}", new Object[]{packet});
			}
		}

		return nodes;
	}

	/**
	 * Method returns list of all nodes to which information about user connecting should be sent.
	 *
	 * @param jid {@code JID} of user that has connected
	 *
	 * @return a {@code List} of all nodes {@code JID} to which information about user connecting should be sent.
	 */
	public List<JID> getNodesForUserConnect(JID jid) {
		return getNodesConnected();
	}

	/**
	 * Method returns list of all nodes to which information about user disconnecting should be sent.
	 *
	 * @param jid {@code JID} of user that has disconnected
	 *
	 * @return a {@code List} of all nod {@code JID} to which information about user disconnecting should be sent.
	 */
	public List<JID> getNodesForUserDisconnect(JID jid) {
		return getNodesConnected();
	}

	/**
	 * Method allows retrieval of a particular {@link SessionManagerClusteredIfc} implementation currently used.
	 *
	 * @return {@link SessionManagerClusteredIfc} implementation currently used.
	 */
	public SessionManagerClusteredIfc getSM() {
		return sm;
	}

	@Override
	public void everyMinute() {
		super.everyMinute();
		maxClusterUsersConnections.add(cache.connectionsSize() + getSM().getXMPPResourceConnections().size());
		maxClusterUsersSessions.add(cache.mapSize() + getSM().getXMPPSessions().size() - 1);
		maxClusterUsersConnectionsWithinLastWeek = maxClusterUsersConnections.getMaxValueInRange(7).orElse(-1);
		maxClusterUsersSessionsWithinLastWeek = maxClusterUsersSessions.getMaxValueInRange(7).orElse(-1);
	}

	@Override
	public void getStatistics(StatisticsList list) {
		super.getStatistics(list);
		list.add(comp, prefix + "Cached JIDs", cache.connectionsSize(), Level.INFO);
		list.add(comp, prefix + "Cached connections", cache.mapSize(), Level.INFO);

		list.add(comp, prefix + "Max daily users connections count last month (whole cluster)",
				 maxClusterUsersConnections, Level.FINE);
		list.add(comp, prefix + "Max daily users sessions count last month (whole cluster)", maxClusterUsersSessions,
				 Level.FINE);
		list.add(comp, prefix + "Max users connections within last week (whole cluster)",
				 maxClusterUsersConnectionsWithinLastWeek, Level.INFO);
		list.add(comp, prefix + "Max users sessions within last week, whole cluster",
				 maxClusterUsersSessionsWithinLastWeek, Level.INFO);

		final String installationId = LicenceChecker.getInstallationId();
		if (installationId != null) {
			list.add(comp, prefix + INSTALLATION_ID_KEY, installationId, Level.INFO);
		}
	}

	@Override
	public boolean hasCompleteJidsInfo() {
		return cache.hasCompleteJidsInfo();
	}

	//~--- set methods ----------------------------------------------------------
	@Override
	public void setProperties(Map<String, Object> props) {
		super.setProperties(props);

		String tmp = (String) props.get(ONLINE_PRESENCE_CACHE_PROP_KEY);

		if (tmp != null) {
			cachePresences = Boolean.parseBoolean(tmp);
		}
		log.log(Level.CONFIG, "Presence caching is set to: {0}", cachePresences);
	}

	//~--- methods --------------------------------------------------------------

	/**
	 * A utility method used to prepare a Map of data with user session data before it can be sent over to another
	 * cluster node. This is supposed to contain all the user's session essential information which directly identify
	 * user's resource and network connection. This information allows to detect two different user's connection made
	 * for the same resource. This may happen if both connections are established to different nodes.
	 *
	 * @param conn is user's XMPPResourceConnection for which Map structure is prepare.
	 * <p>
	 * a Map structure with all user's connection essential data.
	 *
	 * @return a value of {@code Map<String,String>}
	 */
	protected Map<String, String> prepareConnectionParams(XMPPResourceConnection conn)
			throws NotAuthorizedException, NoConnectionIdException {
		Map<String, String> params = new LinkedHashMap<String, String>();

		params.put(USER_ID, conn.getBareJID().toString());
		params.put(RESOURCE, conn.getResource());
		params.put(CONNECTION_ID, conn.getConnectionId().toString());
		params.put(XMPP_SESSION_ID, conn.getSessionId());
		params.put(AUTH_TIME, "" + conn.getAuthTime());
		params.put(CREATION_TIME, "" + conn.getCreationTime());
		params.put(LOGIN_TIME, "" + conn.getAuthTime());
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "Called for conn: {0}, result: ", new Object[]{conn, params});
		}

		return params;
	}

	/**
	 * Method evaluates whether given packet is a valid presence update, i.e. is of correct type and has proper
	 * addressing.
	 *
	 * @param packet to be validated
	 *
	 * @return {@code true} if the packet is a valid presence update, {@code false} otherwise
	 */
	protected boolean presenceStatusUpdate(Packet packet) {
		return ((packet.getElemName() == "presence") && (packet.getType() != StanzaType.error) &&
				(packet.getStanzaFrom() != null) && (packet.getStanzaTo() == null));
	}

	/**
	 * Send synchronization request to a given cluster node. In a response the remote node should return a list of JIDs
	 * for online users on this node.
	 *
	 * @param node is a JID of the target cluster node.
	 */
	protected void requestSync(JID node) {

		cache.setSync(false);
		cluster.sendToNodes(REQUEST_SYNCONLINE_CMD, sm.getComponentId(), node);
	}
}

//~ Formatted in Tigase Code Convention on 13/10/15
