/*
 * Tigase ACS - Tigase Advanced Clustering Strategy
 * Copyright (C) 2004 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.server.cluster.strategy;

//~--- non-JDK imports --------------------------------------------------------

import tigase.cluster.strategy.ConnectionRecord;
import tigase.server.Packet;
import tigase.sys.OnlineJidsReporter;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.util.*;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.logging.Level;
import java.util.logging.Logger;

//~--- JDK imports ------------------------------------------------------------

/**
 * @author kobit
 */
public class ClusteringMetadata
		implements OnlineJidsReporter {

	private static final Logger log = Logger.getLogger(ClusteringMetadata.class.getName());

	//~--- fields ---------------------------------------------------------------
	protected final Map<BareJID, Map<JID, ConnectionRecordExt>> userConnections = new ConcurrentSkipListMap<BareJID, Map<JID, ConnectionRecordExt>>();
	private volatile int connectionsSize = 0;
	private volatile int mapSize = 0;
	private boolean synced = false;

	//~--- methods --------------------------------------------------------------

	public long connectionsSize() {
		return connectionsSize;
	}

	@Override
	public boolean containsJid(BareJID jid) {
		return userConnections.containsKey(jid);
	}

	@Override
	public boolean containsJidLocally(BareJID jid) {
		return false;
	}

	@Override
	public boolean containsJidLocally(JID jid) {
		return false;
	}

	public long mapSize() {
		return mapSize;
	}

	public boolean needsSync() {
		return !synced;
	}

	public void removeAllForNode(JID node) {
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "Removing connections from cache for node: {0}", node);
		}
		for (Map<JID, ConnectionRecordExt> conns : userConnections.values()) {
			synchronized (conns) {
				for (Iterator<Map.Entry<JID, ConnectionRecordExt>> it = conns.entrySet().iterator(); it.hasNext(); ) {
					Map.Entry<JID, ConnectionRecordExt> entry = it.next();

					if (node.equals(entry.getValue().getNode())) {
						it.remove();
						--connectionsSize;
						if (log.isLoggable(Level.FINEST)) {
							log.log(Level.FINEST, "Removed connection: {0}", entry.getValue());
						}
					} else {
						if (log.isLoggable(Level.FINEST)) {
							log.log(Level.FINEST, "Different node: {0}", entry.getValue());
						}
					}
				}
			}
		}
	}

	public void userDisconnected(Queue<Packet> results, ConnectionRecordExt rec) {
		Map<JID, ConnectionRecordExt> conns = userConnections.get(rec.getUserJid().getBareJID());

		if (conns != null) {
			synchronized (conns) {
				if (conns.remove(rec.getConnectionId()) != null) {
					--connectionsSize;
				}

				// TODO: if the conns.size() == 0 now we could in theory remove it from
				// the Map, however, if there is another call waiting on the
				// synchronized we may have again problem with concurrent access. I
				// don't want to synchronize on the whole main Map as that would affect
				// overall performance.
				// Now we may have memory usage problems but not before the installation
				// reaches over 10mln online users. Hopefully...
				// Maybe a better approach would be kind of a periodic cleanup process
				// which would lock the whole Map and perform cleanup of empty Sets.
				// Once a day would be enough.
			}
		}
	}

	public void usersConnected(Queue<Packet> results, ConnectionRecordExt... recs) {
		for (ConnectionRecordExt rec : recs) {
			Map<JID, ConnectionRecordExt> conns;

			// This is necessary to properly handle the case if the same user connects
			// at the same time to 2 or more different nodes, then this method can be
			// executed at the same time for both connections. Rare case but possible.
			synchronized (userConnections) {
				conns = userConnections.get(rec.getUserJid().getBareJID());
				if (conns == null) {

					// The initial size seems low, but most of the users make a single
					// connection anyway, with rare cases when users make 2 or more
					// connections.
					conns = new LinkedHashMap<JID, ConnectionRecordExt>(2);
					userConnections.put(rec.getUserJid().getBareJID(), conns);
					++mapSize;
				}
			}
			synchronized (conns) {
				if (conns.put(rec.getConnectionId(), rec) == null) {
					++connectionsSize;
				}
			}
		}
	}

	//~--- get methods ----------------------------------------------------------

	@Override
	public JID[] getConnectionIdsForJid(BareJID jid) {
		JID[] result = null;
		Map<JID, ConnectionRecordExt> conns = userConnections.get(jid);

		if (conns != null) {
			synchronized (conns) {
				int size = conns.size();

				if (size > 0) {
					result = conns.keySet().toArray(new JID[size]);
				}
			}
		}

		return result;
	}

	public ConnectionRecordExt getConnectionRecord(JID jid) {
		ConnectionRecordExt result = null;
		Map<JID, ConnectionRecordExt> conns = userConnections.get(jid.getBareJID());

		if (conns != null) {
			synchronized (conns) {

				// TODO: look at possible performance improvements in using entrySet
				// instead of values, this depends on how the Map implements it
				// internally
				for (ConnectionRecordExt rec : conns.values()) {
					if (jid.equals(rec.getUserJid())) {
						result = rec;

						break;
					}
				}
			}
		}

		return result;
	}

	public Set<ConnectionRecordExt> getConnectionRecords(BareJID bareJID) {
		Set<ConnectionRecordExt> result = null;
		Map<JID, ConnectionRecordExt> conns = userConnections.get(bareJID);

		if (conns != null) {
			synchronized (conns) {
				result = new LinkedHashSet<ConnectionRecordExt>(conns.values());
			}
		}

		return result;
	}

	public Object getInternalData() {
		return userConnections;
	}

	public List<JID> getNodesForJid(JID jid) {
		List<JID> result = null;
		Map<JID, ConnectionRecordExt> conns = userConnections.get(jid.getBareJID());

		if (conns != null) {
			synchronized (conns) {
				int size = conns.size();

				if (size > 0) {
					result = new ArrayList<JID>(size);

					// TODO: look at possible performance improvements in using entrySet
					// instead of values, this depends on how the Map implements it
					// internally
					for (ConnectionRecord rec : conns.values()) {

						// This should be efficient if users normally make a small number of
						// connections to different cluster nodes. This is what usually
						// happens a single user rarely opens more than 2 connections for
						// one account. If both connections are to the same cluster node we
						// avoid here putting the same node twice in the result collection.
						if (!result.contains(rec.getNode())) {
							result.add(rec.getNode());
						}
					}
				}
			}
		}

		return result;
	}

	@Override
	public boolean hasCompleteJidsInfo() {
		return synced;
	}
}

//~ Formatted in Tigase Code Convention on 13/06/22
