/*
 * Tigase ACS - Tigase Advanced Clustering Strategy
 * Copyright (C) 2004 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.server.cluster.strategy;

//~--- non-JDK imports --------------------------------------------------------

import tigase.sys.OnlineJidsReporter;
import tigase.xml.Element;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author Artur Hefczyc Created Mar 15, 2011
 */
public class CacheContenerV2
		implements OnlineJidsReporter, CacheContenerIfc {

	private static final Logger log = Logger.getLogger(CacheContenerV2.class.getName());

	//~--- fields ---------------------------------------------------------------
	private final ConcurrentMap<BareJID, Map<JID, ConnectionRecordExt>> userConnections = new ConcurrentHashMap<>();
	/** Holds number of all CIDs in the cache */
	private volatile int connectionsSize = 0;
	/** Holds number of users in cache */
	private volatile int mapSize = 0;
	/** Holds the synchronisation status of the cache */
	private boolean synced = true;

	//~--- methods --------------------------------------------------------------

	/**
	 * Returns number of all cached connections (all/total CIDs for all connected users)
	 *
	 * @return number of all cached CIDs
	 */
	public int connectionsSize() {
		return connectionsSize;
	}

	@Override
	public boolean containsJid(BareJID jid) {
		return userConnections.containsKey(jid);
	}

	@Override
	public boolean containsJidLocally(BareJID jid) {
		return false;
	}

	@Override
	public boolean containsJidLocally(JID jid) {
		return false;
	}

	/**
	 * Return number of connected users
	 *
	 * @return number of connected users
	 */
	public int mapSize() {
		return mapSize;
	}

	/**
	 * Allow obtaining information whether cache needs synching with other nodes.
	 *
	 * @return {@code true} if cache is out of synch, {@code false} otherwise.
	 */
	public boolean needsSync() {
		return !synced;
	}

	/**
	 * Set the state of synchronisation.
	 *
	 * @param sync boolean value denoting sync state.
	 */
	@Override
	public void setSync(boolean sync) {
		synced = sync;
	}

	/**
	 * Method removes all cached connections which were related to the given node (e.g. when the node was disconnected)
	 * as well as decreases count of all connections stored in cache.
	 *
	 * @param node {@link JID} of the node for which cached items will be removed.
	 */
	@Override
	public void removeAllForNode(JID node) {
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "Removing connections from cache for node: {0}", node);
		}
		for (Map.Entry<BareJID, Map<JID, ConnectionRecordExt>> userEntry : userConnections.entrySet()) {
			Map<JID, ConnectionRecordExt> conns = userEntry.getValue();
			synchronized (conns) {
				for (Iterator<Map.Entry<JID, ConnectionRecordExt>> it = conns.entrySet().iterator(); it.hasNext(); ) {
					Map.Entry<JID, ConnectionRecordExt> entry = it.next();

					if (node.equals(entry.getValue().getNode())) {
						it.remove();
						--connectionsSize;
						if (log.isLoggable(Level.FINEST)) {
							log.log(Level.FINEST, "Removed connection: {0}", entry.getValue());
						}
					} else {
						if (log.isLoggable(Level.FINEST)) {
							log.log(Level.FINEST, "Different node: {0}", entry.getValue());
						}
					}
				}
				if (conns.isEmpty()) {
					if (userConnections.remove(userEntry.getKey(), conns)) {
						--mapSize;
					}
				}
			}
		}
	}

	/**
	 * Method updates given connection in the cache with the information about presence for the given connection.
	 *
	 * @param presence {@link Element} containing presence information
	 * @param rec {@link ConnectionRecordExt} for which presence should be updated
	 */
	public void updatePresence(Element presence, ConnectionRecordExt rec) {
		Map<JID, ConnectionRecordExt> conns = userConnections.get(rec.getUserJid().getBareJID());

		if (conns != null) {
			synchronized (conns) {
				ConnectionRecordExt cache_rec = conns.get(rec.getConnectionId());

				if (cache_rec != null) {
					cache_rec.setLastPresence(presence);
					if (log.isLoggable(Level.FINEST)) {
						log.log(Level.FINEST, "Last presence set for: {0}", cache_rec);
					}
				} else {

					// This may happen for presence unavailable when user disconnected,
					// ignore is OK
				}
			}
		}
	}

	/**
	 * Method responsible for user disconnection events. Removes given connection record from the cache and decrease
	 * counter of all connections.
	 *
	 * @param rec {@link ConnectionRecordExt} for which presence should be updated
	 */
	public void userDisconnected(ConnectionRecordExt rec) {
		Map<JID, ConnectionRecordExt> conns = userConnections.get(rec.getUserJid().getBareJID());

		if (conns != null) {
			synchronized (conns) {
				if (conns.remove(rec.getConnectionId()) != null) {
					--connectionsSize;

					if (conns.isEmpty()) {
						if (userConnections.remove(rec.getUserJid().getBareJID(), conns)) {
							--mapSize;
						}
					}
				}

				// TODO: if the conns.size() == 0 now we could in theory remove it from
				// the Map, however, if there is another call waiting on the
				// synchronized we may have again problem with concurrent access. I
				// don't want to synchronize on the whole main Map as that would affect
				// overall performance.
				// Now we may have memory usage problems but not before the installation
				// reaches over 10mln online users. Hopefully...
				// Maybe a better approach would be kind of a periodic cleanup process
				// which would lock the whole Map and perform cleanup of empty Sets.
				// Once a day would be enough.
			}
		}
	}

	/**
	 * Method responsible for user connection events. Adds given connection record to the cache and increases counters
	 * of both all connections and number of user JIDs in cache..
	 *
	 * @param recs {@link ConnectionRecordExt} for which presence should be updated
	 */
	public void usersConnected(ConnectionRecordExt... recs) {
		for (ConnectionRecordExt rec : recs) {
			Map<JID, ConnectionRecordExt> conns = null;

			// This is necessary to properly handle the case if the same user connects
			// at the same time to 2 or more different nodes, then this method can be
			// executed at the same time for both connections. Rare case but possible.
//			synchronized ( userConnections ) {
			conns = userConnections.get(rec.getUserJid().getBareJID());
			if (conns == null) {

				// The initial size seems low, but most of the users make a single
				// connection anyway, with rare cases when users make 2 or more
				// connections.
				conns = new LinkedHashMap<JID, ConnectionRecordExt>(2);
				Map<JID, ConnectionRecordExt> old = userConnections.putIfAbsent(rec.getUserJid().getBareJID(), conns);
				if (old != null) {
					conns = old;
				} else {
					++mapSize;
				}
			}
// This was for testing only!!
//			try {
//				Thread.sleep(1000);
//			} catch (InterruptedException ex) {
//				Logger.getLogger(CacheContenerV2.class.getName()).log(Level.SEVERE, null, ex);
//			}
//			}
			synchronized (conns) {
				if (conns.put(rec.getConnectionId(), rec) == null) {
					++connectionsSize;
				}
				Map<JID, ConnectionRecordExt> old = userConnections.putIfAbsent(rec.getUserJid().getBareJID(), conns);
				if (old == null) {
					++mapSize;
//				} else if (old != conns) {
//					synchronized (old) {
//						old.put( rec.getConnectionId(), rec ) == null );
//					}
				}
			}
		}
	}

	/**
	 * Method responsible for replacing used conn id by user connection/resource. In fact it removes connection using
	 * old connection id from connection cache, updates internal value of connection id to new connection id and adds it
	 * to cache using new connection id as a key.
	 * <p>
	 * This happens during stream resumption as connection id changes for session for which stream is resumed.
	 *
	 * @param userId - bare jid of user
	 * @param oldConnectionId - connection id which was changed
	 * @param newConnectionId - connection id which replaced old connection id
	 */
	public void userChangedConnId(BareJID userId, JID oldConnectionId, JID newConnectionId) {
		Map<JID, ConnectionRecordExt> conns = userConnections.get(userId);

		if (conns != null) {
			synchronized (conns) {
				ConnectionRecordExt rec = conns.remove(oldConnectionId);
				if (rec != null) {
					rec.setConnectionId(newConnectionId);
					conns.put(newConnectionId, rec);
				}
			}
		}
	}

	//~--- get methods ----------------------------------------------------------
	@Override
	public JID[] getConnectionIdsForJid(BareJID jid) {
		JID[] result = null;
		Map<JID, ConnectionRecordExt> conns = userConnections.get(jid);

		if (conns != null) {
			synchronized (conns) {
				int size = conns.size();

				if (size > 0) {
					result = conns.keySet().toArray(new JID[size]);
				}
			}
		}

		return result;
	}

	/**
	 * Retrieves a {@link ConnectionRecordExt} for the particular, specific {@link JID}
	 *
	 * @param jid for which {@link ConnectionRecordExt} should be retrieved
	 *
	 * @return {@link ConnectionRecordExt} for the given {@link JID}
	 */
	public ConnectionRecordExt getConnectionRecord(JID jid) {
		ConnectionRecordExt result = null;
		Map<JID, ConnectionRecordExt> conns = userConnections.get(jid.getBareJID());

		if (conns != null) {
			synchronized (conns) {

				// TODO: look at possible performance improvements in using entrySet
				// instead of values, this depends on how the Map implements it
				// internally
				for (ConnectionRecordExt rec : conns.values()) {
					if (jid.equals(rec.getUserJid())) {
						result = rec;

						break;
					}
				}
			}
		}

		return result;
	}

	/**
	 * Method retrieves all {@link ConnectionRecordExt} records for the particular {@link BareJID}
	 *
	 * @param bareJID for which all {@link ConnectionRecordExt} should be retrieved
	 *
	 * @return {@link Set} containing all {@link ConnectionRecordExt} objects for the given {@link JID}
	 */
	public Set<ConnectionRecordExt> getConnectionRecords(BareJID bareJID) {
		Set<ConnectionRecordExt> result = Collections.emptySet();
		Map<JID, ConnectionRecordExt> conns = userConnections.get(bareJID);

		if (conns != null) {
			synchronized (conns) {
				result = new LinkedHashSet<>(conns.values());
			}
		}

		return result;
	}

	/**
	 * Method allows retrieval internal structure underlying cache.
	 *
	 * @return {@link Object} with internal structure of underlying cache.
	 */
	public Object getInternalData() {
		return userConnections;
	}

	/**
	 * Method retrieves all nodes on which particular user has it's connections.
	 *
	 * @param jid for which list of cluster nodes should be returned.
	 *
	 * @return list of cluster nodes on which user has it's connections.
	 */
	public List<JID> getNodesForJid(JID jid) {
		List<JID> result = null;
		Map<JID, ConnectionRecordExt> conns = userConnections.get(jid.getBareJID());

		if (conns != null) {
			synchronized (conns) {
				int size = conns.size();

				if (size > 0) {
					result = new ArrayList<JID>(size);

					// TODO: look at possible performance improvements in using entrySet
					// instead of values, this depends on how the Map implements it
					// internally
					for (ConnectionRecordExt rec : conns.values()) {
						// if we need to get nodes for full jid then we need to return only nodes
						// with this particular resource connected
						if (jid.getResource() != null && !rec.getUserJid().equals(jid)) {
							continue;
						}

						// This should be efficient if users normally make a small number of
						// connections to different cluster nodes. This is what usually
						// happens a single user rarely opens more than 2 connections for
						// one account. If both connections are to the same cluster node we
						// avoid here putting the same node twice in the result collection.
						if (!result.contains(rec.getNode())) {
							result.add(rec.getNode());
						}
					}
				}
			}
		}

		return result;
	}

	@Override
	public boolean hasCompleteJidsInfo() {
		return synced;
	}
}

//~ Formatted in Tigase Code Convention on 13/07/06
