/*
 * Tigase ACS - Tigase Advanced Clustering Strategy
 * Copyright (C) 2004 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
/*
 * Tigase Jabber/XMPP Server
 */
package tigase.licence.callbacks;

import tigase.cluster.strategy.ClusteringStrategyIfc;
import tigase.licence.Licence;
import tigase.licence.LicenceCheckerUpdateCallbackImpl;
import tigase.licence.LicenceCheckerUpdater;
import tigase.server.XMPPServer;
import tigase.server.cluster.strategy.ConnectionRecordExt;
import tigase.stats.MaxDailyCounterQueue;
import tigase.stats.StatisticsCollector;
import tigase.vhosts.VHostItem;
import tigase.vhosts.VHostManager;
import tigase.xml.Element;
import tigase.xmpp.jid.JID;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Session Manager ACS version of {@link tigase.licence.LicenceCheckerUpdateCallback} which provides additional
 * information about number of connected cluster nodes.
 *
 * @author Wojciech Kapcia
 */
public class LicenceCheckerUpdateCallbackImplACS
		extends LicenceCheckerUpdateCallbackImpl {

	private static final String userConnectionsDataId = "sess-man/strategy/OnlineUsersCachingStrategy/Max daily users connections count last month (whole cluster)[C]";
	private static final String userConnectionsSmDataId = "sess-man/Open user connections[I]";
	private static final String userConnectionsStrategyDataId = "sess-man/strategy/OnlineUsersCachingStrategy/Cached connections[I]";
	private static final String clusterNodesDataId = "cl-comp/Max daily cluster nodes count in last month[C]";
	private static final String clusterNodesAltDataId = "cl-comp/Known cluster nodes[I]";

	private static final Logger log = Logger.getLogger(LicenceCheckerUpdateCallbackImplACS.class.getName());
	/**
	 * Reference to {@link tigase.cluster.strategy.ClusteringStrategyIfc} implementation
	 */
	private ClusteringStrategyIfc<ConnectionRecordExt> strategy;

	/**
	 * Constructs Session Manager ACS version of {@link tigase.licence.LicenceCheckerUpdateCallback} which provides
	 * additional information about number of connected cluster nodes.
	 */
	public LicenceCheckerUpdateCallbackImplACS(String cmpName,
											   final ClusteringStrategyIfc<ConnectionRecordExt> strategy) {
		super(cmpName);
		this.strategy = strategy;
	}

	@Override
	public Element getComponentAdditionalData() {
		Element cmpInfo = super.getComponentAdditionalData();
		try {
			final String activeUsers = getMaxCount(userConnectionsDataId).orElseGet(
					() -> getStatisticsValue(userConnectionsStrategyDataId).orElse(-1) +
							getStatisticsValue(userConnectionsSmDataId).orElse(-1)).toString();
			cmpInfo.addChild(new Element("activeUsers", activeUsers));

			final String clusterNodes = getMaxCount(clusterNodesDataId).orElseGet(
					() -> getStatisticsValue(clusterNodesAltDataId).orElse(-1)).toString();
			cmpInfo.addChild(new Element("clusterNodesCount", clusterNodes));

		} catch (Exception ex) {
			log.log(Level.SEVERE, "Problem creating StatisticsData", ex);
			cmpInfo.addChild(new Element("exception", ex.getMessage()));
		}
		return cmpInfo;
	}

	@Override
	public boolean additionalValidation(Licence lic) {
		try {
			final boolean dailyLimitSurpassedUsers = isDailyLimitSurpassed(lic, Licence.MAX_ONLINE_USERS_KEY,
																		   userConnectionsDataId);
			log.log(Level.ALL, "dailyLimitSurpassedUsers: {0}", dailyLimitSurpassedUsers);
			if (dailyLimitSurpassedUsers) {
				return false;
			}

			final boolean dailyLimitSurpassedCluster = isDailyLimitSurpassed(lic, Licence.MAX_CLUSTER_NODES_KEY,
																			 clusterNodesDataId);
			log.log(Level.ALL, "dailyLimitSurpassedCluster: {0}", dailyLimitSurpassedCluster);
			if (dailyLimitSurpassedCluster) {
				return false;
			}

			final String mainVHostName = lic.getPropertyAsString(Licence.VHOST_NAME_KEY);
			if (mainVHostName != null) {
				final JID requiredVHost = JID.jidInstanceNS(mainVHostName);
				VHostManager vhostman = XMPPServer.getComponent(VHostManager.class);
				List<JID> managedVHosts = vhostman.getAllVHosts();

				if (!managedVHosts.contains(requiredVHost)) {
					log.log(Level.WARNING, "Required VHost is not managed by Tigase.");
					return false;
				}

				VHostItem item = vhostman.getVHostItem(requiredVHost.getDomain());
				String error = vhostman.getComponentRepository().validateItem(item);

				if (error != null) {
					log.log(Level.WARNING, "Cannot validate vhost " + requiredVHost + ": " + error);
					return false;
				}
			}

			return true;
		} catch (Exception e) {
			log.log(Level.WARNING, "Problem on validating licence", e);
			return false;
		}

	}

	@Override
	public String getMissingLicenseWarning() {
		//J-
		return "\n" + "This installation contains Tigase ACS package, not an open source software.\n" +
				"The Tigase ACS is only available under a commercial license.\n" +
				"The full text of the commercial license agreement is available upon request.\n" + "\n" +
				"More information about ACS component and licensing can be found here:\n" +
				"http://www.tigase.com/content/tigase-acs-advanced-clustering-strategy" + "\n" +
				"The Tigase ACS component is provided free of charge for testing and\n" +
				"development purposes only. Any use of the component on production systems,\n" +
				"either commercial or not-for-profit, requires the purchase of a license.\n" + "\n" +
				"If the Tigase ACS component is activated without a valid license, it will\n" +
				"continue to work, including its full set of features, but it will send\n" +
				"certain statistical information to Tigase's servers on a regular basis.\n" +
				"If the Tigase ACS component cannot access our servers to send information,\n" +
				"it will stop working. Once a valid license is installed, the Tigase ACS\n" +
				"component will stop sending statistical information to Tigase's servers.\n" + "\n" +
				"By activating the Tigase ACS component without a valid license you agree\n" +
				"and accept that the component will send certain statistical information\n" +
				"(such as DNS domain names, vhost names, number of online users, number of\n" +
				"cluster nodes, etc.) which may be considered confidential and proprietary\n" +
				"by the user. You accept and confirm that such information, which may be\n" +
				"considered confidential or proprietary, will be transferred to Tigase's\n" +
				"servers and that you will not pursue any remedy at law as a result of the\n" +
				"information transfer.\n" +
				"If the Tigase ACS component is installed but not activated, no statistical\n" +
				"information will be sent to Tigase's servers.";
		//J+
	}

	@Override
	public Optional<Map<String, String>> getServerVerifiableMetrics() {
		Map<String, String> metrics = new ConcurrentHashMap<>(2);
		metrics.put("online-users-count", getMaxCount(userConnectionsDataId).orElseGet(
				() -> getStatisticsValue(userConnectionsStrategyDataId).orElse(-1) +
						getStatisticsValue(userConnectionsSmDataId).orElse(-1)).toString());
		metrics.put("cluster-nodes-count", getMaxCount(clusterNodesDataId).orElseGet(
				() -> getStatisticsValue(clusterNodesAltDataId).orElse(-1)).toString());

		return Optional.of(metrics);
	}

	/**
	 * Allows obtaining maximum value of passed (collection) parameter
	 *
	 * @return a <code>int</code> value representing max value of parameter
	 */
	@SuppressWarnings("unchecked")
	private Optional<Integer> getMaxCount(String statisticsName) {
		try {
			final Optional<StatisticsCollector> stats = LicenceCheckerUpdater.getStats();
			if (stats.isPresent()) {
				Collection tmp = (Collection) stats.get().getAllStats().getCollectionValue(statisticsName);
				if (tmp != null) {
					MaxDailyCounterQueue<Integer> maxLastPeriod = (MaxDailyCounterQueue<Integer>) tmp;
					return maxLastPeriod.getMaxValue();
				}
			}
		} catch (Exception e) {
			final String msg = String.format("Cannot retrieve statistics for: %1$s. Field skipped. (msg: %2$s)",
											 statisticsName, e.getMessage());
			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST, msg, e);
			}
		}
		return Optional.empty();
	}

	/**
	 * Allows obtaining number of all connected nodes within cluster.
	 *
	 * @return a <code>int</code> value representing number of all connected nodes.
	 */
	@SuppressWarnings("unchecked")
	private Optional<Integer> getStatisticsValue(String statisticsName) {
		Integer value = null;
		try {
			final Optional<StatisticsCollector> stats = LicenceCheckerUpdater.getStats();
			if (stats.isPresent()) {
				value = (Integer) stats.get().getAllStats().getValue(statisticsName);
			}
		} catch (Exception e) {
			final String message = String.format("Cannot retrieve statistics for: %1$s. Field skipped. (msg:%2$s)",
												 statisticsName, e.getMessage());
			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST, message, e);
			}
		}
		return Optional.ofNullable(value);
	}

	@SuppressWarnings("unchecked")
	private boolean isDailyLimitSurpassed(Licence licence, String propertyName, String statisticsName) {
		Integer limit = licence.getPropertyAsInteger(propertyName);
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "Licence limits for property {0}: {1}", new Object[]{propertyName, limit});
		}
		if (limit != null) {
			getMaxCount(statisticsName);
			final Optional<StatisticsCollector> stats = LicenceCheckerUpdater.getStats();
			Collection tmp = (Collection) LicenceCheckerUpdater.getStats()
					.get()
					.getAllStats()
					.getCollectionValue(statisticsName);
			if (tmp != null) {
				MaxDailyCounterQueue<Integer> maxLastPeriod = (MaxDailyCounterQueue<Integer>) tmp;
				boolean limitSurpassed = maxLastPeriod.isLimitSurpassed(limit);
				if (log.isLoggable(Level.FINEST)) {
					log.log(Level.FINEST, "Limits check result for property {0}: {1} of {2} allowed, surpassed: {3}",
							new Object[]{propertyName, maxLastPeriod, limit, limitSurpassed});
				}

				if (limitSurpassed) {
					log.log(Level.WARNING,
							"Licence limits exceeded! Max of {0} in last period: {1}, Max allowed users (licence): {2}",
							new Object[]{propertyName, maxLastPeriod, limit});
					return true;
				}
			}
		}
		return false;
	}
}
