/*
 * Tigase ACS - PubSub Component - Tigase Advanced Clustering Strategy - PubSub Component
 * Copyright (C) 2013 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.pubsub.cluster;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import tigase.server.Packet;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.Element;
import tigase.xmpp.StanzaType;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import static org.junit.Assert.*;

/**
 * @author andrzej
 */
public class PartitionedStrategyTest {

	private static final JID localNodeJid = JID.jidInstanceNS("pubsub@node1.tigase.im");
	private PubSubCompontentClusteredTest pubsub;
	private PartitionedStrategy strategy;

	@Before
	public void init() {
		strategy = new PartitionedStrategy();
		strategy.setLocalNodeJid(localNodeJid);
		pubsub = new PubSubCompontentClusteredTest(localNodeJid, strategy);
		strategy.setPubSubComponent(pubsub);
	}

	@After
	public void destroy() {
		strategy = null;
	}

	@Test
	public void getAllNodesTest() {
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesConnectedWithLocal().toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		pubsub.nodeConnected(newNode);
		assertArrayEquals(new Object[]{localNodeJid, newNode}, strategy.getNodesConnectedWithLocal().toArray());

		pubsub.nodeDisconnected(newNode);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesConnectedWithLocal().toArray());
	}

	@Test
	public void isLocalNodeTest() {
		BareJID serviceJid = BareJID.bareJIDInstanceNS("pubsub.tigase.im");
		String node = "test3";
		assertTrue("checking if works for single node cluster - should be local " + "as we have only 1 node",
				   strategy.isLocalNode(serviceJid, node));

		JID nodeJid = JID.jidInstanceNS("pubsub@node2.tigase.im");
		pubsub.nodeConnected(nodeJid);
		assertTrue("checking if works for two node cluster - should be other node",
				   !strategy.isLocalNode(serviceJid, node));
	}

	@Test
	public void getNodeForServiceNodeTest() {
		JID serviceJid = JID.jidInstanceNS("pubsub.tigase.im");
		String node = "test3";
		assertEquals(localNodeJid, strategy.getNodeForServiceNode(serviceJid, node));

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		pubsub.nodeConnected(newNode);
		assertEquals(newNode, strategy.getNodeForServiceNode(serviceJid, node));
		assertEquals(newNode, strategy.getNodeForServiceNode(serviceJid, node));
	}

	@Test
	public void getNodeForServiceTest() {
		JID serviceJid = JID.jidInstanceNS("pubsub.tigase.im");
		assertEquals(localNodeJid, strategy.getNodeForServiceJid(serviceJid));

		JID newNode = JID.jidInstanceNS("pubsub@node1.tigase.im");
		pubsub.nodeConnected(newNode);
		assertEquals(newNode, strategy.getNodeForServiceJid(serviceJid));
		assertEquals(newNode, strategy.getNodeForServiceJid(serviceJid));
	}

	@Test
	public void getNodesForPacketPresenceTest() throws TigaseStringprepException {
		Packet presence = Packet.packetInstance("presence", "test1@tigase.im", "pubsub.tigase.im",
												StanzaType.available);
		presence.setPacketFrom(JID.jidInstance("test1@tigase.im"));
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(presence).toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		pubsub.nodeConnected(newNode);
		assertArrayEquals(new Object[]{localNodeJid, newNode}, strategy.getNodesForPacket(presence).toArray());
	}

	@Test
	public void getNodesForPacketPublishItemTest() throws TigaseStringprepException {
		Element iq = new Element("iq", new String[]{"from", "to"}, new String[]{"test1@tigase.im", "pubsub.tigase.im"});
		Element pubsub = new Element("pubsub");
		pubsub.setAttribute("xmlns", "http://jabber.org/protocol/pubsub");
		iq.addChild(pubsub);
		Element publish = new Element("publish");
		publish.setAttribute("node", "test3");
		pubsub.addChild(publish);

		Packet packet = Packet.packetInstance(iq);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		this.pubsub.nodeConnected(newNode);
		assertArrayEquals(new Object[]{newNode}, strategy.getNodesForPacket(packet).toArray());
	}

	@Test
	public void getNodesForPacketConfigureNodeTest() throws TigaseStringprepException {
		Element iq = new Element("iq", new String[]{"from", "to"}, new String[]{"test1@tigase.im", "pubsub.tigase.im"});
		Element pubsub = new Element("pubsub");
		pubsub.setAttribute("xmlns", "http://jabber.org/protocol/pubsub");
		iq.addChild(pubsub);
		Element publish = new Element("configure");
		publish.setAttribute("node", "test2");
		pubsub.addChild(publish);

		Packet packet = Packet.packetInstance(iq);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node1.tigase.im");
		this.pubsub.nodeConnected(newNode);
		assertArrayEquals(new Object[]{newNode}, strategy.getNodesForPacket(packet).toArray());

		publish.setAttribute("node", "test2");
		assertArrayEquals(new Object[]{newNode}, strategy.getNodesForPacket(packet).toArray());
	}

	@Test
	public void getNodesForPacketDiscoInfoCapsResult() throws TigaseStringprepException {
		Element iq = new Element("iq", new String[]{"from", "to", "type"},
								 new String[]{"test1@tigase.im", "pubsub.tigase.im", "result"});
		Element query = new Element("query");
		query.setAttribute("xmlns", "http://jabber.org/protocol/disco#info");
		query.setAttribute("node", "http://tigase.org/test#0.1");
		iq.addChild(query);

		Packet packet = Packet.packetInstance(iq);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		this.pubsub.nodeConnected(newNode);
		assertArrayEquals(new Object[]{localNodeJid, newNode}, strategy.getNodesForPacket(packet).toArray());
	}

}
