/*
 * Tigase ACS - PubSub Component - Tigase Advanced Clustering Strategy - PubSub Component
 * Copyright (C) 2013 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.pubsub.cluster;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import tigase.component.exceptions.RepositoryException;
import tigase.conf.ConfigurationException;
import tigase.db.DataSource;
import tigase.eventbus.EventBusFactory;
import tigase.kernel.beans.Inject;
import tigase.kernel.core.Kernel;
import tigase.pubsub.AbstractNodeConfig;
import tigase.pubsub.NodeType;
import tigase.pubsub.PubSubComponent;
import tigase.pubsub.PubSubConfig;
import tigase.pubsub.exceptions.PubSubException;
import tigase.pubsub.modules.PublishItemModule;
import tigase.pubsub.modules.mam.Query;
import tigase.pubsub.repository.IAffiliations;
import tigase.pubsub.repository.IPubSubDAO;
import tigase.pubsub.repository.IPubSubRepository;
import tigase.pubsub.repository.cached.*;
import tigase.server.Packet;
import tigase.util.common.TimerTask;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.Element;
import tigase.xmpp.Authorization;
import tigase.xmpp.StanzaType;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.util.Date;

import static org.junit.Assert.*;

/**
 * @author andrzej
 */
public class ClusteredNodeStrategyTest {

	private static final JID localNodeJid = JID.jidInstanceNS("pubsub@node1.tigase.im");
	private static final JID smJid = JID.jidInstanceNS("sm@node1.tigase.im");
	private Kernel kernel;
	private PubSubComponentClusteredIfcImpl pubsub;
	private ClusteredNodeStrategy strategy;

	@Before
	public void init() throws ConfigurationException {
		kernel = new Kernel();
		pubsub = new PubSubComponentClusteredIfcImpl();
		kernel.registerBean("service").asInstance(this.pubsub).exec();
		pubsub.register(kernel);
		pubsub.getKernel().setForceAllowNull(true);

		pubsub.getKernel().registerBean("eventBus").asInstance(EventBusFactory.getInstance()).exec();
		pubsub.getKernel().registerBean(PubSubConfig.class).exec();
		pubsub.getKernel().registerBean(PublishItemModule.class).exec();
		pubsub.getKernel().registerBean("strategy").asClass(ClusteredNodeStrategy.class).exec();
		pubsub.getKernel().registerBean("pubsubRepository").asClass(CachedPubSubRepositoryClusteredTest.class).exec();

		// pubsub.getKernel().initAll();

		strategy = pubsub.getKernel().getInstance(ClusteredNodeStrategy.class);
		strategy.setLocalNodeJid(localNodeJid);
		strategy.setPubSubComponent(pubsub);

		pubsub.nodeConnected(localNodeJid.getDomain());

	}

	@After
	public void destroy() {
		strategy = null;
	}

	@Test
	public void getAllNodesTest() {
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesConnectedWithLocal().toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		pubsub.nodeConnected(newNode.getDomain());
		assertArrayEquals(new Object[]{localNodeJid, newNode}, strategy.getNodesConnectedWithLocal().toArray());

		pubsub.nodeDisconnected(newNode.getDomain());
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesConnectedWithLocal().toArray());
	}

	@Test
	public void isLocalNodeTest() {
		BareJID serviceJid = BareJID.bareJIDInstanceNS("pubsub.tigase.im");
		String node = "test2";
		assertTrue("checking if works for single node cluster - should be local " + "as we have only 1 node",
				   strategy.isLocalNode(serviceJid, node));

		strategy.nodeConnected(JID.jidInstanceNS("pubsub@node2.tigase.im"));
		assertTrue("checking if works for two node cluster - should be always local node",
				   strategy.isLocalNode(serviceJid, node));
	}

	@Test
	public void getNodeForServiceTest() {
		JID serviceJid = JID.jidInstanceNS("pubsub.tigase.im");
		assertEquals(localNodeJid, strategy.getNodeForServiceJid(serviceJid));

		JID newNode = JID.jidInstanceNS("pubsub@node1.tigase.im");
		pubsub.nodeConnected(newNode.getDomain());
		assertEquals(newNode, strategy.getNodeForServiceJid(serviceJid));
		assertEquals(newNode, strategy.getNodeForServiceJid(serviceJid));
	}

	@Test
	public void getNodesForPacketPresenceTest() throws TigaseStringprepException, PubSubException {
		Packet presence = Packet.packetInstance("presence", "test1@tigase.im", "pubsub.tigase.im",
												StanzaType.available);
		presence.setPacketFrom(smJid);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(presence).toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		pubsub.nodeConnected(newNode.getDomain());
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(presence).toArray());
	}

	@Test
	public void getNodesForPacketPublishItemTest() throws TigaseStringprepException, PubSubException {
		Element iq = new Element("iq", new String[]{"from", "to"}, new String[]{"test1@tigase.im", "pubsub.tigase.im"});
		Element pubsub = new Element("pubsub");
		pubsub.setAttribute("xmlns", "http://jabber.org/protocol/pubsub");
		iq.addChild(pubsub);
		Element publish = new Element("publish");
		publish.setAttribute("node", "test2");
		pubsub.addChild(publish);

		Packet packet = Packet.packetInstance(iq);
		packet.setPacketFrom(smJid);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		this.pubsub.nodeConnected(newNode.getDomain());
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());
	}

	@Test
	public void getNodesForPacketConfigureNodeTest() throws TigaseStringprepException, PubSubException {
		Element iq = new Element("iq", new String[]{"from", "to"}, new String[]{"test1@tigase.im", "pubsub.tigase.im"});
		Element pubsub = new Element("pubsub");
		pubsub.setAttribute("xmlns", "http://jabber.org/protocol/pubsub");
		iq.addChild(pubsub);
		Element publish = new Element("configure");
		publish.setAttribute("node", "test2");
		pubsub.addChild(publish);

		Packet packet = Packet.packetInstance(iq);
		packet.setPacketFrom(smJid);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node1.tigase.im");
		this.pubsub.nodeConnected(newNode.getDomain());
		assertArrayEquals(new Object[]{newNode}, strategy.getNodesForPacket(packet).toArray());

		publish.setAttribute("node", "test2");
		assertArrayEquals(new Object[]{newNode}, strategy.getNodesForPacket(packet).toArray());
	}

	@Test
	public void getNodesForPacketPublishItemPepTest() throws TigaseStringprepException, PubSubException {
		Element iq = new Element("iq", new String[]{"from", "to"}, new String[]{"test1@tigase.im", "test1@tigase.im"});
		Element pubsub = new Element("pubsub");
		pubsub.setAttribute("xmlns", "http://jabber.org/protocol/pubsub");
		iq.addChild(pubsub);
		Element publish = new Element("publish");
		publish.setAttribute("node", "test2");
		pubsub.addChild(publish);

		Packet packet = Packet.packetInstance(iq);
		packet.setPacketFrom(smJid);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		this.pubsub.nodeConnected(newNode.getDomain());
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		try {
			iq = new Element("iq", new String[]{"from", "to"}, new String[]{"test1@tigase.im", "test2@tigase.im"});
			iq.addChild(pubsub);
			packet = Packet.packetInstance(iq);
			packet.setPacketFrom(smJid);
			strategy.getNodesForPacket(packet).toArray();
		} catch (Exception ex) {
			assertEquals(PubSubException.class, ex.getClass());
			assertEquals(Authorization.FORBIDDEN, ((PubSubException) ex).getErrorCondition());
		}
	}

	@Test
	public void getNodesForPacketPublishItemPepCheckSingleCreationTest()
			throws TigaseStringprepException, PubSubException, RepositoryException {
		Element iq = new Element("iq", new String[]{"from", "to"}, new String[]{"test1@tigase.im", "test1@tigase.im"});
		Element pubsub = new Element("pubsub");
		pubsub.setAttribute("xmlns", "http://jabber.org/protocol/pubsub");
		iq.addChild(pubsub);
		Element publish = new Element("publish");
		publish.setAttribute("node", "test2");
		pubsub.addChild(publish);
		Packet packet = Packet.packetInstance(iq);
		packet.setPacketFrom(smJid);

		PublishItemModule publishItemModule = this.pubsub.getKernel().getInstance(PublishItemModule.class);
		IPubSubRepository pubSubRepository = this.pubsub.getKernel().getInstance(IPubSubRepository.class);

		AbstractNodeConfig node = publishItemModule.ensurePepNode(packet.getStanzaTo().getBareJID(), "test2",
																  packet.getStanzaFrom().getBareJID(), null);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		System.out.println("expected = " + node + " got " +
								   pubSubRepository.getNodeConfig(packet.getStanzaTo().getBareJID(), "test2"));
		assertTrue(node == pubSubRepository.getNodeConfig(packet.getStanzaTo().getBareJID(), "test2"));

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		this.pubsub.nodeConnected(newNode.getDomain());
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		assertTrue(node == pubSubRepository.getNodeConfig(packet.getStanzaTo().getBareJID(), "test2"));
	}

	@Test
	public void getNodesForPacketDiscoInfoCapsResult() throws TigaseStringprepException, PubSubException {
		Element iq = new Element("iq", new String[]{"from", "to", "type"},
								 new String[]{"test1@tigase.im", "pubsub.tigase.im", "result"});
		Element query = new Element("query");
		query.setAttribute("xmlns", "http://jabber.org/protocol/disco#info");
		query.setAttribute("node", "http://tigase.org/test#0.1");
		iq.addChild(query);

		Packet packet = Packet.packetInstance(iq);
		assertArrayEquals(new Object[]{localNodeJid}, strategy.getNodesForPacket(packet).toArray());

		JID newNode = JID.jidInstanceNS("pubsub@node2.tigase.im");
		this.pubsub.nodeConnected(newNode.getDomain());
		assertArrayEquals(new Object[]{localNodeJid, newNode}, strategy.getNodesForPacket(packet).toArray());
	}

	public static class CachedPubSubRepositoryClusteredTest
			extends CachedPubSubRepositoryClustered<Integer> {

		public CachedPubSubRepositoryClusteredTest() {
		}

		@Override
		public void addToRootCollection(BareJID serviceJid, String nodeName) {
		}

		@Override
		public void createNode(BareJID serviceJid, String nodeName, BareJID ownerJid, AbstractNodeConfig nodeConfig,
							   NodeType nodeType, String collection) throws RepositoryException {
			int nodeId = (serviceJid.toString() + nodeName).hashCode();
			IAffiliationsCached nodeAffiliations = newNodeAffiliations(serviceJid, nodeName, nodeId, ()-> null);
			ISubscriptionsCached nodeSubscriptions = newNodeSubscriptions(serviceJid, nodeName, nodeId, ()-> null);
			Items nodeItems = new Items(nodeId, serviceJid, nodeName, dao, this);
			Node node = new Node(nodeId, serviceJid, nodeConfig, nodeAffiliations, nodeSubscriptions, nodeItems, ownerJid,
								 new Date());
			NodeKey key = createKey(serviceJid, nodeName);
			this.nodes.put(key, node);
		}

		@Override
		public AbstractNodeConfig getNodeConfig(BareJID serviceJid, String nodeName) {
			Node node = getNode(serviceJid, nodeName);
			return node == null ? null : node.getNodeConfig();
		}

		@Override
		public void update(BareJID serviceJid, String nodeName, IAffiliations nodeAffiliations)
				throws RepositoryException {
		}

		@Override
		protected Node getNode(BareJID serviceJid, String nodeName) {
			NodeKey key = createKey(serviceJid, nodeName);
			return this.nodes.get(key);
		}

		@Override
		public void setDao(IPubSubDAO<Integer, DataSource, Query> dao) {
			// !
		}
	}

	public static class PubSubComponentClusteredIfcImpl
			extends PubSubComponent
			implements PubSubComponentClusteredIfc {

		@Inject
		private StrategyIfc strategy;

		public PubSubComponentClusteredIfcImpl() {
		}

		@Override
		public boolean isLocalDomain(String domain) {
			return true;
		}

		@Override
		public String getName() {
			return "pubsub";
		}

		@Override
		public JID getComponentId() {
			return localNodeJid;
		}

		@Override
		public BareJID getDefHostName() {
			return localNodeJid.getBareJID();
		}

		@Override
		public void addTimerTask(TimerTask task, long delay) {
		}

		@Override
		public String getComponentVersion() {
			return "1";
		}

		@Override
		public boolean isDiscoNonAdmin() {
			return true;
		}

		@Override
		protected void registerModules(Kernel kernel) {
		}

		@Override
		protected void onNodeConnected(JID jid) {
			super.onNodeConnected(jid);
			strategy.nodeConnected(jid);
		}

		@Override
		protected void onNodeDisconnected(JID jid) {
			super.onNodeDisconnected(jid);
			strategy.nodeDisconnected(jid);
		}

	}

}
