/*
 * Tigase ACS - PubSub Component - Tigase Advanced Clustering Strategy - PubSub Component
 * Copyright (C) 2013 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.pubsub.cluster;

/**
 * Created by andrzej on 26.01.2017.
 */

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import tigase.cluster.api.ClusterControllerIfc;
import tigase.component.exceptions.ComponentException;
import tigase.component.exceptions.RepositoryException;
import tigase.db.DataSource;
import tigase.kernel.core.Kernel;
import tigase.pubsub.AbstractNodeConfig;
import tigase.pubsub.CollectionItemsOrdering;
import tigase.pubsub.NodeType;
import tigase.pubsub.PubSubConfig;
import tigase.pubsub.exceptions.PubSubException;
import tigase.pubsub.modules.mam.Query;
import tigase.pubsub.repository.*;
import tigase.pubsub.repository.cached.CachedPubSubRepository;
import tigase.pubsub.repository.cached.ISubscriptionsCached;
import tigase.pubsub.repository.stateless.UsersAffiliation;
import tigase.pubsub.repository.stateless.UsersSubscription;
import tigase.pubsub.utils.PubSubLogic;
import tigase.server.Packet;
import tigase.xml.Element;
import tigase.xmpp.StanzaType;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;
import tigase.xmpp.mam.MAMRepository;
import tigase.xmpp.rsm.RSM;

import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;

import static org.junit.Assert.*;

/**
 * Created by andrzej on 26.01.2017.
 */
public class CachedPubSubRepositoryClusteredTest {

	private Kernel kernel;

	@Before
	public void setUp() {
		kernel = new Kernel();
	}

	@After
	public void tearDown() {
		kernel = null;
	}

	@Test
	public void test_lazyLoadingOfRootCollections() throws Exception {
		DummyPubSubDAO dao = new DummyPubSubDAO();
		dao.withDelay = true;
		CachedPubSubRepository cachedPubSubRepository = createCachedPubSubRepository(dao);
		setDelayedRootCollectionLoading(cachedPubSubRepository, true);

		BareJID serviceJid = BareJID.bareJIDInstanceNS("pubsub." + UUID.randomUUID() + ".local");
		String[] nodes = new String[10];
		for (int i = 0; i < 10; i++) {
			String node = "node-" + UUID.randomUUID().toString();
			nodes[i] = node;
			dao.createNode(serviceJid, node, serviceJid, null, NodeType.leaf, null, true);
		}

		try {
			String[] result = cachedPubSubRepository.getRootCollection(serviceJid);
			assertFalse(true);
		} catch (CachedPubSubRepository.RootCollectionSet.IllegalStateException ex) {
			assertTrue(true);
		}

		for (int i = 0; i < 2; i++) {
			dao.rootCollections.get(serviceJid).remove(nodes[i]);
			cachedPubSubRepository.removeFromRootCollection(serviceJid, nodes[i]);
			String node = "node-" + UUID.randomUUID().toString();
			nodes[i] = node;
			cachedPubSubRepository.addToRootCollection(serviceJid, node);
		}
		Arrays.sort(nodes);

		Thread.sleep(1000);

		String[] result = cachedPubSubRepository.getRootCollection(serviceJid);
		Arrays.sort(result);

		assertArrayEquals(nodes, result);
	}

	@Test
	public void test_eagerLoadingOfRootCollections() throws Exception {
		DummyPubSubDAO dao = new DummyPubSubDAO();
		dao.withDelay = true;
		CachedPubSubRepository cachedPubSubRepository = createCachedPubSubRepository(dao);

		BareJID serviceJid = BareJID.bareJIDInstanceNS("pubsub." + UUID.randomUUID() + ".local");
		String[] nodes = new String[10];
		for (int i = 0; i < 10; i++) {
			String node = "node-" + UUID.randomUUID().toString();
			nodes[i] = node;
			dao.createNode(serviceJid, node, serviceJid, null, NodeType.leaf, null, true);
		}

		Arrays.sort(nodes);

		new Thread(() -> {
			try {
				String[] result = cachedPubSubRepository.getRootCollection(serviceJid);
				assertArrayEquals(nodes, result);
			} catch (Exception ex) {
				assertFalse(true);
			}
		});

		String[] result = cachedPubSubRepository.getRootCollection(serviceJid);
		Arrays.sort(result);

		assertArrayEquals(nodes, result);
	}

	@Test
	public void test_userRemoved_lazy() throws Exception {
		DummyPubSubDAO dao = new DummyPubSubDAO();
		dao.withDelay = true;
		CachedPubSubRepository cachedPubSubRepository = createCachedPubSubRepository(dao);
		setDelayedRootCollectionLoading(cachedPubSubRepository, true);

		BareJID serviceJid = BareJID.bareJIDInstanceNS("pubsub." + UUID.randomUUID() + ".local");
		String[] nodes = new String[10];
		for (int i = 0; i < 10; i++) {
			String node = "node-" + UUID.randomUUID().toString();
			nodes[i] = node;
			dao.createNode(serviceJid, node, serviceJid, null, NodeType.leaf, null, true);
		}

		try {
			cachedPubSubRepository.getRootCollection(serviceJid);
		} catch (Exception ex) {
		}
		Thread.sleep(1000);
		assertEquals(10, cachedPubSubRepository.getRootCollection(serviceJid).length);

		cachedPubSubRepository.deleteService(serviceJid);

		try {
			cachedPubSubRepository.getRootCollection(serviceJid);
		} catch (Exception ex) {
		}
		Thread.sleep(1000);
		assertEquals(0, cachedPubSubRepository.getRootCollection(serviceJid).length);
		assertNull(dao.getChildNodes(serviceJid, null));
	}

	@Test
	public void test_userRemoved_eager() throws Exception {
		DummyPubSubDAO dao = new DummyPubSubDAO();
		dao.withDelay = true;
		CachedPubSubRepository cachedPubSubRepository = createCachedPubSubRepository(dao);

		BareJID serviceJid = BareJID.bareJIDInstanceNS("pubsub." + UUID.randomUUID() + ".local");
		String[] nodes = new String[10];
		for (int i = 0; i < 10; i++) {
			String node = "node-" + UUID.randomUUID().toString();
			nodes[i] = node;
			dao.createNode(serviceJid, node, serviceJid, null, NodeType.leaf, null, true);
		}

		assertEquals(10, cachedPubSubRepository.getRootCollection(serviceJid).length);

		cachedPubSubRepository.deleteService(serviceJid);

		assertEquals(0, cachedPubSubRepository.getRootCollection(serviceJid).length);
		assertNull(dao.getChildNodes(serviceJid, null));
	}

	@Test
	public void test_notificationFromOtherNode_lazy() throws Exception {
		DummyPubSubDAO dao = new DummyPubSubDAO();
		dao.withDelay = true;
		CachedPubSubRepositoryClustered cachedPubSubRepository = createCachedPubSubRepository(dao);
		setDelayedRootCollectionLoading(cachedPubSubRepository, true);

		BareJID serviceJid = BareJID.bareJIDInstanceNS("pubsub." + UUID.randomUUID() + ".local");
		String[] nodes = new String[10];
		for (int i = 0; i < 10; i++) {
			String node = "node-" + UUID.randomUUID().toString();
			nodes[i] = node;
			dao.createNode(serviceJid, node, serviceJid, null, NodeType.leaf, null, true);
		}

		try {
			String[] result = cachedPubSubRepository.getRootCollection(serviceJid);
			assertFalse(true);
		} catch (CachedPubSubRepository.RootCollectionSet.IllegalStateException ex) {
			assertTrue(true);
		}

		for (int i = 0; i < 2; i++) {
			dao.rootCollections.get(serviceJid).remove(nodes[i]);
			cachedPubSubRepository.getRootCollectionSet(serviceJid).remove(nodes[i]);
			String node = "node-" + UUID.randomUUID().toString();
			nodes[i] = node;
			cachedPubSubRepository.getRootCollectionSet(serviceJid).add(node);
		}
		Arrays.sort(nodes);

		Thread.sleep(1000);

		String[] result = cachedPubSubRepository.getRootCollection(serviceJid);
		Arrays.sort(result);

		assertArrayEquals(nodes, result);
	}

	@Test
	public void test_notificationFromOtherNode_eager() throws Exception {
		DummyPubSubDAO dao = new DummyPubSubDAO();
		dao.withDelay = true;
		CachedPubSubRepositoryClustered cachedPubSubRepository = createCachedPubSubRepository(dao);

		BareJID serviceJid = BareJID.bareJIDInstanceNS("pubsub." + UUID.randomUUID() + ".local");
		String[] nodes = new String[10];
		for (int i = 0; i < 10; i++) {
			String node = "node-" + UUID.randomUUID().toString();
			nodes[i] = node;
			dao.createNode(serviceJid, node, serviceJid, null, NodeType.leaf, null, true);
		}

		new Thread(() -> {
			try {
				Thread.sleep(200);
				String[] result = cachedPubSubRepository.getRootCollection(serviceJid);
				assertArrayEquals(nodes, result);
			} catch (Exception ex) {
				assertFalse(true);
			}
		});

		for (int i = 0; i < 2; i++) {
			dao.rootCollections.get(serviceJid).remove(nodes[i]);
			cachedPubSubRepository.getRootCollectionSet(serviceJid).remove(nodes[i]);
			String node = "node-" + UUID.randomUUID().toString();
			nodes[i] = node;
			cachedPubSubRepository.getRootCollectionSet(serviceJid).add(node);
		}
		Arrays.sort(nodes);

		Thread.sleep(1000);

		String[] result = cachedPubSubRepository.getRootCollection(serviceJid);
		Arrays.sort(result);

		assertArrayEquals(nodes, result);
	}

	protected void setDelayedRootCollectionLoading(CachedPubSubRepository cachedPubSubRepository, boolean value)
			throws NoSuchFieldException, IllegalAccessException {
		Field f = CachedPubSubRepository.class.getDeclaredField("delayedRootCollectionLoading");
		f.setAccessible(true);
		f.set(cachedPubSubRepository, value);
	}

	protected CachedPubSubRepositoryClustered createCachedPubSubRepository(PubSubDAO dao) {
		kernel.registerBean("pubsubDao").asInstance(dao).exec();
		kernel.registerBean("logic").asInstance(new PubSubLogic() {

			@Override
			public Stream<JID> subscribersOfNotifications(BareJID serviceJid, String nodeName)
					throws RepositoryException {
				return null;
			}

			@Override
			public boolean isServiceJidPEP(BareJID serivceJid) {
				return false;
			}

			@Override
			public boolean isMAMEnabled(BareJID serviceJid, String node) throws RepositoryException {
				return false;
			}

			@Override
			public String validateItemId(BareJID toJid, String nodeName, String id) {
				return null;
			}

			@Override
			public boolean isServiceAutoCreated() {
				return true;
			}

			@Override
			public void checkNodeConfig(AbstractNodeConfig nodeConfig) throws PubSubException {

			}

			@Override
			public boolean hasSenderSubscription(BareJID bareJid, IAffiliations affiliations)
					throws RepositoryException {
				return false;
			}

			@Override
			public boolean isSenderInRosterGroup(BareJID bareJid, AbstractNodeConfig nodeConfig,
												 IAffiliations affiliations, ISubscriptions subscriptions)
					throws RepositoryException {
				return false;
			}

			@Override
			public Element prepareNotificationMessage(JID from, String id, String uuid, String nodeName, Element items,
													  String expireAt, Map<String, String> headers,
													  StanzaType stanzaType) {
				return null;
			}
			
			@Override
			public void checkPermission(BareJID serviceJid, String nodeName, JID senderJid, Action action)
					throws PubSubException, RepositoryException {

			}

		}).exec();
		kernel.registerBean("config").asInstance(new PubSubConfig()).exec();
		kernel.registerBean("strategy").asInstance(new StrategyIfc() {
			@Override
			public boolean filterOutPacket(Packet packet) {
				return false;
			}

			@Override
			public List<JID> getNodesForPacket(Packet packet) throws PubSubException {
				return null;
			}

			@Override
			public List<JID> getNodesConnected() {
				return null;
			}

			@Override
			public List<JID> getNodesConnectedWithLocal() {
				return null;
			}

			@Override
			public JID getLocalNodeJid() {
				return null;
			}

			@Override
			public boolean isLocalNode(BareJID serviceJid, String node) {
				return false;
			}

			@Override
			public void nodeAddedToCollection(BareJID serviceJid, String nodeName, String collection) {

			}

			@Override
			public void nodeAddedToRootCollection(BareJID serviceJid, String nodeName) {

			}

			@Override
			public void nodeAffiliationsChanged(BareJID serviceJid, String nodeName,
												Map<BareJID, UsersAffiliation> changedAffiliations) {

			}

			@Override
			public void nodeRemovedFromCollection(BareJID serviceJid, String nodeName, String collection) {

			}

			@Override
			public void nodeRemovedFromRootCollection(BareJID serviceJid, String nodeName) {

			}

			@Override
			public void nodeConfigurationChanged(BareJID serviceJid, String nodeName, AbstractNodeConfig nodeConfig) {
				
			}

			@Override
			public void nodeConnected(JID nodeJid) {

			}

			@Override
			public void nodeDisconnected(JID nodeJid) {

			}

			@Override
			public void nodeSubscriptionsChanged(BareJID serviceJid, String nodeName,
												 Map<BareJID, UsersSubscription> changedSubscriptions) {

			}

			@Override
			public void presenceCapsChanged(BareJID serviceJid, JID jid, String caps) {
				
			}

			@Override
			public void presenceCapsRemoved(BareJID serviceJid, JID jid) {

			}

			@Override
			public void packetProcessed() {

			}

			@Override
			public void setClusterController(ClusterControllerIfc cl_controller) {

			}

			@Override
			public void setConfig(PubSubConfig config) {

			}

			@Override
			public void setPubSubComponent(PubSubComponentClusteredIfc pubSubComponent) {

			}

			@Override
			public boolean sendException(Packet packet, ComponentException ex) {
				return false;
			}

			@Override
			public void userRemoved(BareJID serviceJid) {

			}

			@Override
			public AbstractNodeConfig wrapNodeConfig(BareJID serviceJid, String nodeName, AbstractNodeConfig config) {
				return null;
			}

			@Override
			public ISubscriptionsCached newNodeSubscriptions(BareJID serviceJid, String nodeName, Object nodeId,
															 Map<BareJID, UsersSubscription> nodeSubscriptions) {
				return null;
			}
			
			@Override
			public boolean isOnlineLocally(JID jid) {
				return false;
			}
		}).exec();
		kernel.registerBean("cachedRepository").asClass(CachedPubSubRepositoryClustered.class).exec();
		return kernel.getInstance(CachedPubSubRepositoryClustered.class);
	}

	public static class DummyPubSubDAO
			extends PubSubDAO {

		protected Map<BareJID, Set<String>> rootCollections = new ConcurrentHashMap<>();
		protected boolean withDelay;

		@Override
		public void createService(BareJID serviceJID, boolean isPublic) throws RepositoryException {
			
		}

		@Override
		public Object createNode(BareJID serviceJid, String nodeName, BareJID ownerJid, AbstractNodeConfig nodeConfig,
								 NodeType nodeType, Object collectionId, boolean autocreateService)
				throws RepositoryException {
			Set<String> nodes = rootCollections.computeIfAbsent(serviceJid, bareJID -> new HashSet<String>());
			nodes.add(nodeName);
			return null;
		}

		@Override
		public void deleteItem(BareJID serviceJid, Object nodeId, String id) throws RepositoryException {

		}

		@Override
		public void deleteNode(BareJID serviceJid, Object nodeId) throws RepositoryException {

		}

		@Override
		public String[] getAllNodesList(BareJID serviceJid) throws RepositoryException {
			return new String[0];
		}

		@Override
		public IItems.IItem getItem(BareJID serviceJid, Object nodeId, String id) throws RepositoryException {
			return null;
		}

		@Override
		public List<IItems.IItem> getItems(BareJID serviceJid, List nodeIds, Date after, Date before, RSM rsm,
										   CollectionItemsOrdering ordering) throws RepositoryException {
			return null;
		}
		
		@Override
		public String[] getItemsIds(BareJID serviceJid, Object nodeId, CollectionItemsOrdering order)
				throws RepositoryException {
			return new String[0];
		}
		
		@Override
		public String[] getItemsIdsSince(BareJID serviceJid, Object nodeId, CollectionItemsOrdering order, Date since)
				throws RepositoryException {
			return new String[0];
		}

		@Override
		public List<IItems.ItemMeta> getItemsMeta(BareJID serviceJid, Object nodeId, String nodeName)
				throws RepositoryException {
			return null;
		}
		
		@Override
		public Map<BareJID, UsersAffiliation> getNodeAffiliations(BareJID serviceJid, Object nodeId) throws RepositoryException {
			return null;
		}
		
		@Override
		public INodeMeta getNodeMeta(BareJID serviceJid, String nodeName) throws RepositoryException {
			return null;
		}

		@Override
		public long getNodesCount(BareJID serviceJid) throws RepositoryException {
			return 0;
		}

		@Override
		public String[] getNodesList(BareJID serviceJid, String nodeName) throws RepositoryException {
			return new String[0];
		}

		@Override
		public Map<BareJID, UsersSubscription> getNodeSubscriptions(BareJID serviceJid, Object nodeId) throws RepositoryException {
			return null;
		}

		@Override
		public String[] getChildNodes(BareJID serviceJid, String nodeName) throws RepositoryException {
			Set<String> nodes = rootCollections.get(serviceJid);
			sleep();
			return nodes == null ? null : nodes.toArray(new String[nodes.size()]);
		}

		@Override
		public Map<String, UsersAffiliation> getUserAffiliations(BareJID serviceJid, BareJID jid)
				throws RepositoryException {
			return null;
		}

		@Override
		public Map<String, UsersSubscription> getUserSubscriptions(BareJID serviceJid, BareJID jid)
				throws RepositoryException {
			return null;
		}

		@Override
		public void addMAMItem(BareJID serviceJid, Object nodeId, String uuid, Element message, String itemId)
				throws RepositoryException {

		}
		
		@Override
		public void queryItems(Query query, Object nodeId, MAMRepository.ItemHandler itemHandler)
				throws RepositoryException, ComponentException {

		}
		
		@Override
		public void deleteService(BareJID serviceJid) throws RepositoryException {
			rootCollections.remove(serviceJid);
		}

		@Override
		public List<BareJID> getServices(BareJID domain, Boolean isPublic) throws RepositoryException {
			return null;
		}

		@Override
		public void removeNodeSubscription(BareJID serviceJid, Object nodeId, BareJID jid) throws RepositoryException {

		}

		@Override
		public void updateNodeConfig(BareJID serviceJid, Object nodeId, String serializedData, Object collectionId)
				throws RepositoryException {

		}

		@Override
		public void updateNodeAffiliation(BareJID serviceJid, Object nodeId, String nodeName,
										  UsersAffiliation userAffiliation) throws RepositoryException {

		}

		@Override
		public void updateNodeSubscription(BareJID serviceJid, Object nodeId, String nodeName,
										   UsersSubscription userSubscription) throws RepositoryException {

		}

		@Override
		public void writeItem(BareJID serviceJid, Object nodeId, long timeInMilis, String id, String publisher,
							  Element item, String uuid) throws RepositoryException {

		}

		@Override
		public void setDataSource(DataSource dataSource) {

		}

		protected void sleep() {
			if (!withDelay) {
				return;
			}

			try {
				Thread.sleep(400);
			} catch (InterruptedException ex) {
				assertFalse(true);
			}
		}
	}

}

