/*
 * Tigase ACS - Meet Component - Tigase Advanced Clustering Strategy - Meet Component
 * Copyright (C) 2021 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.meet.cluster;

import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.meet.Meet;
import tigase.meet.MeetRepository;
import tigase.xmpp.jid.BareJID;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

@Bean(name = "meetRepository", parent = MeetComponentClustered.class, active = true)
public class MeetRepositoryClustered
		extends MeetRepository {

	@Inject(nullAllowed = true)
	private StrategyIfc strategy;

	@Override
	public CompletableFuture<Meet> create(BareJID key, int maxNoOfPublishers) {
		// before creating a meet, we need to ensure that meet for this bare JID will be created only on one node
		// and other nodes will be aware of the location of the meet for this JID in the cluster
		//
		// 1. create mapping and publish it across cluster establishing cluster wide lock for creation of the meet with this bare JID.
		// 2. create meet locally
		// 3. release meet creation lock and inform other nodes that creation was successful or not (to remove mapping if needed)
		return strategy.acquireMeetCreationLock(key).thenCompose(lock -> {
			return super.create(key, maxNoOfPublishers).orTimeout(strategy.getCreationLockTimeout() / 4, TimeUnit.MILLISECONDS);
		}).whenComplete((result, ex) -> {
			strategy.releaseMeetCreationLock(key, ex == null);
		});
	}

	@Override
	public void destroyed(BareJID jid) {
		super.destroyed(jid);
		strategy.removeMeetToNodeMapping(jid, strategy.getLocalNodeJid());
	}

	@Override
	public int size() {
		if (strategy == null) {
			return 0;
		}
		return strategy.getMeetsCount();
	}
}