/*
 * Tigase ACS - Meet Component - Tigase Advanced Clustering Strategy - Meet Component
 * Copyright (C) 2021 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.meet.cluster;

import tigase.cluster.api.ClusterCommandException;
import tigase.cluster.api.ClusterControllerIfc;
import tigase.cluster.api.CommandListener;
import tigase.cluster.api.CommandListenerAbstract;
import tigase.component.exceptions.ComponentException;
import tigase.component.modules.StanzaProcessor;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.kernel.beans.selector.ClusterModeRequired;
import tigase.kernel.beans.selector.ConfigType;
import tigase.kernel.beans.selector.ConfigTypeEnum;
import tigase.kernel.beans.selector.ServerBeanSelector;
import tigase.kernel.core.Kernel;
import tigase.meet.MeetComponent;
import tigase.pubsub.cluster.ClusterNodesAware;
import tigase.pubsub.cluster.PubSubComponentClustered;
import tigase.server.ComponentInfo;
import tigase.server.Packet;
import tigase.server.Permissions;
import tigase.server.Priority;
import tigase.sys.TigaseRuntime;
import tigase.util.stringprep.TigaseStringprepException;
import tigase.xml.Element;
import tigase.xmpp.jid.JID;

import java.util.List;
import java.util.Queue;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

@Bean(name = "meet", parent = Kernel.class, active = false)
@ConfigType(ConfigTypeEnum.DefaultMode)
@ClusterModeRequired(active = true)
public class MeetComponentClustered
		extends MeetComponent {

	private static final Logger log = Logger.getLogger(MeetComponentClustered.class.getCanonicalName());
	private static final String PACKET_FORWARD_CMD = "packet-forward-meet-cmd";
	private static final String PERMISSION_KEY = "perm";

	private ClusterControllerIfc clusterController;
	@Inject(nullAllowed = true)
	private List<ClusterNodesAware> clusterNodesAware = Collections.emptyList();
	private ComponentInfo cmpInfo = null;
	@Inject
	private List<CommandListener> commandListeners;
	
	@Inject
	private StrategyIfc strategy;

	@Override
	public void processPacket(Packet packet) {
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "Received packet: {0}", packet);
		}
		try {
			JID toNode = strategy.getNodeForPacket(packet);
			if (strategy.getLocalNodeJid().equals(toNode)) {
				log.log(Level.FINEST, () -> "packet, will be processed locally");
				super.processPacket(packet);
			} else {
				log.log(Level.FINEST, () -> "packet, will be processed by node: " + toNode);
				Map<String, String> data = new HashMap<String, String>();
				if (packet.getPermissions() != null) {
					data.put(PERMISSION_KEY, packet.getPermissions().name());
				}
				clusterController.sendToNodes(PACKET_FORWARD_CMD, data, packet.getElement(), packet.getPacketFrom(),
											  null, new JID[]{toNode});
			}
		} catch (ComponentException ex){
			kernel.getInstance(StanzaProcessor.class).sendException(packet, ex);
		}
	}

	@Override
	public void register(Kernel kernel) {
		if (!ServerBeanSelector.getClusterMode(kernel)) {
			TigaseRuntime.getTigaseRuntime()
					.shutdownTigase(new String[]{
							"You've tried using Clustered version of the component but cluster-mode is disabled",
							"Shutting down system!"});
		}
		kernel.registerBean(PubSubComponentClustered.PacketForwardCommand.class).setActive(true).exec();
		super.register(kernel);
	}

	@Override
	public void setClusterController(ClusterControllerIfc cl_controller) {
		super.setClusterController(cl_controller);
		if (clusterController != null && commandListeners != null) {
			for (CommandListener cmd : commandListeners) {
				log.log(Level.CONFIG, "removing command listener " + cmd.getName());
				clusterController.removeCommandListener(cmd);
			}
		}
		clusterController = cl_controller;
		if (clusterController != null && commandListeners != null) {
			for (CommandListener cmd : commandListeners) {
				log.log(Level.CONFIG, "setting command listener " + cmd.getName());
				clusterController.setCommandListener(cmd);
			}
		}

		kernel.registerBean("clusterController").asInstance(cl_controller).exec();
	}

	@Override
	public ComponentInfo getComponentInfo() {
		cmpInfo = super.getComponentInfo();
		cmpInfo.getComponentData().put(getComponentInfoClusteringStrategyKey(), (strategy != null) ? strategy.getClass() : null);

		return cmpInfo;
	}

	protected String getComponentInfoClusteringStrategyKey() {
		return "PubSubClusteringStrategy";
	}

	@Override
	public String getDiscoDescription() {
		return super.getDiscoDescription() + " acs-clustered";
	}

	public void setCommandListeners(List<CommandListener> commandListeners) {
		if (clusterController != null && this.commandListeners != null) {
			for (CommandListener cmd : this.commandListeners) {
				log.log(Level.CONFIG, "removing command listener " + cmd.getName());
				clusterController.removeCommandListener(cmd);
			}
		}
		this.commandListeners = commandListeners;
		if (clusterController != null && commandListeners != null) {
			for (CommandListener cmd : commandListeners) {
				log.log(Level.CONFIG, "setting command listener " + cmd.getName());
				clusterController.setCommandListener(cmd);
			}
		}
	}

	public void setClusterNodesAware(List<ClusterNodesAware> clusterNodesAware) {
		if (clusterNodesAware == null) {
			clusterNodesAware = Collections.emptyList();
		}
		this.clusterNodesAware = clusterNodesAware;
	}

	@Bean(name = "packetForwardCmd", parent = MeetComponentClustered.class, active = true)
	public static class PacketForwardCommand
			extends CommandListenerAbstract {

		@Inject(bean = "service")
		private MeetComponentClustered component;

		public PacketForwardCommand() {
			super(PACKET_FORWARD_CMD, Priority.HIGH);
		}

		@Override
		public void executeCommand(JID fromNode, Set<JID> visitedNodes, Map<String, String> data,
								   Queue<Element> packets) throws ClusterCommandException {
			for (Element packetEl : packets) {
				try {
					if (log.isLoggable(Level.FINEST)) {
						log.log(Level.FINEST, "processing forwarded packet = {0}", packetEl.toString());
					}
					Packet packet = Packet.packetInstance(packetEl);
					packet.setPacketFrom(fromNode);
					if (data != null && data.get(PERMISSION_KEY) != null) {
						Permissions permission = Permissions.valueOf(data.get(PERMISSION_KEY));
						packet.setPermissions(permission);
					}

					if (component.addPacketNB(packet)) {
						if (log.isLoggable(Level.FINEST)) {
							log.log(Level.FINEST, "forwarded packet added to processing queue of component = {0}",
									packetEl.toString());
						}
					} else {
						log.log(Level.FINE, "forwarded packet dropped due to component queue overflow = {0}",
								packetEl.toString());
					}
				} catch (TigaseStringprepException ex) {
					log.log(Level.FINEST, "Addressing problem, stringprep failed for packet: {0}", packetEl);
				} catch (Throwable ex) {
					log.log(Level.SEVERE, "Exception processing forwarded packet = " + packetEl.toString(), ex);
				}
			}
		}
	}
}
