/*
 * Tigase ACS - Meet Component - Tigase Advanced Clustering Strategy - Meet Component
 * Copyright (C) 2021 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.meet.cluster;

import tigase.cluster.api.ClusterControllerIfc;
import tigase.component.exceptions.ComponentException;
import tigase.kernel.beans.Bean;
import tigase.kernel.beans.Inject;
import tigase.meet.cluster.commands.MeetCreationLockCommand;
import tigase.meet.cluster.commands.MeetSyncRequestCommand;
import tigase.server.AbstractMessageReceiver;
import tigase.server.Packet;
import tigase.util.common.TimerTask;
import tigase.xmpp.Authorization;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

@Bean(name = "meetStrategy", parent = MeetComponentClustered.class, active = true)
public class DefaultStrategy implements StrategyIfc {

	private static final Logger log = Logger.getLogger(DefaultStrategy.class.getCanonicalName());

	private final ConcurrentHashMap<BareJID, JID> meetToNode = new ConcurrentHashMap<>();

	private final ConcurrentHashMap<LockRequestKey,CompletableFuture<Void>> lockRequests = new ConcurrentHashMap();

	private final ConcurrentHashMap<BareJID, TimerTask> lockTimers = new ConcurrentHashMap<>();

	@Inject(bean = "service")
	private AbstractMessageReceiver component;
	@Inject(bean = "clusterController", nullAllowed = true)
	private ClusterControllerIfc cl_controller;

	private JID localNodeJid;

	@Override
	public CompletableFuture<Void> acquireMeetCreationLock(BareJID meetJid) {
		long until = System.currentTimeMillis() + getCreationLockTimeout();

		log.log(Level.FINEST, () -> "trying to acquire cluster wide lock for " + meetJid);
		if (!createMeetCreationLock(meetJid, until, getLocalNodeJid())) {
			log.log(Level.FINEST, () -> "failed to acquire cluster wide lock for " + meetJid + ", it is already locked locally");
			return CompletableFuture.failedFuture(new ComponentException(Authorization.CONFLICT, "Meet creation request in progress!"));
		}

		List<JID> nodes = component.getNodesConnected();
		List<CompletableFuture> futures = new ArrayList<>();
		for (JID node : nodes) {
			CompletableFuture<Void> future = new CompletableFuture();
			futures.add(future.whenComplete((x, ex) -> {
				if (ex != null) {
					log.log(Level.FINEST, ex, () -> "node " + node + " rejected lock creation for " + meetJid);
				} else {
					log.log(Level.FINEST, () -> "node " + node + " confirmed lock creation for " + meetJid);
				}
			}));
			lockRequests.put(new LockRequestKey(meetJid, node), future);
		}
		log.log(Level.FINEST, () -> "send lock acquire requests for " + meetJid + " to " + nodes + " with timeout on " + until);
		MeetCreationLockCommand.acquireLock(this, meetJid, nodes, until);
		return CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new)).orTimeout(until - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
	}

	@Override
	public void releaseMeetCreationLock(BareJID meetJid, boolean success) {
		if (!success) {
			log.log(Level.FINEST, () -> "removing mapping of " + meetJid + " to node " + getLocalNodeJid() + " - local failure");
			meetToNode.remove(meetJid, getLocalNodeJid());
		}
		TimerTask timerTask = lockTimers.remove(meetJid);
		if (timerTask != null) {
			timerTask.cancel();;
		}
		log.log(Level.FINEST, () -> "send lock release requests for " + meetJid + " to " + component.getNodesConnected());
		MeetCreationLockCommand.releaseLock(this, meetJid, component.getNodesConnected(), success);
		lockRequests.entrySet().removeIf(e -> e.getKey().meetJid.equals(meetJid));
	}

	@Override
	public ClusterControllerIfc getClusterController() {
		return cl_controller;
	}
	
	@Override
	public long getCreationLockTimeout() {
		return 60 * 1000;
	}

	@Override
	public JID getLocalNodeJid() {
		return localNodeJid;
	}

	@Override
	public JID getNodeForPacket(Packet packet) throws ComponentException {
		JID meetJid = Optional.ofNullable(packet.getStanzaTo())
				.orElseThrow(() -> new ComponentException(Authorization.NOT_ACCEPTABLE,
														  "Missing 'to' attribute in stanza!"));
		// Meet can be hosted only on one node of the cluster as we use Janus which has no support for clustering, so
		// meet is pinpointed to Janus instance.
		return getNodeForMeet(meetJid.getBareJID());
	}

	@Override
	public void nodeConnected(JID nodeJid) {
		if (!localNodeJid.equals(nodeJid)) {
			requestSync(nodeJid);
		}
	}

	public Stream<BareJID> streamLocalMeets() {
		JID localNode = getLocalNodeJid();
		return meetToNode.entrySet().stream().filter(e -> localNode.equals(e.getValue())).map(e -> e.getKey());
	}

	@Override
	public void nodeDisconnected(JID jid) {
		meetToNode.values().removeIf(node -> node.equals(jid));
	}

	protected void requestSync(JID jid) {
		MeetSyncRequestCommand.request(this, jid);
	}

	@Override
	public void setMeetToNodeMapping(BareJID meetJid, JID jid) {
		log.log(Level.FINEST, () -> "setting mapping of " + meetJid + " to node " + jid + " - timeout");
		meetToNode.put(meetJid, jid);
	}

	public void removeMeetToNodeMapping(BareJID meetJid, JID jid) {
		meetToNode.remove(meetJid, jid);
	}

	public JID getNodeForMeet(BareJID meetJid) {
		// We will process on node from the mapping or we will process locally
		return Optional.ofNullable(meetToNode.get(meetJid)).orElse(getLocalNodeJid());
	}

	public AbstractMessageReceiver getComponent() {
		return component;
	}

	public void setComponent(AbstractMessageReceiver component) {
		this.component = component;
		localNodeJid = JID.jidInstanceNS(component.getName(), component.getDefHostName().getDomain(), null);
	}
	
	@Override
	public boolean createMeetCreationLock(BareJID meetJid, long time, JID node) {
		JID currentNode = meetToNode.putIfAbsent(meetJid, node);
		if (currentNode != null) {
			return false;
		}
		TimerTask timerTask = new TimerTask() {
			@Override
			public void run() {
				log.log(Level.FINEST, () -> "removing mapping of " + meetJid + " to node " + node + " - timeout");
				meetToNode.remove(meetJid, node);
			}
		};
		lockTimers.put(meetJid, timerTask);
		component.addTimerTask(timerTask, time - System.currentTimeMillis());
		return true;
	}
	
	@Override
	public void acquiredMeetCreationLock(BareJID meetJid, JID node, boolean result) {
		CompletableFuture<Void> future = lockRequests.remove(new LockRequestKey(meetJid, node));
		if (future != null) {
			if (result) {
				future.complete(null);
			} else {
				future.completeExceptionally(new ComponentException(Authorization.CONFLICT));
			}
		}
	}

	@Override
	public void releasedMeetCreationLock(BareJID meetJid, JID node, boolean success) {
		if (!success) {
			log.log(Level.FINEST, () -> "removing mapping of " + meetJid + " to node " + node + " - failure");
			meetToNode.remove(meetJid, node);
		}
		TimerTask timerTask = lockTimers.remove(new LockRequestKey(meetJid, node));
		if (timerTask != null) {
			timerTask.cancel();
		}
	}

	public int getMeetsCount() {
		return meetToNode.size();
	}

	class LockRequestKey {
		private final BareJID meetJid;
		private final JID node;

		LockRequestKey(BareJID meetJid, JID node) {
			this.meetJid = meetJid;
			this.node = node;
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) {
				return true;
			}
			if (!(o instanceof LockRequestKey)) {
				return false;
			}
			LockRequestKey that = (LockRequestKey) o;
			return meetJid.equals(that.meetJid) && node.equals(that.node);
		}

		@Override
		public int hashCode() {
			return Objects.hash(meetJid, node);
		}
	}
}
