/*
 * Licence Library - bootstrap configuration for all Tigase projects
 * Copyright (C) 2011 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.licence;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import tigase.xml.Element;
import tigase.xmpp.jid.JID;

import java.io.ByteArrayInputStream;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.Logger;

import static org.junit.Assert.*;

/**
 * Created by andrzej on 23.09.2016.
 */
//@Ignore
public class LicenceRetrieverTest {

	private final static String repoClass = "tigase.db.xml.XMLRepository";
	private final static String repoUri = "memory://xmlRepo?autoCreateUser=true";
	private String URL = null;
	private Map<String, String> additionalParameters;
	private String component = "acs";
	private String installationId = "1A3LGLD4S0P526M1Q05CIIFMJG3TEN5RE";
	private LicenceRetriever retriever;
	private List<JID> vhosts;

	@BeforeClass
	public static void setUpLogger() {

		Logger logger = Logger.getLogger("tigase.licence");
		final ConsoleHandler consoleHandler = new ConsoleHandler();
//		final Level LEVEL = Level.ALL;
		final Level LEVEL = Level.WARNING;
//		final Level LEVEL = Level.OFF;
		consoleHandler.setLevel(LEVEL);
		logger.addHandler(consoleHandler);
		logger.setLevel(LEVEL);
		logger.setUseParentHandlers(false);
	}

	@Before
	public void setUp() {
		retriever = new LicenceRetriever();

		vhosts = new ArrayList<JID>();
		vhosts.add(JID.jidInstanceNS("vhost2"));
		vhosts.add(JID.jidInstanceNS("vhost3"));
		vhosts.add(JID.jidInstanceNS("vhost1"));

		additionalParameters = new ConcurrentHashMap<>();
	}

	@Test
	public void testLicenceRetrievalWithinLimits() throws Exception {

		// Licence for this installation-id is configured on the server to return licence file valid for 30 days
		additionalParameters.put("cluster-nodes-count", "1");
		additionalParameters.put("online-users-count", "1");

		Element result = retriever.loadData(URL, installationId, component, vhosts, additionalParameters);

		assertNotNull(result);
		String licenceString = retriever.extractLicenceText(result);
		assertNotNull(licenceString);

		final LicenceLoader licenceLoader = LicenceLoaderFactory.create();
		final Licence licence = licenceLoader.loadLicence(new ByteArrayInputStream(licenceString.getBytes()));

		System.out.println(licence);
		final Date since = licence.getPropertyAsDate(Licence.VALID_SINCE_KEY);
		final LocalDate sinceLD = Instant.ofEpochMilli(since.getTime()).atZone(ZoneId.systemDefault()).toLocalDate();
		final Date until = licence.getPropertyAsDate(Licence.VALID_UNTIL_KEY);
		final LocalDate untilLD = Instant.ofEpochMilli(until.getTime()).atZone(ZoneId.systemDefault()).toLocalDate();
		assertEquals("Licence is not valid for 30 days!", sinceLD.plusDays(30).plusDays(1), untilLD);

		final Boolean sendingStatistics = licence.getPropertyAsBoolean(Licence.SENDING_STATISTICS_KEY);
		assertFalse("Sending statistics is forced", sendingStatistics);

	}

	@Test
	public void testLicenceRetrievalUserExceeded() throws Exception {

		// Should return "demo" licence file valid for 7 days with forced sending statistics
		additionalParameters.put("cluster-nodes-count", "1");
		additionalParameters.put("online-users-count", "2");

		Element result = retriever.loadData(URL, installationId, component, vhosts, additionalParameters);

		assertNotNull(result);
		String licenceString = retriever.extractLicenceText(result);
		assertNotNull(licenceString);

		final LicenceLoader licenceLoader = LicenceLoaderFactory.create();
		final Licence licence = licenceLoader.loadLicence(new ByteArrayInputStream(licenceString.getBytes()));
		System.out.println(licence);

		final Date since = licence.getPropertyAsDate(Licence.VALID_SINCE_KEY);
		final LocalDate sinceLD = LocalDateTime.ofInstant(since.toInstant(), ZoneId.systemDefault()).toLocalDate();
		final Date until = licence.getPropertyAsDate(Licence.VALID_UNTIL_KEY);
		final LocalDate untilLD = LocalDateTime.ofInstant(until.toInstant(), ZoneId.systemDefault()).toLocalDate();
		// don't check by default valid-since date as licence-server doesn't regenerate licence on each call which can yield wrong result
			//assertEquals(LocalDate.now().minusDays(1), sinceLD);
		// we do add one day to valid-since hence the whole period is longer
		assertEquals("Licence is not valid for 7+1 days!", sinceLD.plusDays(7+1), untilLD);

		final Boolean sendingStatistics = licence.getPropertyAsBoolean(Licence.SENDING_STATISTICS_KEY);
		assertTrue("Sending statistics is NOT forced", sendingStatistics);

	}

	@Test
	public void testLicenceRetrievalNodesExceeded() throws Exception {

		additionalParameters.put("cluster-nodes-count", "2");
		additionalParameters.put("online-users-count", "1");

		Element result = retriever.loadData(URL, installationId, component, vhosts, additionalParameters);

		assertNotNull(result);
		String licenceString = retriever.extractLicenceText(result);
		assertNotNull(licenceString);

		final LicenceLoader licenceLoader = LicenceLoaderFactory.create();
		final Licence licence = licenceLoader.loadLicence(new ByteArrayInputStream(licenceString.getBytes()));
		System.out.println(licence);

		final Date since = licence.getPropertyAsDate(Licence.VALID_SINCE_KEY);
		final LocalDate sinceLD = LocalDateTime.ofInstant(since.toInstant(), ZoneId.systemDefault()).toLocalDate();
		final Date until = licence.getPropertyAsDate(Licence.VALID_UNTIL_KEY);
		final LocalDate untilLD = LocalDateTime.ofInstant(until.toInstant(), ZoneId.systemDefault()).toLocalDate();
		// don't check by default valid-since date as licence-server doesn't regenerate licence on each call which can yield wrong result
			// assertEquals(LocalDate.now().minusDays(1), sinceLD);
		// we do add one day to valid-since hence the whole period is longer
		assertEquals("Licence is not valid for 7+1 days!", sinceLD.plusDays(7+1), untilLD);

		final Boolean sendingStatistics = licence.getPropertyAsBoolean(Licence.SENDING_STATISTICS_KEY);
		assertTrue("Sending statistics is NOT forced", sendingStatistics);

	}

	@Test
	public void validateInstallationId() {
		assertTrue(LicenceRetriever.isValid(installationId));
		assertFalse(LicenceRetriever.isValid("1A3LGLD4S0P526M1Q05CIIFMJG3TEN5Re"));
		assertFalse(LicenceRetriever.isValid("1A3LGLD4S0P526M1-05CIIFMJG3TEN5RE"));
		assertFalse(LicenceRetriever.isValid("1A3LGLD4S0P526M1Q05CIIFMJG3TEN5R"));
		assertFalse(LicenceRetriever.isValid("1A3LGLD4S0P526M1Q05CIIFMJG3TEN5REE"));
		assertFalse(LicenceRetriever.isValid("Error #1: 1A3LGLD4S0P526M1Q05CIIFMJG3TEN5REE"));
	}
}
