/**
 * Licence Library - bootstrap configuration for all Tigase projects
 * Copyright (C) 2011 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.licence;

import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import tigase.db.RepositoryFactory;
import tigase.licence.utils.VHostsRetriever;
import tigase.server.XMPPServer;
import tigase.xml.DomBuilderHandler;
import tigase.xml.Element;
import tigase.xml.SimpleParser;
import tigase.xml.SingletonFactory;
import tigase.xmpp.jid.JID;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.Logger;

import static org.junit.Assert.*;
import static tigase.licence.LicenceChecker.INSTALLATION_ID_KEY;

/**
 * @author Wojtek
 */
//@Ignore
public class LicenceCheckerTest {

	final String INSTALLATION_ID = "1Q2GR2N7MBKNSALRA80KT5Q3JHR7ABM3Q";
	private final String[] VHOSTS_DEF = {"vhost1", "vhost2", "vhost3"};

	private SimpleParser parser = SingletonFactory.getParserInstance();

	public LicenceCheckerTest() {
	}

	@Before
	public void setUp() {
		final String repoClass = "tigase.db.xml.XMLRepository";
		final String repoUri = "memory://xmlRepo?autoCreateUser=true";

		Logger logger = Logger.getLogger("tigase.licence");
		final ConsoleHandler consoleHandler = new ConsoleHandler();
		final Level LEVEL = Level.ALL;
//		final Level LEVEL = Level.WARNING;
//		final Level LEVEL = Level.OFF;
		consoleHandler.setLevel(LEVEL);
		logger.addHandler(consoleHandler);
		logger.setLevel(LEVEL);
		logger.setUseParentHandlers(false);

		String config_class_name = "tigase.conf.Configurator";

		System.setProperty(RepositoryFactory.USER_REPO_CLASS_PROP_KEY, repoClass);
		System.setProperty(RepositoryFactory.GEN_USER_DB_URI_PROP_KEY, repoUri);
		System.setProperty(RepositoryFactory.USER_REPO_POOL_SIZE_PROP_KEY, "1");
		System.setProperty(INSTALLATION_ID_KEY, INSTALLATION_ID);

		XMPPServer.start(new String[0]);
	}

	@Test
	public void testGetLicenceChecker_String() throws Exception {

		for (String vh : VHOSTS_DEF) {
			VHostsRetriever.addDefaultVHost(JID.jidInstanceNS(vh));
		}

		List<String> cmpNames = new CopyOnWriteArrayList<>();
		cmpNames.add("monitor");
		cmpNames.add("ua");
		cmpNames.forEach(LicenceChecker::getLicenceChecker);

		final String module = "acs";
		cmpNames.add(module);
		LicenceChecker.getLicenceChecker(module, new LicenceCheckerUpdateCallbackImpl(module) {
			@Override
			public Optional<Map<String, String>> getServerVerifiableMetrics() {
				Map<String, String> metrics = new ConcurrentHashMap<>(2);
				metrics.put("cluster-nodes-count", "1");
				metrics.put("online-users-count", "10");

				return Optional.of(metrics);
			}
		});

		LicenceChecker.getInstallationId(true);

		Set<String> licencedComponents = LicenceChecker.getLicencedComponents();
		assertTrue("Collections are different", licencedComponents.containsAll(cmpNames));

		Element licencingDetails = LicenceChecker.getLicencingDetails(cmpNames.get(0));
		String installationId = licencingDetails.getCDataStaticStr(
				new String[]{"licence-details", INSTALLATION_ID_KEY});
		System.out.println(licencingDetails);
		String vHosts = licencingDetails.getCDataStaticStr(new String[]{"licence-details", "vhosts"});

		assertNotNull("Installation-id was not received", installationId);

		String[] split = vHosts.split(",");

		Arrays.sort(split);
		Arrays.sort(VHOSTS_DEF);

		assertArrayEquals("VHosts list differ", split, VHOSTS_DEF);

		String acs = LicenceChecker.getCodeForLicenceRetrieval(module);
		acs = new String(tigase.util.Base64.decode(acs));

		DomBuilderHandler handler = new DomBuilderHandler();
		parser.parse(handler, acs.toCharArray(), 0, acs.length());

		final Element element = handler.getParsedElements().poll();

		System.out.println("decoded: " + element.toString());

		Set<String> requiredElements = new HashSet<>(
				Arrays.asList("installation-id", "vhosts", "module", "cluster-nodes-count", "online-users-count"));
		final List<Element> items = element.getChildren(new String[]{"data", "fields"});
		for (Element el : items) {
			final String var = el.getCDataStaticStr(new String[]{"item", "var"});
			requiredElements.remove(var);
			if (var.equals("installation-id")) {
				String val = el.getCDataStaticStr(new String[]{"item", "value"});
				assertEquals("Installation-id is different", installationId, val);
			}
			if (var.equals("vhosts")) {
				List<String> val = new ArrayList<>();
				List<Element> subItems = el.getChildrenStaticStr(new String[]{"item", "value"});
				if (subItems != null && !subItems.isEmpty()) {
					for (Element e : subItems) {
						val.add(e.getCData());
					}
				}

				assertEquals("Installation-id is different", VHOSTS_DEF, val.toArray());
			}
			if (var.equals("module")) {
				String val = el.getCDataStaticStr(new String[]{"item", "value"});
				assertEquals("Module is different", module, val);
			}

			if (var.equals("cluster-nodes-count")) {
				String val = el.getCDataStaticStr(new String[]{"item", "value"});
				assertEquals("Cluster nodes count is different", "1", val);
			}
			if (var.equals("online-users-count")) {
				String val = el.getCDataStaticStr(new String[]{"item", "value"});
				assertEquals("Online users count is different", "10", val);
			}
		}
		assertEquals("Not all elements found", Collections.emptySet(), requiredElements);
	}
}