/**
 * Licence Library - bootstrap configuration for all Tigase projects
 * Copyright (C) 2011 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.licence.utils;

import tigase.stats.collector.provider.HTTPException;
import tigase.xml.DomBuilderHandler;
import tigase.xml.Element;
import tigase.xml.SimpleParser;
import tigase.xml.SingletonFactory;

import javax.net.ssl.*;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.URL;
import java.security.*;
import java.security.cert.CertificateException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

public class DataLoader {

	private static final Logger log = Logger.getLogger(DataLoader.class.getCanonicalName());

	private static final int DEFAULT_REQUEST_TIMEOUT = (int)TimeUnit.SECONDS.toMillis(15);

	private SimpleParser parser = SingletonFactory.getParserInstance();
	private SSLSocketFactory sslSocketFactory = null;

	public static <T> void addRequestItems(StringBuffer rawData, String fieldVar, List<T> items) {
		rawData.append("<item><var>").append(fieldVar).append("</var>");
		rawData.append("<value>");
		if (items != null && !items.isEmpty()) {
			for (T hash : items) {
				rawData.append("<item>").append(hash).append("</item>");
			}
		}
		rawData.append("</value>").append("</item>");
	}

	public static <K,V> void addRequestItems(StringBuffer rawData, Map<K,V> elements) {
		if (elements != null && !elements.isEmpty()) {
			for (Map.Entry<K, V> kvEntry : elements.entrySet()) {
				rawData.append("<item><var>").append(kvEntry.getKey()).append("</var>");
				rawData.append("<value>").append(kvEntry.getValue()).append("</value>").append("</item>");
			}
		}
	}

	public Element loadData(final String url, final String encodedData)
			throws TooManyRequestsHTTPException, GeneralSecurityException, IOException, HTTPException {
		String type = "application/xml";

		URL u = new URL(url);

		HttpURLConnection conn = (HttpURLConnection) u.openConnection();

		System.out.println(InetAddress.getByName(u.getHost()));

		conn.setDoOutput(true);
		conn.setRequestMethod("POST");

		conn.setConnectTimeout(DEFAULT_REQUEST_TIMEOUT);
		conn.setReadTimeout(DEFAULT_REQUEST_TIMEOUT * 5);

		conn.setRequestProperty("Accept", "*/*");

		conn.setRequestProperty("Content-Type", type);
		conn.setRequestProperty("Content-Length", String.valueOf(encodedData.length()));
		OutputStream os = conn.getOutputStream();

		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "Sending data: {0} to url: {1}", new Object[]{encodedData, u});
		}

		os.write(encodedData.getBytes());
		os.flush();

		final int responseCode = conn.getResponseCode();
		final InputStream inputStream = responseCode / 100 == 2 ? conn.getInputStream() : conn.getErrorStream();
		if (responseCode == 429) {
			String retryStr = conn.getHeaderField("Retry-After");
			if (null != retryStr && !retryStr.isEmpty()) {
				int retryAfter = Integer.valueOf(retryStr);
				throw new TooManyRequestsHTTPException(responseCode, retryAfter);
			}
		} else if (responseCode / 100 != 2) {
			throw new HTTPException(responseCode);
		}

		BufferedReader in = new BufferedReader(new InputStreamReader(inputStream));
		String inputLine;

		StringBuilder sb = new StringBuilder();

		DomBuilderHandler handler = new DomBuilderHandler();
		while ((inputLine = in.readLine()) != null) {
			sb.append(inputLine);
			parser.parse(handler, inputLine.toCharArray(), 0, inputLine.length());
		}

		in.close();

		final Element result = handler.getParsedElements().poll();

		log.log(Level.FINEST, "Retrieved data from server, response: {0}, (parsed): {1}, sb: {2} ",
				new Object[]{responseCode, result, sb});

		return result;
	}
}
