/**
 * Licence Library - bootstrap configuration for all Tigase projects
 * Copyright (C) 2011 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.licence;

import tigase.licence.utils.DataLoader;
import tigase.licence.utils.VHostsRetriever;
import tigase.stats.collector.provider.HTTPException;
import tigase.xml.Element;
import tigase.xmpp.jid.JID;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

import static tigase.licence.LicenceChecker.INSTALLATION_ID_KEY;

public class LicenceRetriever {

	private static final Logger log = Logger.getLogger("tigase.licence");

	private final String DEFAULT_LICENCE_SERVER_URL = "https://license.tigase.software/retrieve";
//	private final String DEFAULT_LICENCE_SERVER_URL = "https://license.tigase.net/retrieve";
//	private final String DEFAULT_LICENCE_SERVER_URL = "http://127.0.0.1:8080/rest/licence/retrieve";
//	private final String DEFAULT_LICENCE_SERVER_URL = "http://t2.tigase.org:8080/rest/licence/retrieve";

	private DataLoader dl = new DataLoader();

	public Element loadData(final String url, final String installationId, final String componentId,
							final List<JID> managedVHosts, Map<String, String> additionalMetrics) throws Exception {
		String encodedData = prepareRequest(installationId, componentId, managedVHosts, additionalMetrics);
		String u = (url == null ? DEFAULT_LICENCE_SERVER_URL : url);
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "Loading data, encodedData: {0}, url: {1}", new Object[]{encodedData, u});
		}
		return dl.loadData(u, encodedData);
	}

	String prepareRequest(final String installationId, final String componentId, final List<JID> managedVHosts) {
		return prepareRequest(installationId, componentId, managedVHosts, Collections.emptyMap());
	}

	String prepareRequest(final String installationId, final String componentId, final List<JID> managedVHosts,
						  Map<String, String> additionalMetrics) {
		StringBuffer sb = new StringBuffer();
		sb.append("<data><fields>");
		if (installationId != null) {
			sb.append("<item><var>" + INSTALLATION_ID_KEY + "</var><value>")
					.append(installationId)
					.append("</value></item>");
		}
		if (componentId != null) {
			sb.append("<item><var>module</var><value>").append(componentId).append("</value></item>");
		}
		DataLoader.addRequestItems(sb, Licence.VHOSTS_KEY, managedVHosts);
		DataLoader.addRequestItems(sb, additionalMetrics);
		sb.append("</fields></data>");
		return sb.toString();
	}

	String retrieve(final String componentID) {
		try {
			String installationId;
			installationId = InstallationIdRetriever.getInstallationIdRetriever().getInstallationId();

			if (null == installationId) {
				// can't retrieve licence for and installation without ID
				return null;
			}

			List<JID> managedVHosts = VHostsRetriever.getManagedVHosts();

			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST, "Loading data, installationId: {0}, managedVHosts: {1}",
						new Object[]{installationId, managedVHosts});
			}
			final Optional<Map<String, String>> serverVerifiableMetrics = LicenceChecker.getLicenceChecker(componentID)
					.getUpdateCall()
					.getServerVerifiableMetrics();
			Element data = loadData(null, installationId, componentID, managedVHosts,
									serverVerifiableMetrics.orElse(Collections.emptyMap()));
			if (data != null) {
				return extractLicenceText(data);
			} else {
				return null;
			}
		} catch (Exception e) {
			log.log(Level.WARNING, "Cannot retrieve licence: " + e.getMessage());
			if (log.isLoggable(Level.FINEST)) {
				log.log(Level.FINEST, "Cannot retrieve licence: " + e.getMessage() +
						(e instanceof HTTPException ? ", code: " + ((HTTPException) e).getStatusCode() : ""), e);
			}
			return null;
		}
	}

	protected String extractLicenceText(Element e) {
		List<Element> l = e.getChildrenStaticStr(new String[]{"command", "fields", "item", "value"});
		String result = "";
		for (Element x : l) {
			result += x.getCData() + "\n";
		}

		if (result.contains("signature=")) {
			return result;
		} else {
			return new String(Base64.decode(result));
		}
	}

}
