/**
 * Licence Library - bootstrap configuration for all Tigase projects
 * Copyright (C) 2011 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.licence;

import tigase.cluster.SessionManagerClustered;
import tigase.server.ComponentRegistrator;
import tigase.server.MessageReceiver;
import tigase.server.XMPPServer;
import tigase.server.xmppsession.SessionManager;
import tigase.stats.StatisticsCollector;
import tigase.stats.collector.StatisticsData;
import tigase.sys.TigaseRuntime;
import tigase.util.dns.DNSResolverFactory;
import tigase.vhosts.VHostManager;
import tigase.vhosts.VHostManagerIfc;
import tigase.xml.Element;
import tigase.xmpp.jid.BareJID;
import tigase.xmpp.jid.JID;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Wojciech Kapcia
 */
public class LicenceCheckerUpdater {

	/**
	 * Constant
	 * <code>CLUSTER_MODE</code> sets the cluster mode to either
	 * <code>true</code> or
	 * <code>false</code>. By default cluster mode is set to
	 * <code>false</code>.
	 */
	public static final String CLUSTER_MODE = "cluster-mode";
	private static final Logger log = Logger.getLogger("tigase.stats");
	//~--- fields ---------------------------------------------------------------
//	/** Holds configurator implementation which handles all configuration of
//	 * Tigase */
//	protected static ConfiguratorAbstract configurator;
	private final static BareJID invalidHostname = BareJID.bareJIDInstanceNS("invalid-hostname");
	/** Holds {@link tigase.server.xmppsession.SessionManager} */
	protected static SessionManager sess;
	/**
	 * Holds {@link tigase.vhosts.VHostManagerIfc} implementation which manages
	 * all virtual host related funcionality in Tigase
	 */
	protected static VHostManagerIfc vhost = null;
	/**
	 * Holds {@link tigase.stats.StatisticsCollector} implementation which
	 * manages all statistics related functionality in Tigase
	 */
	private static StatisticsCollector stats;

	//~--- constructors ---------------------------------------------------------

	public static int getClusterNodesCount() {
		int size;
		if (sess instanceof SessionManagerClustered) {
			size = ((SessionManagerClustered)sess).getStrategy().getNodesConnected().size() + 1;
		} else {
			size = 0;
		}
		return size;
	}

	/**
	 * Returns Component Information about all system components
	 *
	 * @return an  <code>{@link tigase.xml.Element}</code> representation of all
	 * system Components information
	 */
	protected static Element getComponentInfo() {

		Element result = new Element("components");

		try {
			result.addChildren(Stream.concat(XMPPServer.getComponents(MessageReceiver.class),
											 XMPPServer.getComponents(ComponentRegistrator.class))
									   .distinct()
									   .map(mr -> mr.getComponentInfo().toElement())
									   .collect(Collectors.toList()));
		} catch (Exception ex) {
			log.log(Level.SEVERE, "Problem obtaining components info", ex);
		}
		return result;
	}

	/**
	 * Returns Maxim Heap Memory usage, either OLD memory pool size as this is
	 * what is the most interesting to us or if this is not possible then total
	 * Heap size.
	 *
	 * @return maximum heap memory usage
	 */
	protected static long getHeapMax() {
		return TigaseRuntime.getTigaseRuntime().getHeapMemMax();
	}

	/**
	 * Returns Heap Memory usage, either OLD memory pool size as this is what is
	 * the most interesting to us or if this is not possible then total Heap size.
	 *
	 * @return heap memory usage
	 */
	protected static long getHeapUsed() {
		return TigaseRuntime.getTigaseRuntime().getHeapMemUsed();
	}

	/**
	 * Return default hostname of the node (i.e. machine hostname)
	 *
	 * @return hostname of the machine
	 */
	protected static BareJID getHostname() {
		BareJID hostname = null;
		try {
			String host = DNSResolverFactory.getInstance().getDefaultHost();
			if (host != null && !host.trim().isEmpty()) {
				hostname = BareJID.bareJIDInstance(host);
			} else if (sess != null && sess.getDefVHostItem() != null) {
				hostname = sess.getDefVHostItem();
			} else {
				hostname = invalidHostname;
			}
		} catch (Exception ex) {
			hostname = invalidHostname;
			log.log(Level.SEVERE, "Error during obtaining hostname", ex);
		}
		return hostname;
	}

	protected static long getMaxSessionsToday() {
		long number;
		String sNum;
		sNum = (String) stats.getAllStats().getValue("sess-man/Maximum user sessions today");
		if (sNum != null) {
			number = Long.valueOf(sNum);
		} else {
			System.out.println("error getting number");
			number = 0;
		}
		return number;
	}

	protected static long getMaxSessionsYesterday() {
		long number;
		String sNum;
		sNum = (String) stats.getAllStats().getValue("sess-man/Maximum user sessions yesterday");
		if (sNum != null) {
			number = Long.valueOf(sNum);
		} else {
			System.out.println("error getting number");
			number = 0;
		}
		return number;
	}

	public static StatisticsCollector getStats() {
		return stats;
	}

	/**
	 * Returns uptime of the instance.
	 *
	 * @return a  <code>long</code> value of the uptime in milisenconds
	 */
	protected static long getUptime() {
		return TigaseRuntime.getTigaseRuntime().getUptime();
	}

	/**
	 * Returns number of all active users connections (i.e. which session wasn't
	 * idle more than 5 minutes) on given node.
	 *
	 * @return a  <code>long</code> value representing all online users connection
	 * on given node.
	 */
	protected static long getUserActive() {
		long number;
		String sNum;
		sNum = (String) stats.getAllStats().getValue("sess-man/Active user connections");
		if (sNum != null) {
			number = Long.valueOf(sNum);
		} else {
			System.out.println("error getting number");
			number = 0;
		}
		return number;
	}

	/**
	 * Returns number of all online users connections on given node.
	 *
	 * @return a  <code>long</code> value representing all online users connection
	 * on given node.
	 */
	protected static long getUserOnline() {
		long number;
		String sNum;
		sNum = (String) stats.getAllStats().getValue("sess-man/Open user connections");
		if (sNum != null) {
			number = Long.valueOf(sNum);
		} else {
			System.out.println("error getting number");
			number = 0;
		}
		return number;
	}

	/**
	 * Returns list of all Virtual Hosts on that particular Tigase instance.
	 *
	 * @return a  <code>BareJID </code> list containing all Virtual Hosts on that
	 * particular instance
	 */
	protected static List<BareJID> getVHosts() {
		List<BareJID> list = new ArrayList<BareJID>();

		for (JID jid : vhost.getAllVHosts()) {
			list.add(jid.getBareJID());
		}

		return list;
	}

	protected static void init() {
		sess = XMPPServer.getComponent(SessionManager.class);
		stats = XMPPServer.getComponent(StatisticsCollector.class);
		vhost = XMPPServer.getComponent(VHostManager.class);
	}

	/**
	 * Constructs default implementation of handling updates of statistics data on
	 * per-component basis providing most basic statistics data about system.
	 *
	 * @return constructed {@link StatisticsData} object
	 */
	public static StatisticsData updateData() {
		log.finest("updating statistics data");

		StatisticsData data = null;
		try {
			init();
			data = new StatisticsData(getHostname());

			data.addVHosts(getVHosts());
			data.setUptime(getUptime());
			data.setHeapUsed(getHeapUsed());
			data.setHeapMax(getHeapMax());
			data.setUsersOnline(getUserOnline());
			data.setUsersActive(getUserActive());
			data.setClusterNodesCount(getClusterNodesCount());
			data.setMaxSessionsToday(getMaxSessionsToday());
			data.setMaxSessionsYesterday(getMaxSessionsYesterday());

			/*
			 * list all licenced component being used and whether they have valid licence
			 */
			data.setAdditionalData(getComponentInfo().toString());

		} catch (Exception ex) {
			log.log(Level.SEVERE, "Problem creating StatisticsData", ex);
		}
		return data;
	}

	//~--- get methods ----------------------------------------------------------
	public String getMissingLicenseWarning() {
		//J-
		return "This installation contains Tigase ACS package, not an open source software.\n" +
				"The Tigase ACS is only available under a commercial license.\n" +
				"The full text of the commercial license agreement is available upon request.\n" + "\n" +
				"The Tigase ACS component is provided free of charge for testing and\n" +
				"development purposes only. Any use of the component on production systems,\n" +
				"either commercial or not-for-profit, requires the purchase of a license.\n" + "\n" +
				"If the Tigase ACS component is activated without a valid license, it will\n" +
				"continue to work, including its full set of features, but it will send\n" +
				"certain statistical information to Tigase's servers on a regular basis.\n" +
				"If the Tigase ACS component cannot access our servers to send information,\n" +
				"it will stop working. Once a valid license is installed, the Tigase ACS\n" +
				"component will stop sending statistical information to Tigase's servers.\n" + "\n" +
				"By activating the Tigase ACS component without a valid license you agree\n" +
				"and accept that the component will send certain statistical information\n" +
				"(such as DNS domain names, vhost names, number of online users, number of\n" +
				"cluster nodes, etc.) which may be considered confidential and proprietary\n" +
				"by the user. You accept and confirm that such information, which may be\n" +
				"considered confidential or proprietary, will be transferred to Tigase's\n" +
				"servers and that you will not pursue any remedy at law as a result of the\n" +
				"information transfer.\n" +
				"If the Tigase ACS component is installed but not activated, no statistical\n" +
				"information will be sent to Tigase's servers.";
		//J+
	}
}
