/**
 * Licence Library - bootstrap configuration for all Tigase projects
 * Copyright (C) 2011 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.licence;

import tigase.licence.utils.VHostsRetriever;
import tigase.xml.Element;
import tigase.xmpp.jid.JID;

import java.io.*;
import java.nio.charset.Charset;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.*;
import java.util.logging.FileHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

/**
 * Class used to perform periodic licence check and/or upload of statistics
 * data.
 * <br>
 *
 * @author Wojciech Kapcia
 */
public class LicenceChecker {

	public static final String INSTALLATION_ID_KEY = "installation-id";

	/** Collection for storing all available {@link tigase.licence.LicenceChecker} instances */
	private static final Map<String, LicenceChecker> licenceCheckers = new ConcurrentHashMap<>(5);
	/** Logger object */
	private static final Logger log = Logger.getLogger("tigase.licence");
	/** Executor service handling timed execution of licence checking */
	private final static ScheduledExecutorService timer = Executors.newSingleThreadScheduledExecutor();
	/** Callback object to handle responses from server */
	private final static String STATISTICS_LOG_LEVEL = "statistics-log-level";
	private final static String TEST_MODE = "licence-library-test-mode";
	static String installationId = null;
	/** indicates whether it's the first run of the checker */
	private static long initialCheckDelay = (long) (ThreadLocalRandom.current().nextDouble(1, 16) * 60);
	/** Timer task responsible for licence checking */
	private static LicenceCheckDailyTask licenceDailyCheck;
	private static long subsequentCheckDelay = TimeUnit.DAYS.toSeconds(1);
	private static boolean testMode = false;
	/** Holds licence file */
	private final File LICENCE_FILE;
	/** Stores component name for which licence check is performed */
	private final String componentName;
	private final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HHmm");
	/** Holds licence object */
	private Licence lic;
	private boolean licenceShown = false;
	/** Callback used to update statistics data in component */
	private LicenceCheckerUpdateCallback updateCall;
	/** Holds date until which licence is valid */
	private Date validUntil;

	{
		String property = System.getProperty(TEST_MODE);
		if (null != property && Boolean.valueOf(property)) {
			log.log(Level.WARNING, "Licence library running in demo mode (faster checks!)");
			testMode = true;

			initialCheckDelay = 60;
			subsequentCheckDelay = TimeUnit.MINUTES.toSeconds(15);
		}

		property = System.getProperty(STATISTICS_LOG_LEVEL);
		Level level = Level.ALL;
		if (null != property) {
			level = Level.parse(property);
		}
		setLogger(level);
	}

	public static String getCodeForLicenceRetrieval(String component) throws Exception {
		if (LicenceCheckerUpdater.getStats() == null) {
			LicenceCheckerUpdater.init();
		}
		List<JID> managedVHosts = VHostsRetriever.getManagedVHosts();

		LicenceRetriever retriever = new LicenceRetriever();

		final Optional<Map<String, String>> serverVerifiableMetrics = LicenceChecker.getLicenceChecker(component)
				.getUpdateCall()
				.getServerVerifiableMetrics();
		String rawData = retriever.prepareRequest(installationId, component, managedVHosts, serverVerifiableMetrics.orElse(Collections.emptyMap()));
		return tigase.util.Base64.encode(rawData.getBytes("UTF-8"));
	}

	static String getInstallationId(boolean forceRetrieval) {
		if (forceRetrieval && installationId == null) {
			installationId = InstallationIdRetriever.getInstallationIdRetriever().getInstallationId();
		}
		return installationId;
	}

	public static String getInstallationId() {
		return getInstallationId(false);
	}

	/**
	 * Creates a {@link tigase.licence.LicenceChecker} with default update
	 * callback {@link tigase.licence.LicenceCheckerUpdateCallbackImpl} for the
	 * component which name was provided as parameter. Component name will be
	 * used to find licence file with the pattern of
	 * <code>etc/${cmpName}.licence</code>
	 * <p>
	 *
	 * @param cmpName name of the component for which
	 * {@link tigase.licence.LicenceChecker} will be instantiated.
	 *
	 * @return {@link tigase.licence.LicenceChecker} for the given cmpName
	 */
	public static LicenceChecker getLicenceChecker(String cmpName) {
		return getLicenceChecker(cmpName, null);
	}

	/**
	 * Creates a {@link tigase.licence.LicenceChecker} with custom
	 * implementation of {@link tigase.licence.LicenceCheckerUpdateCallback}
	 * interface for the component which name was provided as parameter.
	 * Component name will be used to find licence file with the pattern of
	 * <code>etc/${cmpName}.licence</code>
	 * <p>
	 *
	 * @param cmpName name of the component for which
	 * {@link tigase.licence.LicenceChecker} will be instantiated.
	 * @param call custom implementation of
	 * {@link tigase.licence.LicenceCheckerUpdateCallback}
	 *
	 * @return {@link tigase.licence.LicenceChecker} for the given cmpName
	 */
	public static LicenceChecker getLicenceChecker(String cmpName, LicenceCheckerUpdateCallback call) {

		final StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
		String callingClass = Arrays.stream(stackTrace)
				.skip(1)
				.limit(5)
				.map(st -> st.getClassName() + "." + st.getMethodName())
				.collect(Collectors.joining(" > "));

		log.log(Level.FINEST, "Getting logger for: {0} (from: {1}), callback: {2}, exists: {3}",
				new Object[]{cmpName, callingClass, call, licenceCheckers.get(cmpName)});

		LicenceChecker licenceChecker = licenceCheckers.get(cmpName);

		if (licenceChecker == null) {
			if (call == null) {
				call = new LicenceCheckerUpdateCallbackImpl(cmpName);
			}
			licenceChecker = new LicenceChecker(cmpName, call);
			licenceCheckers.put(cmpName, licenceChecker);
		} else if (call != null) {
			licenceChecker.setUpdateCall(call);
		}

		return licenceChecker;

	}

	public static Set<String> getLicencedComponents() {
		return Collections.unmodifiableSet(licenceCheckers.keySet());
	}

	public static Element getLicencingDetails(String component) {
		Element details = new Element("licence-details");
		details.addChild(new Element(INSTALLATION_ID_KEY, getInstallationId(true)));
		details.addChild(new Element("module", component));
		List<JID> managedVHosts = VHostsRetriever.getManagedVHosts();
		if (managedVHosts != null && !managedVHosts.isEmpty()) {
			details.addChild(new Element("vhosts", managedVHosts.stream()
					.map(Object::toString)
					.collect(Collectors.joining(","))));
		}
		return details;
	}

	static boolean isTestMode() {
		if (System.getProperty(TEST_MODE) != null && Boolean.valueOf(System.getProperty(TEST_MODE))) {
			log.log(Level.WARNING, "Licence library running in demo mode (faster checks!)");
			initialCheckDelay = 60;
			subsequentCheckDelay = TimeUnit.MINUTES.toSeconds(3);
			return true;
		} else {
			return false;
		}
	}

	public static void main(String[] args) {
		System.setProperty(TEST_MODE, "true");
		getLicenceChecker("acs");
	}

	/**
	 * Constructs new LicenceChecker object which entails creation of new
	 * licence file representation. If there wasn't one previously, a new timer
	 * task, which performs periodic licence check is created and first check is
	 * scheduled 5 minutes after startup.
	 * <p>
	 *
	 * @param cmpName name of the component for which licence checker was created
	 * @param call callback object used to update statistics data in component
	 */
	private LicenceChecker(String cmpName, final LicenceCheckerUpdateCallback call) {
		this.updateCall = call;
		this.componentName = cmpName;

		// create only one daily licence checker task and schedule first check
		// after 1-16 minutes of startup.
		if (licenceDailyCheck == null) {
			licenceDailyCheck = new LicenceCheckDailyTask(licenceCheckers);
			log.log(Level.FINEST, "initialCheckDelay: {0}, subsequentCheckDelay: {1}",
					new Object[]{initialCheckDelay, subsequentCheckDelay});
			timer.scheduleAtFixedRate(licenceDailyCheck, initialCheckDelay, subsequentCheckDelay, TimeUnit.SECONDS);

		}

		log.log(Level.CONFIG, "Created licence checker for {0}", cmpName);
		LICENCE_FILE = new File("etc/" + componentName + ".licence");
	}

	public boolean isLicenceShown() {
		return licenceShown;
	}

	public Licence reloadLicenceFromServer() throws NoSuchAlgorithmException, IOException, InvalidKeySpecException {
		return loadLicence(true);
	}

	public String getComponentName() {
		return componentName;
	}

	/**
	 * Returns Licence digest
	 *
	 * @return a {@link Date} representation of licence expiration date.
	 */
	public String getLicenceDigest() {
		if (lic != null) {
			return lic.getLicenceDigest();
		} else {
			return null;
		}
	}

	/**
	 * Returns date until which licence is valid.
	 *
	 * @return a {@link Date} representation of licence expiration date.
	 */
	public Date getValidUntil() {
		return validUntil;
	}

	public void setLicenceShown() {
		this.licenceShown = true;
	}

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder("LicenceChecker{");
		sb.append("LICENCE_FILE=").append(LICENCE_FILE);
		sb.append(", componentName='").append(componentName).append('\'');
		sb.append(", initialCheckDelay=").append(initialCheckDelay);
		sb.append(", subsequentCheckDelay=").append(subsequentCheckDelay);
		sb.append(", updateCall=").append(updateCall);
		sb.append(", lic=").append(lic);
		sb.append('}');
		return sb.toString();
	}

	/**
	 * Performs actual verification of licence validity
	 * <p>
	 *
	 * @return <strong>true</strong> for valid licence, <strong>false</strong>
	 * otherwise
	 */
	boolean validateLicence() {
		return validateLicence(false);
	}

	boolean validateLicence(boolean forceReload) {
		try {
			lic = loadLicence(forceReload);
			if (lic == null) {
				log.log(Level.WARNING, "Missing licence file ({0})!", new Object[]{LICENCE_FILE});
				if (!forceReload && !isLicenceShown()) {
					log.log(Level.WARNING, getUpdateCall().getMissingLicenseWarning());
					setLicenceShown();
				}
				return false;
			}
			validUntil = lic.getPropertyAsDate(Licence.VALID_UNTIL_KEY);

			try {
				switch (lic.check()) {
					case invalidDates:
						log.log(Level.WARNING, "Licence expired.");
						return false;

					case invalidVHosts:
						log.log(Level.WARNING, "VHost list does not match.");
						return false;

					case invalidSignature:
						log.log(Level.WARNING, "Licence file has been tempered with!");
						return false;

					case valid:
						log.log(Level.INFO, "Licence OK");
						break;
				}
			} catch (Exception e) {
				log.log(Level.WARNING, "Licence invalid", e);
				return false;
			}

			String appId = lic.getPropertyAsString(Licence.APP_ID_KEY);
			if ((appId == null) || !appId.equals(componentName)) {
				log.log(Level.WARNING, "This is not licence for {0} Component!", componentName);
				return false;
			}
		} catch (Exception e) {
			log.log(Level.WARNING, "Can't load licence file. Error: {0}", new Object[]{e.getMessage()});
			return false;
		}
		return true;
	}

	Licence getLicence() {
		return lic;
	}

	/**
	 * Return callback used to update statistics data for particular
	 * licenceChecker
	 *
	 * @return callback used to update statistics data for particular
	 * licenceChecker
	 */
	LicenceCheckerUpdateCallback getUpdateCall() {
		return updateCall;
	}

	private void setUpdateCall(LicenceCheckerUpdateCallback updateCall) {
		this.updateCall = updateCall;
	}

	private Licence loadRemoteLicence(LicenceLoader loader) {
		LicenceRetriever retriever = new LicenceRetriever();
		String licenceData = retriever.retrieve(this.componentName);
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST, "Loading licence for component: {0} from server, licenceData: {1}",
					new Object[]{this.componentName, licenceData});
		}

		if (licenceData == null || licenceData.isEmpty()) {
			log.info("Cannot load licence from Server.");
			return null;
		}
		try {
			ByteArrayInputStream in = new ByteArrayInputStream(licenceData.getBytes());
			Licence licence = loader.loadLicence(in);
			in.close();
			return licence;
		} catch (Exception e) {
			log.log(Level.WARNING, "Cannot parse licence data.", e);
			return null;
		}
	}

	private Licence loadRemoteLicenceAndOverwrite(LicenceLoader loader) {
		final Licence licence = loadRemoteLicence(loader);
		if (licence != null && licence instanceof ImmutableLicence) {
			try {
				Writer writer = new OutputStreamWriter(new FileOutputStream(LICENCE_FILE),
													   Charset.forName("UTF-8").newEncoder());
				((ImmutableLicence) licence).save(writer);
				writer.close();
			} catch (Exception e) {
				log.log(Level.WARNING, "Cannot store licence in file " + LICENCE_FILE, e);
			}
		}
		return licence;
	}

	/**
	 * Method loads licence file either from server of from file. It uses following logic:
	 *
	 * <pre>
	 * a) if the {@code force} flag IS set:
	 *      * if it WAS possible to retrieve licence from server it is returned;
	 *      * if it WAS NOT possible to retrieve licence from server:
	 *        - if there IS NO file present - null will be returned;
	 *        - if there IS already some licence file it will be loaded.
	 * b) if the {@code force} flag IS NOT set:
	 *      * if the licence file DOES NOT exist - an attempt will be made to load licence
	 *          from the server and return it's result (either null or actual licence file);
	 *      * if the licence file DOES exists - it will be loaded.
	 * </pre>
	 *
	 * @param forceReloadFromServer specifies whether a licence should be force-loaded
	 * from the server instead of loading local file
	 *
	 * @return a {@link Licence} if it was possible to load it from server or local file
	 */
	private Licence loadLicence(boolean forceReloadFromServer)
			throws NoSuchAlgorithmException, IOException, InvalidKeySpecException {
		final LicenceLoader loader = LicenceLoaderFactory.create();
		if (log.isLoggable(Level.FINEST)) {
			log.log(Level.FINEST,
					"Trying to load licence file from file: {0} (exists: {1}, empty: {2}), forcing load from server: {3}",
					new Object[]{LICENCE_FILE, LICENCE_FILE.exists(), LICENCE_FILE.length() == 0,
								 forceReloadFromServer});
		}

		if (forceReloadFromServer) {
			Licence serverLicence = loadRemoteLicenceAndOverwrite(loader);
			if (serverLicence != null) {
				lic = serverLicence;
				return serverLicence;
			}
		}
		if (!LICENCE_FILE.exists() || LICENCE_FILE.length() == 0) {
			log.log(Level.INFO, "Missing licence file ({0}), retrieving from the server!", new Object[]{LICENCE_FILE});
			if (!forceReloadFromServer) {
				return loadRemoteLicenceAndOverwrite(loader);
			} else {
				return null;
			}
		}
		Licence licence = loader.loadLicence(LICENCE_FILE);

		try {
			Date vu = licence.getPropertyAsDate(Licence.VALID_UNTIL_KEY);
			// it should start download new licence two days before end of
			// current licence.
			if (vu != null && System.currentTimeMillis() > vu.getTime() - 172800000) {
				Licence l = loadRemoteLicenceAndOverwrite(loader);
				if (l != null) {
					licence = l;
				}
			}

			if (licence.getPropertyAsString("installation-id") == null) {
				// legacy licence! force reload from server!
				log.log(Level.FINEST, "Legacy Licence! Force reload from server!");
				Licence l = loadRemoteLicenceAndOverwrite(loader);
				if (l != null) {
					log.log(Level.FINEST, "Loaded new licence from server: " + l);
					licence = l;
				}
			}

		} catch (ParseException e) {
			log.log(Level.WARNING, "Ouch! We can't load new licence from server! ", LICENCE_FILE);
		}

		return licence;
	}

	/**
	 * Performs logger setup - all statistics related logging is also written to
	 * separate log file.
	 * <p>
	 */
	private void setLogger(Level lvl) throws SecurityException {
		Logger stats_log = Logger.getLogger("tigase.stats.collector");
		log.setLevel(lvl);
		stats_log.setLevel(lvl);
		try {
			Handler fileHandler = new FileHandler("logs/statistics.log", 10000000, 5, true);
			fileHandler.setLevel(lvl);

			stats_log.addHandler(fileHandler);
			log.addHandler(fileHandler);
		} catch (IOException ex) {
			log.log(Level.CONFIG, componentName);
		}
	}

}
