/**
 * Licence Library - bootstrap configuration for all Tigase projects
 * Copyright (C) 2011 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.licence;

import tigase.stats.collector.DefaultElementSigner;
import tigase.stats.collector.ElementSigner;
import tigase.stats.collector.StatisticsData;
import tigase.stats.collector.provider.StatisticsUploader;
import tigase.sys.TigaseRuntime;
import tigase.xml.Element;

import javax.crypto.spec.SecretKeySpec;
import java.util.Map;
import java.util.Set;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Timer task responsible for license checking.
 * <p>
 * Following scheme is used:
 * <ul>
 * <li>in case of valid license next check is schedule after validity
 * expired</li>
 * <li>in case of invalid or missing license StatisticsData is collected and
 * an attempt to upload it to Tigase REST server is made; following
 * additional conditions apply:
 * <ul>
 * <li>if this is a cold start of the server and thus a first attempt at
 * statistics generation and upload a complete fail to upload it resulting
 * in StatisticsUploader executing {@code .onFailure()} callback of
 * {@link tigase.stats.collector.provider.StatisticsUploader.ResultCallback}
 * object (i.e. after 30 attempt, each scheduled twice longer than the
 * previous, amounting to around 15 hours) then an immediate system shutdown
 * is performed.</li>
 * <li>if there were an successful submission of statistics then this is
 * only a temporary issue with uploading statistics then a subsequent check
 * is schedule with 1 day delay and we allow 10 days of lax treatment; in
 * case of 10 failed attempts to upload data (in the above step) a system
 * shutdown is performed</li>
 * </ul>
 * </ul>
 */
class LicenceCheckDailyTask
		extends TimerTask {

	/** Limit number of all possible attempts to send statistics */
	protected final static int STATISTICS_FAIL_COUNT_LIMIT = 10;
	private final static ScheduledExecutorService timer = Executors.newSingleThreadScheduledExecutor();
	/** Holds secret used to to sign uploaded data */
	private static final String SECRET = "0000";
	/** Holds key based on secret used to sign uploaded data */
	private static final SecretKeySpec key = new SecretKeySpec(SECRET.getBytes(), "HmacSHA1");
	/** DefaultElementSigner used to sing uploaded data */
	private static final ElementSigner signer = new DefaultElementSigner(key);
	private static final long INITIAL_LICENCE_READ_DELAY = 60;
	/** Licence information and warning should be shown only once */
	private final static int NOT_LICENCED_ERROR_CODE = 402;
	private static double licenceReadRetryDelayFactor = 1.9D;
	private static Logger log = Logger.getLogger("tigase.licence");
	private static int maxCheckRetryCount = 10;
	private static StatisticsUploader.ResultCallback resCall;
	/** Number of failed attempts to send the statistics */
	private static int statisticsUploadFailCount = 0;
	/** Uploader object using created signer to upload data */
	private static StatisticsUploader uploader = new StatisticsUploader(signer);
	private boolean firstCheck = true;
	private Map<String, LicenceChecker> licenceCheckers = null;

	LicenceCheckDailyTask(Map<String, LicenceChecker> checkers) {
		this.licenceCheckers = checkers;

		if (LicenceChecker.isTestMode()) {
			licenceReadRetryDelayFactor = 1.3D;
			maxCheckRetryCount = 3;
		}
	}

	@Override
	public void run() {
		runLicenceCheck(false, 0, 0);
	}

	private void runLicenceCheck(final boolean forceLicenceReload, final int count, final long delay) {
		log.log(Level.FINE, "==================================================================");
		log.log(Level.FINE,
				"Daily License verification runtime (run: {0} out of: {1}; force reloading: {2}, delay: {3}).",
				new Object[]{count, maxCheckRetryCount, forceLicenceReload, delay});

		timer.schedule(new TimerTask() {

			@Override
			public void run() {
				if (LicenceChecker.installationId == null) {
					LicenceChecker.installationId = InstallationIdRetriever.getInstallationIdRetriever()
							.getInstallationId();
				}

				// it has to be done here because we allow for 10 consecutive failing *DAYS*

				log.log(Level.FINE, "statisticsUploadFailCount: {0}, STATISTICS_FAIL_COUNT_LIMIT: {1}",
						new Object[]{statisticsUploadFailCount, STATISTICS_FAIL_COUNT_LIMIT});

				if (statisticsUploadFailCount > STATISTICS_FAIL_COUNT_LIMIT) {
					log.log(Level.FINE, "Could not upload statistics within required period of time ({0})",
							new Object[]{STATISTICS_FAIL_COUNT_LIMIT});

					String[] msg = new String[]{"ERROR! Statistics upload failed!", "",
												"Could not upload statistics within required period of time" +
														STATISTICS_FAIL_COUNT_LIMIT + ",", "shutting down the system!",
												"", "Your installation-id is: " + LicenceChecker.installationId};
					TigaseRuntime.getTigaseRuntime().shutdownTigase(msg, NOT_LICENCED_ERROR_CODE);
				}

				Element additionalData = new Element("unlicensed-components");

				Set<String> invalidLicences = new ConcurrentSkipListSet<>();
				boolean forceSendingStatistics = false;
				boolean bannedShutdown = false;
				boolean displayLicenceNotice = false;

				if (!licenceCheckers.isEmpty()) {
					LicenceCheckerUpdater.init();
				}

				// Check all licence checkers.
				for (String key : licenceCheckers.keySet()) {
					LicenceChecker licChecker = licenceCheckers.get(key);

					log.log(Level.FINEST, "[1]Checking licence for component: {0}, licChecker: {1}, licence: {2}",
							new Object[]{key, licChecker, licChecker.getLicence()});

					if (licChecker.validateLicence(forceLicenceReload) && licChecker.getLicence() != null &&
							licChecker.getUpdateCall().additionalValidation(licChecker.getLicence())) {

						log.log(Level.INFO, "Licence valid until: {0}", licChecker.getValidUntil());

						Boolean st = licChecker.getLicence().getPropertyAsBoolean(Licence.SENDING_STATISTICS_KEY);
						forceSendingStatistics |= (st != null && st);
						log.log(Level.INFO, "forceSendingStatistics: " + forceSendingStatistics);

					} else {
						// licence is not valid, grab additional information from component (if provided)
						if (!licChecker.isLicenceShown()) {
							log.log(Level.WARNING, licChecker.getUpdateCall().getMissingLicenseWarning());
							licChecker.setLicenceShown();
						}
						Element cmpAdditionalData = licChecker.getUpdateCall().getComponentAdditionalData();
						additionalData.addChild(cmpAdditionalData);

						invalidLicences.add(licChecker.getComponentName());
					}

					if (licChecker != null && licChecker.getLicence() != null) {
						Boolean sd = licChecker.getLicence().getPropertyAsBoolean(Licence.BANNED_KEY);
						bannedShutdown |= (sd != null && sd);

						Boolean licenceNotice = licChecker.getLicence()
								.getPropertyAsBoolean(Licence.DISPLAY_LICENCE_NOTICE_KEY);
						displayLicenceNotice |= (licenceNotice != null && licenceNotice);
					}

					log.log(Level.FINEST,
							"[2]Checking licence for component: {0}, invalidLicencePresent: {1}, forceSendingStatistics: {2}, bannedShutdown: {3}, displayLicenceNotice: {4}, licenceShown: {5}",
							new Object[]{key, invalidLicences, forceSendingStatistics, bannedShutdown,
										 displayLicenceNotice, licChecker.isLicenceShown()});

					if (displayLicenceNotice && !licChecker.isLicenceShown()) {
						log.log(Level.WARNING, licChecker.getUpdateCall().getMissingLicenseWarning());
						licChecker.setLicenceShown();
					}
				}

				log.log(Level.FINEST,
						"Checking result: invalidLicencePresent: {0}, LicenceChecker.installationId: {1}, count: {2}, maxCheckRetryCount: {3}, delay: {4}, bannedShutdown: {5}, forceSendingStatistics: {6}",
						new Object[]{invalidLicences, LicenceChecker.installationId, count, maxCheckRetryCount, delay,
									 bannedShutdown, forceSendingStatistics});

				if (invalidLicences.size() > 0 || LicenceChecker.installationId == null) {
					if (count >= maxCheckRetryCount) {

						for (String cmp : invalidLicences) {
							final LicenceChecker checker = licenceCheckers.get(cmp);
							if (!checker.isLicenceShown()) {
								String warning = checker.getUpdateCall().getMissingLicenseWarning();
								log.log(Level.WARNING, warning);
							}
						}

						String[] msg = new String[]{"ERROR! No valid licence found for listed components!", "",
													"Please obtain correct licence file for your installation.",
													"Your installation-id is: " + LicenceChecker.installationId};
						TigaseRuntime.getTigaseRuntime().shutdownTigase(msg, NOT_LICENCED_ERROR_CODE);

					} else {
						long nextDelay = (long) (delay == 0
												 ? INITIAL_LICENCE_READ_DELAY
												 : (delay * licenceReadRetryDelayFactor));
						log.log(Level.FINEST,
								"Invalid licence file detected! Remaining checks: {0} (next attempt in: {1}s)",
								new Object[]{maxCheckRetryCount - count, nextDelay});
						runLicenceCheck(true, count + 1, nextDelay);
					}
				} else if (forceSendingStatistics) {
					// if at least one licence is forces sending statistics upload
					StatisticsData data;
					data = LicenceCheckerUpdater.updateData();
					// concat additionalData
					data.setAdditionalData(new Element(LicenceChecker.INSTALLATION_ID_KEY,
													   LicenceChecker.getInstallationId()).toString() +
												   data.getAdditionalData());
					data.setAdditionalData(data.getAdditionalData() + additionalData.toString());

					log.log(Level.WARNING, "Uploading data (please see logs/statistics.log for more details)");
					log.log(Level.FINEST, "Uploading data: " + data.toElement(false));
					resCall = new ResCall();
					uploader.upload(data, resCall);
				}

				if (bannedShutdown) {
					// shutdown only if the installation is banned or
					//      we have licence which limits has been crossed and property to shut down
					String[] msg = new String[]{
							"ERROR! You are using licenced component and you are not respecting licencing conditions!",
							"", "Please contact us and provide your installation-id: " + LicenceChecker.installationId,
							"to resolve the issue"};

					TigaseRuntime.getTigaseRuntime().shutdownTigase(msg, NOT_LICENCED_ERROR_CODE);
				}

			}
		}, delay, TimeUnit.SECONDS);
	}

	/**
	 * Implementation for handling results of uploading statistics. On failed
	 * attempt either the system is shut down (if it was a cold run, i.e. the
	 * first one) or the counter of fails is increased and on success attempt
	 * the counter is reset and the system allows for less strict enforcement of
	 * failed statistics upload.
	 */
	public class ResCall
			implements StatisticsUploader.ResultCallback {

		/**
		 * <code>onFailure</code> callback. If it's called on the cold run (i.e.
		 * first execution) then immediately shut down the system; otherwise if
		 * there were previously successful attempts to submit data we can
		 * assume that this is not intentional thus allow less strict
		 * enforcement and run the check for the consecutive 10 days each time
		 * resulting in increase of failed attempts counter.
		 * <p>
		 *
		 * @param ex is a <code>Exception</code>
		 */
		@Override
		public void onFailure(Exception ex) {
			if (firstCheck) {
				log.log(Level.SEVERE,
						"Could not upload statistics within required period of time, shutting down the system");
				System.exit(402);
			} else {
				statisticsUploadFailCount++;
				log.log(Level.FINEST,
						"Could not upload statistics; " + (STATISTICS_FAIL_COUNT_LIMIT - statisticsUploadFailCount) +
								" attempts before system shutdown", ex);
			}
		}

		/**
		 * <code>onSuccess</code> callback resulting in reset of failed attempts
		 * counter as well as switching to less strict licence checking which
		 * allows running component for 10 days without successful execution.
		 * <p>
		 *
		 * @param result is a <code>Element</code>
		 */
		@Override
		public void onSuccess(Element result) {
			log.log(Level.FINE, "statistics uploaded = {0}; resetting number of attempts",
					((result != null) ? result.toString() : null));
			// reset number of attempts
			statisticsUploadFailCount = 0;
			// run checking in regular mode (i.e. allow 10 days of failed
			// submission before shutting down the system)
			firstCheck = false;
		}
	}
}
