/**
 * Licence Library - bootstrap configuration for all Tigase projects
 * Copyright (C) 2011 Tigase, Inc. (office@tigase.com) - All Rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential
 */
package tigase.licence;

import tigase.licence.LicenceValidator.ValidationResult;

import java.io.*;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

final class ImmutableLicence
		implements Licence, Serializable {

	public static final String HASH_ALGO = "SHA-256";
	private static final Logger log = Logger.getLogger("tigase.licence");

	private static final String DATE_FORMAT = "yyyy-MM-dd";
	private static final long serialVersionUID = 1L;
	;
	private List<Entry> entries = new ArrayList<Entry>();
	private Map<String, Entry> entriesMap = new HashMap<String, Entry>();
	private LicenceValidatorImpl validator;

	ImmutableLicence() {
	}

	@Override
	public ValidationResult check()
			throws InvalidKeyException, NoSuchAlgorithmException, SignatureException, ParseException {
		return this.validator.check(this);
	}

	@Override
	public synchronized byte[] getBytes() {
		StringBuilder sb = new StringBuilder();
		for (Entry entry : this.entries) {
			if (entry.key == null || entry.key.startsWith("#") || !entry.isEqualSign()) {
				continue;
			}
			if ("signature".equals(entry.key)) {
				continue;
			}
			sb.append(new String(entry.getBytes()));
		}
		return sb.toString().getBytes();
	}

	public String getLicenceDigest() {
		String result = null;
		try {
			final MessageDigest md;
			md = MessageDigest.getInstance(HASH_ALGO);
			final byte[] digest = md.digest(getBytes());
			final String encodeHex = encodeHex(digest);
			if (encodeHex != null && !encodeHex.isEmpty()) {
				result = encodeHex;
			}
		} catch (NoSuchAlgorithmException e) {
			log.log(Level.FINE, "There was a problem with creating a licence digest");
		}
		return result;
	}

	@Override
	public Boolean getPropertyAsBoolean(String key) {
		String value = getProperty(key);
		return (value == null ? null : value.trim().equalsIgnoreCase("yes") || Boolean.valueOf(value.trim()));
	}

	@Override
	public Calendar getPropertyAsCalendar(String key) throws ParseException {

		String value = getProperty(key);
		Calendar cal = Calendar.getInstance();
		final DateFormat DATE_FORMATTER = new SimpleDateFormat(DATE_FORMAT);

		Date dt = value == null ? null : DATE_FORMATTER.parse(value);
		if (dt != null) {
			cal.setTime(dt);
		} else {
			cal = null;
		}

		if (key.equals(Licence.VALID_UNTIL_KEY) && cal != null) {
			cal.add(Calendar.DATE, 1);
			cal.add(Calendar.MILLISECOND, -1);
		}

		return cal;
	}

	@Override
	public Date getPropertyAsDate(String key) throws ParseException {
		String value = getProperty(key);
		final DateFormat dateFormatter = new SimpleDateFormat(DATE_FORMAT);

		Date val = value == null ? null : dateFormatter.parse(value);

		if (key.equals(Licence.VALID_UNTIL_KEY) && val != null) {
			Calendar cal = Calendar.getInstance();
			cal.setTime(val);
			cal.add(Calendar.DATE, 1);
			cal.add(Calendar.MILLISECOND, -1);
			val = cal.getTime();
		}

		return val;
	}

	@Override
	public Double getPropertyAsDouble(String key) {
		String value = getProperty(key);
		return value == null ? null : Double.valueOf(value);
	}

	@Override
	public Integer getPropertyAsInteger(String key) {
		String value = getProperty(key);
		return value == null ? null : Integer.valueOf(value);
	}

	@Override
	public String getPropertyAsString(String key) {
		return getProperty(key);
	}

	@Override
	public String toString() {

		StringBuilder sb = new StringBuilder();
		sb.append("ImmutableLicence: ");
		entries.forEach((k) -> sb.append(k.key).append(": ").append(k.value).append(", "));
		return sb.toString();
	}

	void save(Writer out) throws IOException {
		for (Entry entry : this.entries) {
			out.write(entry.key);
			if (entry.key != null && entry.key.trim().length() > 0 && entry.isEqualSign()) {
				out.write("=");
			}
			if (entry.value != null) {
				out.write(entry.value);
			}
			out.write('\n');
		}
		out.close();
	}

	void load(InputStream in) throws IOException {
		BufferedReader reader = new BufferedReader(new InputStreamReader(in));

		entries.clear();
		entriesMap.clear();

		String line;
		while ((line = reader.readLine()) != null) {
			put(line);
		}

		reader.close();
	}

	void setValidator(LicenceValidatorImpl licenceValidatorImpl) {
		this.validator = licenceValidatorImpl;
	}

	private String encodeHex(byte[] b) {
		String result = "";
		for (byte aB : b) {
			result += Integer.toString((aB & 0xff) + 0x100, 16).substring(1);
		}
		return result;
	}

	private String getProperty(String key) {
		Entry entry = this.entriesMap.get(key);
		if (entry == null) {
			return null;
		} else {
			return entry.value;
		}
	}

	private void put(String line) {
		if (line == null) {
			return;
		}
		int p = line.indexOf("=");

		String key;
		String value;
		if (p > 0) {
			key = line.substring(0, p).trim();
			value = line.substring(p + 1).trim();
		} else {
			key = line;
			value = null;
		}

		Entry entry = new Entry();
		entry.setKey(key);
		entry.setValue(value);
		entry.setEqualSign(p != -1);
		entries.add(entry);
		entriesMap.put(key, entry);
	}

	private static final class Entry
			implements Serializable {

		private static final long serialVersionUID = 8338663780387437875L;

		private boolean equalSign;

		private String key;

		private String value;

		public byte[] getBytes() {
			String out = key + "=" + (value == null ? "" : value) + new String(new byte[]{0x0a});
			return out.getBytes();
		}

		public boolean isEqualSign() {
			return equalSign;
		}

		public void setEqualSign(boolean equalSign) {
			this.equalSign = equalSign;
		}

		public void setKey(String key) {
			this.key = key;
		}

		public void setValue(String value) {
			this.value = value;
		}
	}
}
